#!/usr/bin/perl -w

# TODO: check that ratings are calculated correctly with RRs
# TODO: make games count against director

# tourney.pl - perform Scrabble tournament calculations

# Copyright (C) 1996 by John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

# $Id: tourney.pl,v 1.8 2004/08/24 17:51:19 jjc Exp jjc $

# Input File Format
#
# One line per player, reading:
#
#   name rating rr pairings ; scores
#
# e.g.
#
#   John Chew 1823*75 R2/7.5/+30 1 2 3 0 ; 400 450 350 50 # comment
#
# meaning that John Chew was 
#   rated 1823 before this tournament
#   played a double round robin and won 7.5 games with a +30 spread
#   played three additional games
#     scoring 400 against player #1, 450 against player #2,
#     350 against player #3
#   and had a 50-point bye
#
# name: given name(s) followed by surname
# rating: pre-tournament rating, followed optionally by an asterisk and the 
#   number of games on which the rating is based, or by two asterisks to 
#   indicate that the rating is fixed (as e.g. for a Club #3 director)
# rr: round robin information (optional), if present prevents scoring
#   statistics from being calculated.  must be a capital 'R' followed by
#   number of round robins, games won and spread, separated by '/'s.
# pairings: opponent numbers; first in file is 1, bye is 0.
# scores: player's scores; opponent's scores are found on opponent's lines.

# 1998-10-20 byes scoring 0 points are not recorded as ties
#
# $Log: tourney.pl,v $
# Revision 1.8  2004/08/24 17:51:19  jjc
# ! changed -t to conform with Scrabble News requirements
#
# Revision 1.7  2004/08/24 16:39:47  jjc
# + Added -t
#
# Revision 1.6  2004/08/08 23:23:27  jjc
# + added -N option
#
# Revision 1.5  2004/07/24 02:19:34  jjc
# + added -e tor80
#
# Revision 1.4  2003/11/02 18:13:35  jjc
# ! minor bug fix
#
# Revision 1.3  2003/11/02 18:08:06  jjc
# ! code cleanup
# + credits listed in reports
#
# Revision 1.2  2003/11/02 17:21:16  jjc
# + added NSC78 credit option
# + added NSC78 credit support to scorecard option
#
# Revision 1.1  2003/10/28 04:46:41  jjc
# Initial revision

sub Usage { 
  die "Usage: $0 [-A] [-C] [-d r] [-e nsc78|wls] [-f] [-O] [-S] [-c|-n|-N|-r|-s pn|-p formula|-t|-v] file...\n"
     ."  -A     do not use acceleration or feedback points\n"
     ."  -c     output in Club #3's format\n"
     ."  -C     use club tournament multipliers\n"
     ."  -d r   divide tournament after round r for ratings purpose\n"
     ."  -e era specify time period\n"
     ."           nsc78  50-point credits\n"
     ."           tor80  spread capped at 200\n"
     ."           wls    W-L and spread\n"
     ."  -f     use fixed player ids\n"
     ."  -n     output NSA ratings input files\n"
     ."  -N     output NSA ratings input files without inverting names\n"
     ."  -O     suppress online names\n"
     ."  -p f   output pairings according to one of the following formulae:\n"
     ."           koth - King of the Hill\n"
     ."  -r     output regular readable reports\n"
     ."  -s pn  output a scoresheet for a player specified by number\n"
     ."  -S     suppress scores\n"
     ."  -t     output a simple table of rankings, W-L and spread\n"
     ."  -v     display version number of this script\n"
     ."  -w n   wrap at column n (default 80)\n";
     ;
  }

## include libraries

unshift(@INC, "$ENV{'HOME'}/lib/perl") if defined $ENV{'HOME'};
if ($^O eq 'MacOS') { use lib ':'; }

require 'getopts.pl';
require 'ratings.pl';
require 'ratings2.pl';

## parse command line

# Macintosh stuff
# @ARGV = ('-f','c.in'); open(OUT, ">c.out") || die "can't create c.out";
# select(OUT); &MacPerl'SetFileInfo('MSWD','TEXT','c.out');
@argv = split(/:/, $0); @argv = split(/\s+/, pop @argv); 
shift(@argv); unshift(@ARGV, @argv);

&Getopts('ACcd:e:fnNOp:rs:tvSw:-:') || &Usage;
$n = 0; 
$opt_A = 0 unless defined $opt_A;
$opt_w = 80 unless defined $opt_w;
$opt_C = 0 unless defined $opt_C;
$opt_f = 0 unless defined $opt_f;
$opt_d = 1000000 unless defined $opt_d;
$opt_e = 'wls' unless defined $opt_e;
$opt_O = 0 unless defined $opt_O;
defined $opt_c && $n++;
defined $opt_n && $n++;
defined $opt_N && $n++;
defined $opt_p && $n++;
defined $opt_r && $n++;
defined $opt_s && $n++;
defined $opt_t && $n++;
defined $opt_v && $n++;
$n == 0 ? ($opt_r = 1) : $n > 1 && &Usage;

&ratings2'UseAccelerationBonuses(!$opt_A);
&ratings2'UseClubMultipliers($opt_C);
# maximum number of times to iterate when calculating initial ratings
&ratings2'SetMaximumIterations(25);

## global variables

# length of output lines
$global'lineLength   = $opt_w;

# length of longest player name
$global'nameLength   = 0;

# length of longest player ID
$global'numberLength = 2;

# version number of this script
$global'version     = '1.4.1';
# 1.2.1: apostrophe okay in player name
# 1.3: oldstyle credits
# 1.3.1: code cleanup
# 1.4: credits shown in reports
# 1.4.1: bug fix

# prototypes

sub CalculateSeeds ($);
sub Online ($);
sub ProcessFile ($);
sub WritePairings (\@);
sub WriteReport (\@);
sub WriteSS (\@$);

## main code

dbmopen(%ONLINE, 'online', 0600) unless $::opt_O;
if ($opt_v) { print "$0: version $global'version.\n"; exit 0; }
elsif ($#ARGV == -1) { &ProcessOpenFile(*STDIN); }
else { for $ARGV (@ARGV) { ProcessFile $ARGV; } }
dbmclose(%ONLINE) unless $::opt_O;

## subroutines

use strict;

# CalculateSeeds $players
sub CalculateSeeds ($) {
  my $ps = shift;
  my($id, $last, $lastseed, $seed) = (0, -1, 1, 0);
  for my $id (sort {$ps->[$b]{'oldr'} <=> $ps->[$a]{'oldr'}} 0..$#$ps) { 
    $seed++;
    if ($ps->[$id]{'oldr'} != $last) 
      { $lastseed=$seed; $last=$ps->[$id]{'oldr'}; }
    $ps->[$id]{'seed'} = $lastseed;
    }
  }

# $full_name = &Online($real_name);
sub Online ($) { 
  our(%ONLINE);
  defined $ONLINE{$_[0]} ? "$_[0] ($ONLINE{$_[0]})" : $_[0]; 
  }

# ProcessFile $filename
sub ProcessFile ($) { my $filename = shift;
  if (open(FILE, "<$filename")) { &ProcessOpenFile(*FILE); close(FILE); }
  else { warn "Can't read file \`$filename': $!\n"; }
  }

no strict;

# &ProcessOpenFile(*FH); 
sub ProcessOpenFile { local(*FH) = @_;
  local($players) = &ReadFile(*FH);
  defined $opt_c ? &WriteClub3($players) :
  defined $opt_r ? (WriteReport @$players) :
  defined $opt_n ? &WriteNSA($players) :
  defined $opt_N ? &WriteNSA($players, 1) :
  defined $opt_t ? &WriteSimpleTable($players) :
  defined $opt_p ? (WritePairings @$players) :
    WriteSS @$players, $opt_s;
  }

# [ $player_structs ] = &ReadFile(*FH);
sub ReadFile { local(*FH) = @_;
  local($games, $l, $o, $opts, $os, $osc, $p, $pn, $ps, $pts, $round, $sc, 
    $scs, $spread);

  # read players
  $ps = [];
  $global'nameLength = length("Name (online)");
  while ($p = &ReadPlayer(*FH)) { 
    push(@$ps, $p); 
    $p->{'id'} = $#$ps;
    $os = $p->{'opps'}; $scs = $p->{'scores'};
    $l = length($p->{'fname'} = Online $p->{'name'});
    $global'nameLength = $l if $global'nameLength < $l;
    printf STDERR "%s: number (%d) of opponents (%s) "
      ."is less than number (%d) of scores (%s).\n",
      $p->{'name'}, 1+$#$os, "@$os", 1+$#$scs, "@$scs" if $#$os < $#$scs;
    }
  $global'numberLength = length(1+$#$ps);
  $global'numberLength = 2 if $global'numberLength < 2;

  # analyse and check data
  for $pn (0..$#$ps) { $p = $ps->[$pn];
    $games = 0; $opts = $pts = 0;
    $p->{'ewins1'} = $p->{'ewins2'} = $p->{'hi'} = $p->{'rgames'} = 
      $p->{'spread'} = $p->{'tagn'} = $p->{'tfor'} = $p->{'wins'} = 
      $p->{'credits'} = 0;
    $os = $p->{'opps'}; 
    $p->{'games'} = $#$os + 1;
    for $round (0..$#$os) {
      $o = $os->[$round];
      $sc = $p->{'scores'}[$round];
      if ($o == -1) { # bye
	next unless defined $sc;
	$spread = $sc;
	if ($opt_c) {
	  printf STDERR "%s: bye in round %d scored %+d instead of standard 0.\n",
	    $p->{'fname'}, 1+$round, $p->{'scores'}[$round] 
	    if $p->{'scores'}[$round];
	  }
	else {
	  my $score = $p->{'scores'}[$round];
	  printf STDERR
	    "%s: bye in round %d scored %+d instead of standard -50, 0, +50.\n",
	    $p->{'fname'}, 1+$round, $p->{'scores'}[$round] 
	    if $score != 50 && $score != -50 && $score != 0;
	  }
	}
      elsif ($o == $pn) {
	printf STDERR "%s: played self in round %d\n", $p->{'fname'}, 1+$round;
	next;
	}
      elsif ($o > $#$ps) {
	printf STDERR "%s: opponent number (%d) in round %d is too big.\n",
	  $p->{'fname'}, $o, 1+$round;
	next;
	}
      else {
	$o = $ps->[$o];
	printf STDERR "In round %d, %s's opp was %s but %s's opp was %s.\n",
	  $round+1, $p->{'fname'}, $o->{'fname'}, $o->{'fname'},
	  $ps->[$o->{'opps'}[$round]]{'fname'}
	  if $pn != $o->{'opps'}[$round];
	next unless defined $sc;
	$p->{'hi'} = $sc if $p->{'hi'} < $sc;
	$pts  += $sc;
	$osc = $o->{'scores'}[$round];
	if (defined $osc) {
	  $opts += $osc;
	  $spread = $sc - $osc;
	  $p->{$round<$::opt_d ? 'ewins1' : 'ewins2'} += (($spread<=>0)+1)/2;
	  $p->{'rgames'} ++;
	  $games++;
	  }
	else {
	  printf STDERR "In round %d, %s's opp (%s) had no score.\n",
	    $round+1, $p->{'fname'}, $o->{'fname'}; 
	  }
	}
      if ($opt_e eq 'tor80') {
        if ($spread > 200) { $p->{'spread'} += 200; }
        elsif ($spread < -200) { $p->{'spread'} += -200; }
	else { $p->{'spread'} += $spread; }
        }
      else { $p->{'spread'} += $spread; }
      $p->{'wins'}   += (($spread<=>0)+1)/2 unless $spread == 0 && $o == -1;
	# unless zero-scoring bye
      $p->{'credits'} 
        += int($sc/50) # score credits
	+  ($spread > 0 ? int($spread/50) : 0) # spread credits
	+  (($spread <=> 0) * 3 + 3)/2; # win credits
      }
    if (defined $p->{'rr'}) {
      $p->{'ewins'} += $p->{'rr'}[1];
      $p->{'rgames'} += $p->{'rr'}[0] * $#$ps;
      $p->{'spread'} += $p->{'rr'}[2];
      $p->{'wins'} += $p->{'rr'}[1];
      $opt_S = 1;
      }
    if ($games > 0) { 
      $p->{'afor'} = $pts/$games;
      $p->{'aagn'} = $opts/$games; 
      $p->{'tfor'} = $pts;
      $p->{'tagn'} = $opts;
      }
    else { $p->{'afor'} = $p->{'aagn'} = 0; }
    }
  $ps;
  }

# the player_struct returned by the following sub and used elsewhere
# has the following fields:
#   aagn    average points scored by opponents
#   afor    average points scored by player
#   curr    current rating during iteration of initial rating
#   ewins   earned wins (not including byes)
#   fname   full name with online id appended if any
#   games   games played (including byes)
#   hi      high game score
#   id      0-based id
#   midr    mid-tournament rating in a split-rated tournament
#   name    full name
#   newr    post-tournament rating
#   oldr    pre-tournament rating
#   opps    [ opponent ids (0-based) ]
#   rank    ranking
#   rgames  real games (not including byes)
#   rr      [ # of round robins played, wins, spread ] or undef
#   scores  [ own score in each game ]
#   spread  point spread
#   tagn    total points scored by opps
#   tfor    total points scored by player
#   totalg  number of games played prior to this tournament
#           (-1 if rating is fixed)
#   wins    games won (including byes)

# $player_struct = &ReadPlayer(FH);
sub ReadPlayer { local(*FH) = @_;
  local($games, $input, $n, $o, @opps, $rr, $r, $s, $t, @t);
  while (<FH>) { $input = $_; s/#.*//; next unless /\S/;
    if (($n, $r, $games, $rr, $o, $s) 
= m!^([a-zA-Z][-.,'a-zA-Z ]+[.a-zA-Z]) +(\d+)(\*\*|\*\d+)? +(R\d+/\d*\.?\d*/[+-]?\d+ )? *([\d ]*); *([-\d ]*)$!)
      {
      if (defined $rr) { $rr =~ s/^R//; $rr = [split(/\//, $rr)];}
      for $t (@opps = split(/\s+/, $o)) { $t--; }
      return { 
	input => $input,
	name => $n, 
	oldr => $r, 
	rr => $rr,
	opps => \@opps,
	scores => [split(/\s+/,$s)], 
	totalg => 
	  (defined $games) ? ($games eq '**') ? -1 : substr($games, 1): 100
	};
      }
    else {
      warn "Can't parse (and am ignoring) the following line:\n$_";
      }
    }
  undef;
  }

sub SortByCredits {
  $b->{'credits'}   <=> $a->{'credits'} || 
  $b->{'wins'} <=> $a->{'wins'} || 
  $b->{'afor'}   <=> $a->{'afor'} || 
  $a->{'aagn'}   <=> $b->{'aagn'} || 
  $b->{'oldr'}   <=> $a->{'oldr'} || 
  $b->{'name'}   cmp $a->{'name'} 
  }

sub SortByWinsAndCume {
  $b->{'wins'}   <=> $a->{'wins'} || 
  $b->{'spread'} <=> $a->{'spread'} || 
  $b->{'oldr'}   <=> $a->{'oldr'} || 
  $b->{'name'}   cmp $a->{'name'} 
  }

# &WriteClub3($players)
sub WriteClub3{ local($ps) = @_;
  local($p, @ranked);

  &ratings2'CalculateRatings($ps, 'oldr', 1, 'newr', 10000, 'ewins1');
  @ranked = sort { 
    $b->{'newr'}   <=> $a->{'newr'} || 
    $a->{'name'}   cmp $b->{'name'} 
    } @$ps;

  printf "%-${global::nameLength}s   W-L   Sprd OldR NewR +-R PFor PAgn HiG\n\n",
    'Name';

  for $p (@ranked) {
    if ($p->{'games'}) {
      printf "%-${global::nameLength}s %3g-%-3g", 
	$p->{'fname'}, $p->{'ewins'}, $p->{'rgames'}-$p->{'ewins'};
      printf " %+4d", $p->{'spread'} unless $opt_S;
      if ($p->{'oldr'}) { 
	printf " %4d %4d %+3d ", $p->{'oldr'}, $p->{'newr'}, $p->{'newr'}-$p->{'oldr'}; 
	}
      else { printf " n.r. %4d     ", $p->{'newr'}; }
      printf "%4d %4d %3d", $p->{'tfor'}, $p->{'tagn'}, $p->{'hi'} 
	unless $opt_S;
      print "\n";
      }
    else {
      printf "%-${global::nameLength}s                   %4d\n",
        $p->{'fname'}, $p->{'newr'};
      }
    }
  }

use strict;

# &WriteNSA($players, $noinvert)
sub WriteNSA { 
  my $ps = shift;
  my $noinvert = shift;

  for my $pn (0..$#$ps) { 
    my $p = $ps->[$pn]; 
    printf "%d ", $pn+1;
    if ($noinvert) { print "$p->{'name'}/$p->{'spread'}:"; }
    else {
      my @n = split(/ /, $p->{'name'});
      if ($#n != 1) {
	if ($p->{'name'} =~ /^(?:\w+ [^o] \w+|mary ellen bergeron|robin pollock daniel|ida ann shapiro)$/i) 
	  { @n = ("$n[0] $n[1]", $n[2]); }
	elsif ($p->{'name'} =~ /^(?:muriel de silva|sherrie saint john|muriel sparrow reedy|john van zeyl|john van pelt|annie st denis|sharon crawford mackay)$/i) 
	  { @n = ($n[0], "$n[1] $n[2]"); }
	elsif ($p->{'name'} =~ /^(?:verna richards berg)$/i) 
	  { @n = ($n[0], "$n[2] $n[1]"); }
	elsif ($p->{'name'} =~ /^(?:james l kille jr)$/i) 
	  { @n = ("$n[0] $n[1] $n[3]", "$n[2]"); }
	elsif ($p->{'name'} =~ /^eugene van de walker$/i) 
	  { @n = ('eugene', 'van de walker'); }
	else { die "Don't know how to parse: $p->{'name'}\n"; }
	}
      print "\U@n[1,0]/$p->{'spread'}:";
      }
    my $os = $p->{'opps'};
    for my $round (0..$#$os) {
      my $o = $os->[$round];
      if ($o == -1) { print " B"; } # bye
      else {
	print ' ', 
	  ('L','T','W')[($p->{'scores'}[$round]<=>$ps->[$o]{'scores'}[$round])+1],
	  $o+1;
	}
      }
    print "\n";
    }
  }

# WritePairings @players
sub WritePairings (\@) { 
  my $ps = shift;
  our($opt_e, $opt_p);
  my $pfmt = '%' . (length($#$ps+1)+1) . 'd';
  if ($opt_p =~ /^koth$/i) {
    print "# automatically generated KOTH pairings\n";
    my @ranked = $opt_e eq 'nsc78' 
      ? (sort SortByCredits @$ps)
      : (sort SortByWinsAndCume @$ps);
    for my $i (0..$#ranked) {
      my $p = $ranked[$i];
      my $opp = sprintf($pfmt, $ranked[$i^1]->{'id'}+1);
      $p->{'input'} =~ s/;/$opp;/;
      }
    for my $p (@$ps) {
      print $p->{'input'};
      }
    }
  }

sub WriteReport (\@) { 
  my $ps = shift;
  our($opt_d, $opt_e, $opt_f, $opt_O, $opt_S);

  CalculateSeeds $ps;
  &ratings2'CalculateRatings($ps, 'oldr', 1, 'midr', $opt_d, 'ewins1');
  &ratings2'CalculateRatings($ps, 'midr', $opt_d+1, 'newr', 10000, 'ewins2');

  print 'Rank';
  print ' Seed' unless $opt_e eq 'nsc78';
  printf " %-${global::nameLength}s",
    $opt_O ? 'Name' : 'Name (online)';
  print ' Creds' if $opt_e eq 'nsc78';
  print ' Wins'; 
  print  " Cumul" unless $opt_S;
  print  " OldR NewR Chng" unless $opt_e eq 'nsc78';
  print  " For Agn  Hi" unless $opt_S;
  print "\n"; 

  my @ranked = $opt_e eq 'nsc78' 
    ? (sort SortByCredits @$ps)
    : (sort SortByWinsAndCume @$ps);
  my $i = 1;
  my $rank = 1;
  my $last_spread = -10000;
  my $last_wins = -10000;

  # print standings
  for my $p (@ranked) {
    next if $p->{'fname'} eq 'F NORD';
    if ($p->{'wins'} != $last_wins || $p->{'spread'} != $last_spread) {
      $last_wins = $p->{'wins'}; $last_spread = $p->{'spread'};
      $rank = $i;
      }
    printf '%3d ', $rank;
    printf "  %3d", $p->{'seed'} unless $opt_e eq 'nsc78';
    printf " %-${global::nameLength}s", $p->{'fname'};
    printf ' %5.1f', $p->{'credits'} if $opt_e eq 'nsc78';
    printf " %4.1f", $p->{'wins'};
    printf " %+5d", $p->{'spread'} unless $opt_S;
    unless ($opt_e eq 'nsc78') {
      if ($p->{'oldr'}) { 
	printf " %4d %4d %+4d", $p->{'oldr'}, $p->{'newr'}, $p->{'newr'}-$p->{'oldr'}; 
	}
      else { printf " n.r. %4d     ", $p->{'newr'}; }
      }
    printf " %3d %3d %3d", $p->{'afor'}+0.5, $p->{'aagn'}+0.5, $p->{'hi'} 
      unless $opt_S;
    print "\n";
    $p->{'rank'} = $i++;
    }
  print "\n";

  # print cross tables
  $rank = 1;
  for my $p ($opt_f ? @$ps : @ranked) {
    next if $p->{'fname'} eq 'F NORD';
    printf "%${global::numberLength}d  %-${global::nameLength}s ", $rank++, $p->{'fname'};
    my $pos = $global'numberLength + $global'nameLength + 2;
    for my $round (0..$#{$p->{'opps'}}) {
      if ($pos > $global'lineLength - ($global'numberLength + 2)) {
	printf "\n%${global::numberLength}s  %${global::nameLength}s ", '', '';
	$pos = $global'numberLength + $global'nameLength + 2;
	}
      my $o = $p->{'opps'}[$round];
      if ($o == -1) { printf "B%s ", ('-' x $global'numberLength); }
      elsif ($round > $#{$p->{'scores'}}) {
	printf "?%0${global::numberLength}d ",
	  $ps->[$o]{$opt_f ? 'id' : 'rank'} + $opt_f; 
	}
      else {
        printf "%s%0${global::numberLength}d ", 
  	  ('L','T','W')[($p->{'scores'}[$round]<=>$ps->[$o]{'scores'}[$round])+1],
	  $ps->[$o]{$opt_f ? 'id' : 'rank'} + $opt_f;
	}
      $pos += 2 + $global'numberLength;
      }
    print "\n";
    unless ($opt_S) { # $opt_S: suppress scores
      printf "%${global::numberLength}s  %-${global::nameLength}s ", '', '';
      $pos = $global'numberLength + $global'nameLength + 2;
      for my $round (0..$#{$p->{'scores'}}) {
	if ($pos > $global'lineLength - ($global'numberLength + 2)) {
	  printf "\n%${global::numberLength}s  %${global::nameLength}s ", '', '';
	  $pos = $global'numberLength + $global'nameLength + 2;
	  }
	my $o = $p->{'opps'}[$round];
	if ($o == -1) { 
	  my $nlp1 = $global'numberLength + 1;
	  printf "%+${nlp1}d ", $p->{'scores'}[$round]; 
	  }
	else {
	  printf "%s%3d ", (' ' x ($global'numberLength-2)),
	    $p->{'scores'}[$round]; 
	  }
	$pos += 2 + $global'numberLength;
	}
      print "\n";
      }
    }
  }

# &WriteSimpleTable($players)
sub WriteSimpleTable { 
  my $ps = shift;
  our($opt_e, $opt_S);

  my @ranked = $opt_e eq 'nsc78' 
    ? (sort SortByCredits @$ps)
    : (sort SortByWinsAndCume @$ps);
  my $i = 1;
  my $rank = 1;
  my $last_spread = -10000;
  my $last_wins = -10000;

  # print standings
  for my $p (@ranked) {
    if ($p->{'wins'} != $last_wins || $p->{'spread'} != $last_spread) {
      $last_wins = $p->{'wins'}; $last_spread = $p->{'spread'};
      $rank = $i;
      }
    my $wins = $p->{'wins'};
    $wins =~ s/\.5$/+/;
    my (@name) = split(/\s+/, $p->{'fname'});
    if (@name == 2) { }
    elsif (@name == 3) {
      if ($p->{'fname'} =~ /^(?:DANIEL ROBIN POLLOCK|BERGERON MARY ELLEN|WISNIEW DAWN CAMILLE|COHEN JO ANNE|AGDEPPA GLORIOSA ONDOY|MARIA JULIE ELLEN|GOODRICH ALICE ANN|RIBLE FRED III|SHAPIRO IDA ANN|SPARROW REEDY MURIEL|WEISSKOPF MARY ELLEN)$/) { }
      elsif ($p->{'fname'} =~ /^(?:\w+ \w+ \w|\w+ \w \w+)$/) { }
      elsif ($p->{'fname'} =~ /^(VAN PELT JOHN|SAITO STEWART PATRICIA|D AMBROSIO BRUCE|BERG RICHARDS VERNA|VAN ALEN BARBARA|POLAK SCOWCROFT CAROLINE)$/) { 
	splice(@name, 0, 2, join(' ', @name[0,1]));
        }
      else { die "Not sure how to split @name.\n"; }
      }
    elsif ($p->{'fname'} =~ /^(?:KILLE JAMES L JR)$/) {
      }
    else {
      die "Not sure how to split @name.\n";
      }
    printf "%d.\t%s, %s\t%s\t%+d\n",
      $rank, $name[0], join(' ', @name[1..$#name]), $wins, $p->{'spread'};
    $p->{'rank'} = $i++;
    }
  print "\n";
  }

# WriteSS(@players, $player_number)
sub WriteSS (\@$) { 
  my $ps = shift;
  my $pn = shift;
  our($opt_e);
  if (--$pn >= 0 && $pn <= $#$ps) {
    printf "Scoresheet for player %d: %s\n", 1+$pn, $ps->[$pn]{'fname'};
    my $p = $ps->[$pn];
    my $os = $p->{'opps'};
    my $cume = 0;
    my $l = 0;
    my $w = 0;
    for my $round (0..$#$os) {
      last if $round > $#{$p->{'scores'}};
      my $o = $os->[$round];
      if ($o == -1) { # bye
	my $psc = $p->{'scores'}[$round];
	$cume += $psc;
	my $result = (($psc <=> 0) + 1) / 2;
	$w += $result; $l += 1 - $result;
	die "Unimplemented" if $opt_e eq 'nsc78';
	printf "%${global::numberLength}d. %-${global::nameLength}s %4s %4.1f %4.1f  %3s %3s"
	  ." %+4d %+5d\n",
	  $round+1, 'bye', '', $w, $l, '', '', $psc, $cume;
	}
     else {
	$o = $ps->[$o];
	my $osc = $o->{'scores'}[$round];
	my $psc = $p->{'scores'}[$round];
        my $diff = $psc - $osc;
	if ($opt_e eq 'nsc78') {
	  my $score_credits = int($psc/50);
	  my $spread_credits = $diff > 0 ? int($diff/50) : 0;
	  my $win_credits = (($diff <=> 0) * 3 + 3)/2;
	  $cume += $score_credits + $spread_credits + $win_credits;
	  printf "%${global::numberLength}d. %-${global::nameLength}s %4d %3d %3d %+4d"
	    ." %4.1f %4.1f %4.1f %5.1f\n",
	    $round+1, $o->{'fname'}, $o->{'oldr'}, $psc, $osc, $diff, 
	    $win_credits, $score_credits, $spread_credits, 
	    $cume;
	  }
	else {
	  if ($opt_e eq 'tor80') {
	    if ($diff > 200) { $cume += 200; }
	    elsif ($diff < -200) { $cume -= 200; }
	    else { $cume += $diff; }
	    }
	  else
	    { $cume += $diff; }
	  my $result = (($diff <=> 0) + 1) / 2;
	  $w += $result; $l += 1 - $result;
	  printf "%${global::numberLength}d. %-${global::nameLength}s %4d %4.1f %4.1f  %3d %3d"
	    ." %+4d %+5d\n",
	    $round+1, $o->{'fname'}, $o->{'oldr'}, $w, $l, $psc, $osc, $diff, $cume;
	  }
	}
      }
    }
  else { 
    printf STDERR "Player number %d is outside of the range 1..%d.\n",
      ++$pn, $#$ps+1;
    }
  }
