#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TFile;

=pod

=head1 NAME

TFile - manipulate John Chew's Scrabble tournament .t files 

=head1 SYNOPSIS

  my $tf = new TFile 'a.t';
  while (my $datap = $tf->ReadLine()) {
    print "$datap->{'name'}\n";
    }
  $tf->Close();

  
=head1 ABSTRACT

This Perl module is used to read tournament data files in '.t' file format.

=head1 DESCRIPTION

=over 4

=cut

use Symbol;

sub new ($$);
sub Close($);
sub ReadLine($);

=item $tf = new TFile($filename);

Create a new TFile object.  
Opens file, returns undef on failure.

=cut

sub new ($$) {
  my $proto = shift;
  my $filename = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $fh = gensym;
  my $this = {
    'filename' => $filename,
    'id' => 0,
    };
  return undef unless open($fh, "<$filename");
  $this->{'handle'} = $fh;
  bless($this, $class);
  return $this;
  }

=item $success = $fh->Close();

Explicitly closes the .t file.

=cut

sub Close ($) {
  my $this = shift;
  
  close($this->{'handle'});
  delete $this->{'handle'};
  return 1;
  }

=item $datap = $fh->ReadLine();

Read and parse one line from the file.
Returns a hash whose keys are C<id>, C<name>, C<rating>, C<pairings>,
C<scores>, C<rnd> and C<etc>.

=cut

sub ReadLine ($) {
  my $this = shift;
  my %data;

  my $fh = $this->{'handle'};
  while (1) {
    local($_);
    $_ = <$fh>;
    return undef unless defined $_;
    s/#.*//; 
    s/^\s*//;
    s/\s*$//; 
    next unless /\S/;
    s/$/;/ unless /;/;
    my($player, $rating, $pairings, $scores, $etc) 
      = /^([^;]+[^;\s\d])\s+(\d+)\s*([\d\s]*);\s*([-\d\s]*)((?:;[^;]*)*)$/;
    die "Can't parse in $this->{'filename'}: $_\n" unless defined $scores;
    my (@pairings) = split(/\s+/, $pairings);
    my (@scores) = split(/\s+/, $scores);
    my $id = ++($this->{'id'});
    %data = (
#     'division' => $dp,
      'id'       => $id,
      'name'     => $player,
      'rating'   => $rating,
#     'rnd'=>rand,
      'rnd'      => ((length($player) * (100+$id) * ord($player)) % 641),
      'pairings' => [split(/\s+/, $pairings)],
      'scores'   => [split(/\s+/, $scores)],
      );
    for my $extra (split(/;\s*/, $etc)) {
      next unless $extra =~ /\S/;
      my ($tag, @words) = split(/\s+/, $extra);
      if (exists $data{$tag}) {
	die "Duplicate $tag field for $player.\n";
        }
      $data{'etc'}{$tag} = \@words;
      }
    last;
    }
  
  return \%data;
  }


=back

=cut

1;

