#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::InitFontes;

use strict;
use warnings;

use TSH::Player;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::InitFontes - implement the C<tsh> InitFontes command

=head1 SYNOPSIS

  my $command = new TSH::Command::InitFontes;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::InitFontes is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the InitFontes command to manually add three rounds suitable
for starting a Swiss-paired tournament.
Groups of four players are randomly drawn, one from each quartile,
to form round-robin groups of three.
EOF
  $this->{'names'} = [qw(if initfontes)];
  $this->{'argtypes'} = [qw(NumberOfIFRounds Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($nrounds, $dp) = @_;
  if ($dp->LastPairedRound0() != -1) {
    $tournament->TellUser('eifhasp');
    return 0; 
    }
  $tournament->TellUser('iifok', $dp->Name());

  # calculate pairings
  if ($nrounds == 3) {
    my @rrs = ();
    my $sortedp = $dp->GetUnpaired();
    TSH::Player::SpliceInactive @$sortedp, 3;
    @$sortedp = TSH::Player::SortByInitialStanding @$sortedp;
    my $np = $#$sortedp + 1;
    if ($np < 4) { $tournament->TellUser('eifneed4', $np); return; }
    if ($np % 4) { # players don't divide exactly into quads
      my @oddballs;
      my @formats;
      if ($np % 4 == 1) { # number of players is 1 (mod 4)
        # pick five players one from each quintile
	# TODO: think about avoiding giving byes to top seeds
        for (my $section=4; $section>=0; $section--) {
          push(@oddballs, splice(@$sortedp, int($np*($section+rand(1))/5),1));
          }
        @formats = ([3,4,undef,0,1],[1,0,3,2,undef],[undef,2,1,4,3]);
        }
      elsif ($np % 4 == 2) { # number of players is 2 (mod 4)
        # pick six players one from each sextile
        for (my $section=5; $section>=0; $section--) {
          push(@oddballs, splice(@$sortedp, int($np*($section+rand(1))/6),1));
          }
        @formats = ([5,2,1,4,3,0],[3,4,5,0,1,2],[1,0,3,2,5,4]);
        }
      elsif ($np % 4 == 3) { # number of players is 3 (mod 4)
	# TODO: think about avoiding giving byes to top seeds
        # pick three players one from each third
        for (my $section=2; $section>=0; $section--) {
          push(@oddballs, splice(@$sortedp, int($np*($section+rand(1))/3),1));
          }
        @formats = ([2,undef,0],[1,0,undef],[undef,2,1]);
        }
#     print "[", join(',', map($_->{'id'}, @oddballs)), "]\n";
      for my $format (@formats) {
            for my $i (0..$#$format) {
               my $opp = $format->[$i];
               $opp = defined $opp ? $oddballs[$opp]{'id'} : 0;
               push(@{$oddballs[$i]{'pairings'}}, $opp);
                }
        }
      }
    # at this point, number of remaining players in @$sortedp is divisible by four.
    if ($#$sortedp % 4 != 3) { die "Assertion failed."; }
    # repeatedly pick one random player from each quartile
    for (my $n4 = int(@$sortedp/4); $n4 > 0; $n4--) {
      my @rr = ();
# print "sortedp:[", join(',', map($_->{'id'}, @$sortedp)), "]\n";
      for (my $quartile = 3; $quartile >= 0; $quartile--) {
        push(@rr, splice(@$sortedp, $quartile*$n4 + rand($n4), 1));
        }
      push(@rrs, \@rr);
      }
    # assign pairings
    for my $rr (@rrs) {
#     print "[", join(',', map($_->{'id'}, @$rr)), "]\n";
      if ($#$rr == 3) {
        # RR4 gets paired 14.23, 13.24, 12.34
        for my $format ([3,2,1,0],[2,3,0,1],[1,0,3,2]) {
          for my $i (0..3) {
            push(@{$rr->[$i]{'pairings'}}, $rr->[$format->[$i]]{'id'});
            }
          }
        }
      }
    $dp->Dirty(1);
    $dp->Update();
    $tournament->TellUser('idone');
    }
  else { 
    $tournament->TellUser('eifnot3', $nrounds);
    }
  return;
  }

=back

=cut

=head1 BUGS

Makes some inappropriate use of TSH::Player internals.

=cut

1;
