#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::KOTH;

use strict;
use warnings;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::KOTH - implement the C<tsh> KOTH command

=head1 SYNOPSIS

  my $command = new TSH::Command::KOTH;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::KOTH is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the KOTH command to manually add a round of king-of-the-hill pairings
to a division, specifying the maximum number of allowable repeats,
and the round on whose standings the pairings are to be based.
EOF
  $this->{'names'} = [qw(koth)];
  $this->{'argtypes'} = [qw(Repeats BasedOnRound Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($repeats, $sr, $dp) = @_;
  my $sr0 = $sr-1;

  $dp->CheckRoundHasResults($sr0) or return 0;
  print "Calculating King-Of-The-Hill pairings for Division $dp->{'name'} based on round $sr.\n";
  
  my $sortedp = $dp->GetRegularUnpaired($sr0);
  unless (@$sortedp) { $tournament->TellUser('ealpaird'); return; }
  die "Assertion failed" unless @$sortedp % 2 == 0;
  @$sortedp = TSH::Player::SortByStanding $sr0, @$sortedp;

  { my $n = $#$sortedp; for my $i (0..$n) {
    my $p = $sortedp->[$i];
    my @pref = ();
    for my $j (1..$n) {
      {
	my $k = $i+$j;
	my $opp = $sortedp->[$k];
	push (@pref, $opp) if $k <= $n && $p->{'repeats'}[$opp->{'id'}] <= $repeats;
      }
      {
	my $k = $i-$j;
	my $opp = $sortedp->[$k];
	push (@pref, $opp) if $k >=0 && $p->{'repeats'}[$opp->{'id'}] <= $repeats;
      }
      } # for $j
    $p->{'pref'} = \@pref;
    } # for $i
  } # my $n
  if (main::ResolvePairings $sortedp) {
    $dp->Dirty(1);
    $dp->Update();
    }
  $tournament->TellUser('idone');
  }

=back

=cut

=head1 BUGS

Makes inappropriate use of private functions in TSH::Player.

Should be rewritten using PairGRT.

=cut

1;
