#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ShowscoreCard;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ShowscoreCard - implement the C<tsh> ShowscoreCard command

=head1 SYNOPSIS

  my $command = new TSH::Command::ShowscoreCard;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ShowscoreCard is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display an electronic version of a player's
scorecard, either to find discrepancies or to reprint a lost
card.
Specify the division name and player number of the scorecard
that you wish to view.
EOF
  $this->{'names'} = [qw(sc showscorecard)];
  $this->{'argtypes'} = [qw(Division Player)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($dp, $pn) = @_;
  my $p = $dp->Player($pn);
  unless (defined $p) { $tournament->TellUser('ebadp', $pn); return; }
  my $logp = new TSH::Log($dp, 'scorecard', "p$pn", 1);

  # print header
  {
    my $text = '';
    my $html = '';
    my $rating = $p->Rating();
    $text .= 'Player Scorecard: ';
    $text .= $p->TaggedName();
    $text .= " ($rating)";
    if ($config::track_firsts) {
      my $firsts = $p->{'p1'} || 0;
      my $seconds = $p->{'p2'} || 0;
      $text .= " (S${firsts}R$seconds)";
      }
    $text .= " INACTIVE" if exists $p->{'etc'}{'off'};
    $text .= "\n";
    {
      my $span = $config::track_firsts ? 3 : 2;
      my $dname = $dp->Name();
      my $pid = $p->ID();
      $html .= "<td class=number colspan=$span>\U$dname\E$pid</td>\n";
    }
    my $clean_name = $p->Name();
    $clean_name =~ s/,$//; # kludge to allow names ending in digits
    $html .= "<td class=name colspan=5><span class=label>Name:</span> $clean_name</td>\n";
    $html .= "<td class=rating colspan=2><span class=label>Rating:</span> $rating</td>\n";
    $html .= "</tr>\n";
    $logp->Write($text, $html);
  }
  
  if ($p->{'etc'}{'time'}) {
    my $age = int((time - $p->{'etc'}{'time'}[0])/60);
    if ($age < 200) {
      my $s = $age == 1 ? '' : 's';
      $logp->Write("- Last score was entered $age minute$s ago.\n", '');
      }
    }
  $logp->Write('', "<tr class=top2>");
  $logp->Write('Rnd ', "<th class=round>Round</th>\n");
  $logp->Write('1/2 ', '');
  $logp->Write('Opp ', "<th class=onum>Opp. #</th>\n");
  $logp->Write('Rtng ', "<th class=orat>Rating</th>\n");
  $logp->Write(sprintf("$config'name_format  ", 'Opponent Name'),
    "<th class=onam>Opponent Name</th>\n");
  $logp->Write('Won-Lost ',
    "<th class=won>Won</th><th class=lost>Lost</th>\n");
  $logp->Write('For ', "<th class=for>Player<br>Score</th>\n");
  $logp->Write('Agn ', "<th class=against>Opponent<br>Score</th>\n");
  $logp->Write('Sprd ', "<th class=spread>Spread</th>\n");
  $logp->Write('Cumul ', '');
  $logp->Write("\n", "</tr>\n");

  # print one line for each paired or played round
  my $pairingsp = $p->{'pairings'};
  my $scoresp = $p->{'scores'};
  my $max = $config'max_rounds ? $config'max_rounds - 1
    : ($#$pairingsp > $#$scoresp ? $#$pairingsp : $#$scoresp);
  my $cume = 0;
  my $games = 0;
  my $wins = 0;
  for my $r0 (0..$max) {
    my $r = $r0 + 1;
    my $oid = $pairingsp->[$r0];
    unless (defined $oid) { # no data yet for round
      $logp->Write('', 
        "<tr class=round1>\n"
	. "<td class=round>$r</td>\n"
	. "<td class=onum rowspan=2>&nbsp;</td>\n"
	. "<td class=orat rowspan=2>&nbsp;</td>\n"
	. "<td class=onam rowspan=2>&nbsp;</td>\n"
	. "<td class=won rowspan=2>&nbsp;</td>\n"
	. "<td class=lost rowspan=2>&nbsp;</td>\n"
	. "<td class=for rowspan=2>&nbsp;</td>\n"
	. "<td class=against>&nbsp;</td>\n"
	. "<td class=spread>&nbsp;</td>\n"
	. "</tr>\n"
        . "<tr class=round2>\n"
	. "<td class=p12>1st 2nd</td>\n"
        . "<td class=cumelabel>cumulative</td>"
	. "<td class=cume>&nbsp;</td>"
	. "</tr>\n"
      );
      next;
      }
    my $opp = $dp->Player($oid);
    my $score = $scoresp->[$r0];
    $logp->Write(sprintf("%3d ", $r), 
      "<tr class=round1" . ($oid ? '' : 'bye') . '>'
      . "<td class=round>$r</td>");
    my $p12;
    if ($config'track_firsts) {
      $p12 = $p->{'etc'}{'p12'}[$r0];
      $logp->Write(
        (!defined $p12 ? '    ' 
	  : ' ' . ('- ',"$p12 ","$p12 ",'? ', '??')[$p12] . ' '),
	'');
      }
    if ($oid) {
      $logp->Write(
        sprintf("%3d %4d $config'name_format",
	  $opp->{'id'}, $opp->{'rating'}, $opp->{'name'}),
	"<td class=onum rowspan=2>$opp->{'id'}</td>"
	  . "<td class=orat rowspan=2>$opp->{'rating'}</td>"
	  . "<td class=onam rowspan=2>$opp->{'name'}</td>"
	);
      }
    else {
      $logp->Write(
        sprintf("%3s %4s $config'name_format", '', 'bye', ''),
	"<td class=onum rowspan=2>-</td>"
	  . "<td class=orat rowspan=2>-</td>"
	  . "<td class=onam rowspan=2>bye</td>");
      }
    if (defined $score) {
      $games++;
      my $oscore = $p->OpponentScore($r0) || 0;
      my $spread = $score - $oscore;
      $cume += $spread;
      $wins += (1+($spread <=> 0))/2;
      my $losses = $games - $wins;
      $logp->Write(
        sprintf(" %4.1f-%4.1f %3d %3d %+4d %+5d",
	  $wins, $losses, $score, $oscore, $spread, $cume),
        "<td class=won rowspan=2>$wins</td>"
        . "<td class=lost rowspan=2>$losses</td>"
        . "<td class=for rowspan=2>$score</td>"
        . "<td class=against>$oscore</td>"
        . sprintf("<td class=spread>%+d</td>", $spread)
      );
      }
    else {
      $logp->Write('',
        "<td class=won rowspan=2>&nbsp;</td>"
        . "<td class=lost rowspan=2>&nbsp;</td>"
        . "<td class=for rowspan=2>&nbsp;</td>"
        . "<td class=against>&nbsp;</td>"
        . "<td class=spread>&nbsp;</td>"
      );
      }
    $logp->Write('', "</tr>\n"
      . "<tr class=round2" . ($oid ? '' : 'bye') . '>');
    if ($config'track_firsts) {
      $logp->Write('', 
	'<td class=p12>'
	. (!defined $p12 ? '' : ('-','1st&nbsp;&nbsp;&nbsp;&nbsp;','&nbsp;&nbsp;&nbsp;&nbsp;2nd','1st 2nd','1st 2nd')[$p12])
	. '</td>');
      }
    if (defined $score) {
      $logp->Write('', 
        "<td class=cumelabel><span class=label>cumulative</span><span class=strut>&nbsp;</span></td>"
        . sprintf("<td class=cume>%+d</td>", $cume)
	. "\n");
      }
    else {
      $logp->Write('', 
        "<td class=cumelabel><span class=label>cumulative</span><span class=strut>&nbsp;</span></td><td class=cume>&nbsp;</td>\n");
      }
    $logp->Write("\n", "</tr>\n");
    } # for $r0

  $logp->Write('', "<tr class=bottom><td colspan=9>&nbsp;</td></tr>\n");
  $logp->Close();
  }

=back

=cut

=head1 BUGS

Should display board/table number if set.

Code makes inappropriate use of private fields in Player.pm.

=cut

1;
