#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::showWallChart;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::showWallChart - implement the C<tsh> showWallChart command

=head1 SYNOPSIS

  my $command = new TSH::Command::showWallChart;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::showWallChart is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display a version of the self-posting
wall chart suitable for checking what the players have
posted themselves.  If you specify a round number, the
wall chart starts with that round; if you don't, it starts
at round 1.
EOF
  $this->{'names'} = [qw(wc showwallchart)];
  $this->{'argtypes'} = [qw(OptionalRound Divisions)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;

  my $from = 1;
  if (ref($_[0]) eq '') {
    $from = shift;
    }
  $from--;
  for my $dp (@_) {
    my $logp = new TSH::Log($dp, 'wallchart');

    $logp->Write(sprintf($config'name_format, 'Player'), "<th>Player</th>\n");
    for my $r0 ($from..$dp->{'maxs'}) {
      my $r = $r0 + 1;
      $logp->Write(sprintf(" Rd %02d", $r), "<th>Round $r</th>\n");
      }
    $logp->Write("\n", "</tr>\n");
    
    for my $p ($dp->Players()) {
      my $pname = $p->Name();
      my $line1 = sprintf("$config::name_format ", $pname);
      my $line2 = sprintf("$config::name_format ", '');
      my $html = "<tr><td class=name>$p->{'id'}. $pname</td>\n";
      my $spread = 0;
      my $wins = 0;
      my $cols = 0;
      my $id = $p->ID();
      for my $r (1..$p->CountScores()) {
	my $r0 = $r - 1;
        my $oppid = $p->OpponentID($r0);
	my $opp = $p->Opponent($r0);
	my $os = ($p->OpponentScore($r0) || 0);
	my $ms = $p->Score($r0);
        my $thisSpread = $ms - $os;
        $spread += $thisSpread;
        $wins += (1 + ($thisSpread <=> 0))/2;
	if ($r0 >= $from) {
	  $line1 .= sprintf("%5.1f ", $wins);
	  $line2 .= sprintf("%+5d ", $spread);
	  my $oppinfo = $dp->FormatPairing($r0, $id, 'brief');
	  if ($oppid) { 
	    $oppinfo = qq(<div class=opp><a href="#$oppid.$r" name="$id.$r">$oppinfo</a></div>); 
	    }
	  else {
	    $oppinfo = qq(<div class=bye>$oppinfo</div>);
	    }
	  $html .= sprintf(qq(<td class=%s>%s<div class=score>%d-%d</div><div class=record>%.1f-%.1f</div><div class=spread>%+d = %+d</div></td>\n),
	    $oppid ? $thisSpread > 0 ? 'win' : $thisSpread < 0 ? 'loss'
	      : 'tie' : 'bye',
	    $oppinfo,
	    $ms, $os,
	    $wins, $r - $wins,
	    $thisSpread, $spread,
	    );
	  }
        unless ((1+$cols-$from) % 9) {
	  if ($r0 >= $from) {
	    $logp->Write("$line1\n$line2\n", '');
	    $line1 = $line2 = sprintf("$config'name_format ", '');
	    }
          }
	$cols++;
        } # for $r0
      if ($line1 =~ /\S/) {
	$logp->Write("$line1\n$line2\n", '');
        }
      $logp->Write('', $html . "</tr>\n");
      } # for my $p
    $logp->Close();
    }
  }

=back

=cut

=bugs

=head1 BUGS

None reported.

=cut


1;
