#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::ParseArgs;

use strict;
use warnings;

use Carp;

=pod

=head1 NAME

TSH::ParseArgs - parse typed command-line arguments

=head1 SYNOPSIS

  # within main routine
  my $parser = new TSH::ParseArgs;
  my (@argv) = $parser->ParseArgs($argvp, \@argv_types);
  my $arg_parser = $parser->ArgumentParser('Division');
  my $error = $parser->Error();
  die $error if $error;

  # argument parser
  sub ParseArgs::Type::Parse ($$) {
    my $this = shift;
    my $line_parser = shift;
    my $value = $line_parser->GetArg();
    $line_parser->UnGetArg($value);
    }
  
=head1 ABSTRACT

This Perl module is used to validate arguments to commands entered
into C<tsh>.

=cut

# TODO: these should be dispatched using the OO mechanism
# %ParseArgsDispatch = (
#   'based-on-round' => \&ParseBasedOnRoundNumber,
#   'division' => \&ParseDivisionName,
#   'factor' => \&ParseFactor,
#   'nothing' => \&ParseNothing,
#   'nrounds' => \&ParseNRounds,
#   'player-number-or-0' => \&ParsePlayerNumberOrZero,
#   'player-number' => \&ParsePlayerNumber,
#   'repeats' => \&ParseRepeats,
#   'round' => \&ParseRoundNumber,
#   'round-or-0' => \&ParseBasedOnRoundNumber,
#   'score' => \&ParseScore,
#   );

=head1 DESCRIPTION

=over 4

=cut

sub GetArg ($);
sub Error ($;$);
sub initialise ($);
sub new ($);
sub LoadArgumentParser ($$);
sub Parse ($$$);
sub UnGetArg ($$);

=item $parserp->ArgumentParser($type);

Used externally to fetch an argument parser object.
Used internally to store a cached argument parser object.

=cut

sub ArgumentParser ($$;$) {
  my $this = shift;
  my $type = shift;
  my $parser = shift;
  my $class = "TSH::ParseArgs::$type";
  my $old = $this->{'parser_cache'}{$class};
  if (defined $parser) {
    $this->{'parser_cache'}{$class} = $parser;
    }
  elsif (!defined $old) {
    $old = $this->{'parser_cache'}{$class} = $this->LoadArgumentParser($type);
    }
  return $old;
  }

=item $parserp->GetArg()

Used by individual argument parsers to obtain the current argument
needing to be parsed.

=cut

sub GetArg ($) {
  my $this = shift;
  my $value = $this->{'argv'}[$this->{'parser_index'}++];
# unless (defined $value) {
#   confess "GetArg failed with p_i=$this->{'parser_index'}.\n";
#   }
  return $value;
  }

=item $parserp->Error();

Used externally to fetch the current parser error message,
or undef if none.
Used internally to set the current parser error message.

=cut

sub Error ($;$) {
  my $this = shift;
  my $message = shift;
  if (defined $message) {
    $message .= "\n" unless $message =~ /\n$/;
    if (defined $this->{'error'}) {
      $this->{'error'} .= $message;
      }
    else {
      $this->{'error'} = $message; 
      }
    }
  return $this->{'error'};
  }

=item $parserp->initialise();

Used internally to (re)initialise a parser object.

=cut

sub initialise ($) {
  my $this = shift;
  $this->{'division'} = undef;
  $this->{'error'} = undef;
  return $this;
  }

=item $parserp = new TSH::ParseArgs();

Create a new parser object.

=cut

sub new ($) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $this = {
    'parser_cache' => {},
    };
  bless($this, $class);
  {
    my $parser_class = 'TSH::ParseArgs::Ignore';
    eval "require $parser_class";
    die $@ if $@;
    $this->ArgumentParser('Ignore', $parser_class->new());
  }
  $this->initialise();
  return $this;
  }

=item $arg_parser = $parser->LoadArgumentParser($type);

Load the appropriate parser for the given type.
Should only be used internally for classes known to be as yet unloaded.
Loaded classes should be found using ArgumentParser().

=cut

sub LoadArgumentParser($$) {
  my $this = shift;
  my $type = shift;
  my $class = "TSH::ParseArgs::$type";
  my $parser = $this->{'parser_cache'}{$class};
  eval "require $class";
  die "Error loading code for parser of class $class: $@" if $@;
  eval { $parser = $class->new(); };
  if ($@) {
    my $error = "Error instantiating parser of class $class";
    $this->Error($error);
    TSH::Utility::Error($error);
    return ();
    }
  $this->ArgumentParser($type, $parser);
  return $parser;
  }

=item @checked_argv = $parserp->Parse(\@argv, \@types);

Check to make sure each member of @argv has the type indicated in
@types and return normalized values.

=cut

sub Parse ($$$) {
  my $this = shift;
  my $argvp = shift;
  my $typesp = shift;

  $this->initialise();
  $this->{'arg0'} = $argvp->[0];
  $this->{'argv'} = [@$argvp[1..$#$argvp]];
# print STDERR "argv has largest index $#$argvp and values @$argvp\n";
# print STDERR "types has largest index $#$typesp and values @$typesp\n";

  my @parsers;
  for my $type (@$typesp, 'Nothing') {
    push(@parsers, $this->ArgumentParser($type));
    }
  $this->{'parsers'} = \@parsers;
  $this->{'parser_index'} = 0;
  $this->{'parsed'} = [];
  while ($this->{'parser_index'} <= $#{$this->{'parsers'}}) {
    my (@values) = $this->{'parsers'}[$this->{'parser_index'}]->Parse($this);
    if (@values) {
      push(@{$this->{'parsed'}}, @values);
      }
    elsif ($this->Error()) {
      TSH::Utility::Error($this->Error());
      return ();
      }
    }
  return @{$this->{'parsed'}};
  }

=item $parserp->UnGetArg($value)

Used by individual argument parsers to return the current argument
if it didn't need to be parsed after all. 

=cut

sub UnGetArg ($$) {
  my $this = shift;
  my $value = shift;
  splice(@{$this->{'argv'}}, $this->{'parser_index'}-1, 1, undef, $value);
  return $value;
  }
=back

=cut

1;

