#!/usr/bin/perl -w

## tsh - tournament shell
## Copyright (C) 1998-2003 by John J. Chew, III.

=head1 NAME

B<tsh> - Scrabble tournament management shell

=head1 SYNOPSIS

B<tsh> [directory|configuration-file]

=head1 DESCRIPTION

For user information, see the accompanying HTML documentation.
This builtin (pod) documentation is intended for program maintenance 
and development.

=cut

## Version history
#
# 3.000 MWMST 2005
#   fixed a bug where if two games were stationed at the same board
#     one of them didn't appear in pairings
#   delete board assignments when unpairing rounds
#   HUH command, message database
#   added detailed help on many commands
#   DEBUG command
#   @config'external_path is now $config'external_path
# 2.980 NSC 2005
#   ResultsByRound logs to HTML and text files
#   external high-loss-round
#   external high-win-round
#   changed syntax of showWallChart
#   config save_interval
#   show12 shows counts of firsts and seconds
#   etc board
#   FactorPair1 command (experimental)
#   FactorPair command (experimental)
#   config prize_bands
#   config rating_system
#   external maketm
#   utility parsetm
#   rotisserie teams can be any size
#   'ShowPair' command reports more problems
# 2.975 For Anand Buddhdev
#   ABSPgrid command
#   config colour
#   external commands work with Windows XP
#   searching for config.tsh in subdirectories
#   Added external: update
# 2.960 Albany
#   Added tuffluck external
#   Partly functional version of ChewPair command
#   HTML scorecards
#   HTML wall charts
#   round-by-round html files are automatically indexed
#   RATING subcommand in EditScores
#   contextual help in EditScores
#   initial .t files may omit trailing ;
# 2.950 Toronto LCT
#   Safeguards against entering scores before earlier rounds complete
#   Added external: show12
#   first/second forecasts assignments are more accurate
# 2.940 Toronto LCT/Stamford CT
#   Optional specification of configuration file on command line
#   NewSwiss tries to minimize repeat pairings
# 2.930 CNSC
#   Addscore reports number of scores left
#   two-part tournaments rated correctly
# 2.920 Thunder Bay ON
#   Added support for some more three-part names in tourney.pl
#   Added externals: high-win high-loss low-win low-loss
#   Improved diagnostics
# 2.910 BAT
#   config gibson (alpha testing, warnings only)
#   config max_rounds
#   timestamps on data entry
#   ROTO
#   reserved table code updated
#   first/second forecasts are purged before data is synched
#   config no_text_files
#   config html_directory
#   can now autopair using p1324
#   fixed a bug that prevented calculating ratings of unrated players
# 2.900 BAT
#   Some errors now have numbers.
# 2.850 NSSC
#   Fixed a bug with RoundStanding introduced in 2.800 
# 2.840 NSSC
#   Fixed a bug introduced in 2.800 (?) where missing pairs were duplicated
# 2.830 NSSC
#   Slightly more readable and conformant HTML code
# 2.820 NSSC
#   Corrected determination of firsts and seconds
# 2.810 NSSC
#   WC and SC automatically measure max player name width
#   $config'assign_firsts
#   @config::external_path
#   %config'externals
#   $config'max_name_length
#   $config'name_format
#   MISSING and SP show "draws" if players must draw
#   'off' extension field for '.t' files
#   partial support for active/inactive players
#   hooks for external commands
# 2.800 NSSC
#   ES prompts with current scorecard before each command
#   added FIRST and SECOND subcommands to ES
#   MISSING and SP show p12 order if available
#   added "config track_firsts" configuration option to track 1sts & 2nds
#   Partially paired divisions can still be completed with autopair
#   Added tsh.config 'config' command
#   P1324 automatically assigns byes
#   KOTH automatically assigns byes
#   NS automatically assigns byes
#   BYE command replaced by PAIR and A
#   SP command doesn't list everyone if there aren't any pairings
#   PAIR command only reports changes
#   un-documented deprecated commands: SW PRESW PSW PPSW
#   added ES subcommand to A (goes into edit mode on last-entered score)
#   recorrected ES bug that sometimes discarded changes
#   corrected diagnostics emitted by UPR command
#   corrected documentation of P1324 command
#   InitFontes doesn't emit spurious errors with <8 players in division
#   several command argument parsers rewritten
#   RoundStandings reports results in partial current round correctly
#   division name doesn't have to be omitted in 1-division events
#   randomization of orderings made less so for stable table assignments
#   some spurious warnings deleted
#   support for new generalized .t files
#   HTML output from pairings command
# 2.740 Newsday SSC
#   'PairMany' command debugged 
# 2.730 Newsday SSC
#   'ShowPair' command reports more problems
# 2.720 Newsday SSC
#   'PairMany' command
# 2.710 Newsday SSC
#   Player names may include nonterminal numerals
# 2.700 Newsday SSC
#   HTML output from standings command
# 2.600 Oshawa 2005
#   'RoundRobin' command
#   'RATings' command
# 2.500 MWMST 2004
#   trying to autopair a round twice doesn't crash tsh.pl
# 2.400 Albany 2004
#   fixed 'wc -f #'
#   autopair works at start of tournament
#   missing lists divisions in alphabetical order
#   'm' synonym for 'missing' in score entry mode
# 2.300 BAT 2004
#   autopair
#   editscores emits updated scorecard
# 2.210 Montreal QC 2004-03 club tournament
#   suppressed a spurious error message
# 2.200 Cambridge ON 2004
#   "cambridge pairings"
# 2.100 CNSC 2003
#   Addscore gives updated cumes as a check
# 2.000 MWMST 2003
#   'missing' lists missing data in score entry mode
#   teed log files have '.doc' extension, no embedded spaces
# 1.900 CWSCQT 2003
#   added ShowPairings -p as kludge to work with make-rr.pl
# 1.800 Albany 2003
#   scorecard shows opp ratings
#   unpairround reports on problem players
#   table numbers can be arbitrary strings 
#   entering a division name changes divisions in score entry mode
#   ShowWallChart -f r lists beginning with round r only
# 1.700 Boston 2003
#   EditScore added
#   NewSwiss code added, first manually paired player can be 0 for an unscored bye pairing
# 1.600 Boston 2001
#   InitFontes works for any number of players, not just 4n and 4n+2
# 1.500 Toronto 2001
# 1.400 Boston 2001
# 1.300 Danbury 2001
# 1.200 incorporates 2000 CNSC changes
# 1.103 version used at 1999 Toronto SCRABBLE Tournament
# 1.000 first release

## public libraries

BEGIN { unshift(@::INC, "$::ENV{'HOME'}/lib/perl") if defined $::ENV{'HOME'}; }
use strict;
use lib './lib/perl';
use Fcntl ':flock';
use FileHandle;
use Symbol;

# use warnings FATAL => 'uninitialized';

## private libraries

use TSH::Command;
use TSH::Division;
use TSH::ParseArgs;
use TSH::Log;
use TSH::Player;
use TSH::Tournament;
use TSH::XCommand;
use UserMessage;

require 'dawg.pl';

## global constants
if ($^O eq 'MacOS') { 
  $config'backup_directory = ':old:'; 
  $config'html_directory = ':html'; 
  }
else { 
  $config'backup_directory = './old/'; 
  $config'html_directory = './html/'; 
  }
$config'max_name_length = 22;
$config'name_format = '%-22s';
$config'external_path = [qw(./bin)];
our $gkVersion = '3.000';

## prototypes

sub CheckGroupRepeats ($$);
sub ChooseBye ($$$);
sub ChooseConfig ();
# sub CmdFactorPair ($$);
# sub CmdFactorPair1 ($$);
sub CmdLook ($$);
sub CmdPair ($$);
sub CmdPair1324 ($$);
sub CmdPairMany ($$);
sub CmdPartialSwiss ($$);
sub CmdPreSwiss ($$);
sub CmdPrePreSwiss ($$);
sub CmdRandomScores ($$);
sub CmdRoto ($$);
sub CmdRoundRobin ($$);
sub CmdRoundStandings ($$);
sub CmdStandings ($$);
sub CountByes ($);
sub DefineExternal ($$$);
sub DoClark (\%$);
# sub DoFactor ($$$$);
# sub DoFactorGroup ($$$);
sub DoSwiss ($$$$;$);
sub Error ($);
sub lint ();
sub LockFailed ($);
sub LockOff ();
sub LockOn ();
sub Main ();
sub MakePlayerMap ();
sub ParseArgs ($$);
sub ParseBasedOnRoundNumber ($$);
sub ParseDivisionName ($$);
sub ParseFactor ($$);
sub ParseInteger ($$$$$);
sub ParseNothing ($$);
sub ParseNRounds ($$);
sub ParsePlayerNumber ($$);
sub ParsePlayerNumberOrZero ($$);
sub ParseRepeats ($$);
sub ParseRoundNumber ($$);
sub ParseScore ($$);
sub Prompt ();
sub ReadConfig ($);
sub ReadDivision (\%);
sub ReadDivisions ();
sub ReopenConsole ();
sub ResolvePairings ($;$);
sub SynchDivision ($);

## global variables

%global'commands = (
# 'fp' => \&CmdFactorPair,
# 'fp1' => \&CmdFactorPair1,
# 'factorpair' => \&CmdFactorPair,
# 'factorpair1' => \&CmdFactorPair1,
  'l' => \&CmdLook,
  'look' => \&CmdLook,
  'pair' => \&CmdPair,
  'pairmany' => \&CmdPairMany,
  'p1324' => \&CmdPair1324,
  'pair1324' => \&CmdPair1324,
  'partialswiss' => \&CmdPartialSwiss,
  'pm' => \&CmdPairMany,
  'ppsw' => \&CmdPrePreSwiss,
  'prepreswiss' => \&CmdPrePreSwiss,
  'presw' => \&CmdPreSwiss,
  'preswiss' => \&CmdPreSwiss,
  'psw' => \&CmdPartialSwiss,
  'rand' => \&CmdRandomScores,
  'roundrobin' => \&CmdRoundRobin,
  'rr' => \&CmdRoundRobin,
  'rs' => \&CmdRoundStandings,
  'roto' => \&CmdRoto,
  'roundstandings' => \&CmdRoundStandings,
  'randomscores' => \&CmdRandomScores,
  'st' => \&CmdStandings,
  'standings' => \&CmdStandings,
  );
%global'player_name_to_id = ();
my %gDivision;
my $gNDivisions;
# $gTournament should gradually subsume %gDivision, $gNDivisions and others
our $gTournament; 

=head1 SUBROUTINES

=over 4

=cut

sub CheckGroupRepeats ($$) {
  my $psp = shift;
  my $repeats = shift;
  for my $i (0..$#$psp) {
    my $repeatsp = $psp->[$i]{'repeats'};
    die "Player $psp->[$i]{'name'} has no repeats information.\n"
      unless ref($repeatsp) eq 'ARRAY';
    for my $j ($i+1..$#$psp) {
      if ((my $this_repeats = $repeatsp->[$psp->[$j]{'id'}]) > $repeats) {
#	Error "Warning: $psp->[$i]{'name'} and $psp->[$j]{'name'} have played each other $this_repeats time(s).\n";
	return 0;
        }
      }
    }
  return 1;
  }

sub ChooseConfig () {
  $config'root_directory = '.';
  # first choice: something specified on the command line
  if (@::ARGV) {
    my $argv = shift @::ARGV;
    # if it's a directory
    if (-d $argv) {
      for my $try (qw(config.tsh tsh.config)) {
	if (-e "$argv/$try") {
	  $config'root_directory = $argv;
	  return $try;
	  }
        }
      die "$argv is a directory but has neither config.tsh nor tsh.config\n";
      }
    # else it's a configuration file
    elsif (-f $argv) {
      if ($argv =~ /\//) {
	($config'root_directory, $argv) = $argv =~ /^(.*)\/(.*)/;
	return $argv;
        }
      else { return $argv; }
      }
    else { die "No such file or directory: $argv\n"; }
    }
  # second choice: the directory containing the newest of */config.tsh
  if (my (@glob) = glob('*/config.tsh')) {
    # if more than one, choose newest
    if (@glob > 1) {
      my $newest_age = -M $glob[0];
      while (@glob > 1) {
	my $this_age = -M $glob[1];
	if ($this_age < $newest_age) {
	  $newest_age = $this_age;
	  shift @glob;
	  }
	else {
	  splice(@glob, 1, 1);
	  }
        }
      }
    ($config'root_directory, $glob[0]) = $glob[0] =~ /^(.*)\/(.*)/;
#   print "Directory: $config'root_directory\n";
    if ($config'root_directory =~ /^sample\d$/) {
      print "If you didn't mean to use one of the sample directories,\n";
      print "you need to create a subdirectory of files for your own event.\n";
      }
    return $glob[0];
    }
  # third choice: ./tsh.config
  if (-f 'tsh.config') {
    # don't use colour here, as colour configuration hasn't been found
    print "Warning: use of tsh.config is deprecated.\n";
    print "Please place your event files in a subdirectory, and save your\n";
    print "configuration in that subdirectory under the name config.tsh.\n";
    }
  return 'tsh.config';
  }

# FactorPair is not yet ready for public use
# sub CmdFactorPair ($$) { my($argvp, $args) = @_;
#   my ($factor, $repeats, $sr, $dp) 
#     = ParseArgs $argvp, [qw(factor repeats based-on-round division)];
#   return 0 unless defined $dp;
#   my $sr0 = $sr-1;
#   $dp->CheckRoundHasResults($sr0) or return 0;
#   print "Calculating Factored Pairings for Division $dp->{'name'} based on round $sr, $repeats repeats allowed, factoring by $factor.\n";
#   DoFactor $dp, $repeats, $sr0, $factor;
#   return 0;
#   }
# 
# sub CmdFactorPair1 ($$) { my($argvp, $args) = @_;
#   my ($factor, $repeats, $sr, $dp) 
#     = ParseArgs $argvp, [qw(factor repeats based-on-round division)];
#   return 0 unless defined $dp;
#   my $sr0 = $sr-1;
#   $dp->CheckRoundHasResults($sr0) or return 0;
#   print "Calculating one-round factored pairings for Division $dp->{'name'} based on round $sr.\n";
#   
#   my $sortedp = $dp->GetRegularUnpaired($sr0);
#   unless (@$sortedp) {
#     Error "No players can be paired.\n";
#     return 0;
#     }
#   die "Assertion failed" unless @$sortedp % 2 == 0;
#   @$sortedp = TSH::Player::SortByStanding $sr0, @$sortedp;
# 
#   { my $n = $#$sortedp; for my $i (0..$n) {
#     my $p = $sortedp->[$i];
#     my @pref = ();
#     for my $j (1..$n) {
#       {
# 	my $k = $i+$factor+$j;
# 	my $opp = $sortedp->[$k];
# 	push (@pref, $opp) if $k <= $n && $p->{'repeats'}[$opp->{'id'}] <= $repeats;
#       }
#       {
# 	my $k = $i+$factor-$j;
# 	my $opp = $sortedp->[$k];
# 	die $k unless defined $opp;
# 	push (@pref, $opp) 
# 	  if $k >=0 && $p->{'repeats'}[$opp->{'id'}] <= $repeats;
#       }
#       } # for $j
#     $p->{'pref'} = \@pref;
#     } # for $i
#   } # my $n
#   if (ResolvePairings $sortedp) {
#     $dp->Dirty(1);
#     $dp->Update();
#     }
#   0;
#   }
  
sub CmdLook ($$) {
  my ($argvp) = @_;
  shift @$argvp;

  Error "The word lookup feature is not enabled in this copy.\n";
  return 0;
  my $ok = 1;
  for my $word (@$argvp) {
    if (&dawg'check(*WORDS, lc $word)) { 
#     print "'$word' is acceptable.\n"; 
      }
    else { 
#     print "'$word' is not acceptable.\n";
      $ok = 0; 
      }
    }
  printf "The play is%s acceptable.\n", $ok ? '' : ' not';
  0;
  }

# TODO: fix pairings for opponents of repaired players
sub CmdPair ($$) { my ($argvp, $args) = @_;
  my ($p1, $p2, $round, $dp) 
    = ParseArgs $argvp, [qw(player-number-or-0 player-number round division)];
  return 0 unless defined $dp;
  my $datap = $dp->{'data'};

  for my $p ($p1,$p2) {
    unless ($p <= $#$datap) { Error "No such player: $p\n"; return 0; }
    }

  $round--;

  my $p1pair = $datap->[$p1]{'pairings'} if $p1;
  my $p2pair = $datap->[$p2]{'pairings'} if $p2;
  printf "%s used to be paired to %s\n", 
    (TSH::Utility::TaggedName $datap->[$p1]), 
    (TSH::Utility::TaggedName $datap->[$p1pair->[$round]])
    if $p1 && defined $p1pair->[$round] && $p2 != $p1pair->[$round];
  printf "%s used to be paired to %s\n", 
    (TSH::Utility::TaggedName $datap->[$p2]), 
    (TSH::Utility::TaggedName $datap->[$p2pair->[$round]])
    if defined $p2pair->[$round] && $p1 != $p2pair->[$round];
  $p1pair->[$round] = $p2 if $p1;
  $p2pair->[$round] = $p1 if $p2;
    
  $dp->Dirty(1);
  $gTournament->UpdateDivisions();
  0;
  }

sub CmdPair1324 ($$) { my($argvp, $args) = @_;
  my ($repeats, $sr, $dp)
    = ParseArgs $argvp, [qw(repeats based-on-round division)];
  return 0 unless defined $dp;
  my $sr0 = $sr - 1;
  $dp->CheckRoundHasResults($sr0) or return 0;

  print "Calculating 1324 pairings for Division $dp->{'name'} based on round $sr.\n";

  my $sortedp = $dp->GetRegularUnpaired($sr0);
  unless (@$sortedp) {
    Error "No players can be paired.\n";
    return 0;
    }
  die "Assertion failed" unless @$sortedp % 2 == 0;
  @$sortedp = TSH::Player::SortByStanding $sr0, @$sortedp;

  { my $n = $#$sortedp; for my $i (0..$n) {
    my $p = $sortedp->[$i];
    my @pref = ();
    for my $j (2,1,3..$n) {
      {
	my $k = $i+$j;
	my $opp = $sortedp->[$k];
	push (@pref, $opp) if $k <= $n && $p->{'repeats'}[$opp->{'id'}] <= $repeats;
      }
      {
	my $k = $i-$j;
	my $opp = $sortedp->[$k];
	push (@pref, $opp) if $k >=0 && $p->{'repeats'}[$opp->{'id'}] <= $repeats;
      }
      } # for $j
    $p->{'pref'} = \@pref;
    } # for $i
  } # my $n
  if (ResolvePairings $sortedp) {
    $dp->Dirty(1);
    $dp->Update();
    }
  0;
  }
  
# TODO: fix pairings for opponents of repaired players
sub CmdPairMany ($$) { my ($argvp, $args) = @_;
  my ($round, $dp) = ParseArgs $argvp, [qw(round division)];
  return 0 unless defined $dp;
  my $datap = $dp->{'data'};

  while (1) {
    print((TSH::Utility::Colour('blue', "[$dp->{'name'}${round}]:pn1 pn2?")), ' ');
    local($_) = scalar(<STDIN>);
    s/\s+$//;
    if (defined $gDivision{lc $_}) {
      $dp = $gDivision{lc $_};
      $datap = $dp->{'data'};
      last unless defined $datap;
      next;
      }
    elsif (/^\s*r\s*(\d+)\s*$/i) {
      $round = $1;
      next;
      }
    last if /[^-\d\s]/;
    my (@pns) = split;
    last if @pns != 2;
    my $errors = 0;
    for my $i (0..1) {
      my $pn = $pns[$i];
      my $oppn = $pns[1-$i];
      if ($pn > $#$datap) {
	Error "There is no player #$pn.\n";
	$errors++;
	next;
        }
      next unless $pn;
      my $pp = $datap->[$pn];
      my $oldoppn = $pp->{'pairings'}[$round-1];
      if (defined $oldoppn) {
	next if $oldoppn == $oppn;
	print "#$pn $pp->{'name'} used to be paired to #$oldoppn $datap->[$oldoppn]{'name'}\n" 
        }
      $dp->Dirty(1);
      $pp->{'pairings'}[$round-1] = $oppn;
      }
    {
      printf "#%d %s (%.1f %+d) - #%d %s (%.1f %+d).\n",
        $datap->[$pns[0]]{id}, 
        $datap->[$pns[0]]{name}, 
	$datap->[$pns[0]]{wins},
	$datap->[$pns[0]]{spread},
        $datap->[$pns[1]]{id}, 
        $datap->[$pns[1]]{name}, 
	$datap->[$pns[1]]{wins},
	$datap->[$pns[1]]{spread},
	;
    }
    }
  $gTournament->UpdateDivisions();
  0;
  }

sub CmdPartialSwiss ($$) { my($argvp, $args) = @_;
  Error "WARNING: deprecated\n";
  shift @$argvp;
  my $usage = "Usage: Swiss repeats p1 p2 division-name\n";
  my $repeats = shift @$argvp;
  unless (defined $repeats && $repeats =~ /^\d+$/) { Error $usage; return 0; }
  my $p1 = shift @$argvp;
  unless (defined $p1 && $p1 =~ /^\d+$/ && $p1 >= 1) { Error $usage; return 0; }
  my $p2 = shift @$argvp;
  unless (defined $p2 && $p2 =~ /^\d+$/ && $p2 >= $p1) 
    { Error $usage; return 0; }
  my $dp = (ParseDivisionName $argvp, $usage);
  return 0 unless defined $dp;
  my $datap = $dp->{'data'};

  unless ($p2 <= $#$datap) { Error $usage; return 0; }

  DoSwiss $dp, $repeats, $p1, $p2;
  0;
  }

# Swiss based on round n-3, not n-1
sub CmdPrePreSwiss ($$) { my($argvp, $args) = @_;
  Error "WARNING: deprecated\n";
  shift @$argvp;
  my $usage = "Usage: PrePreSWiss repeats division-name\n";
  my $repeats = shift @$argvp;
  unless (defined $repeats && $repeats =~ /^\d+$/) { Error $usage; return 0; }
  my $dp = (ParseDivisionName $argvp, $usage);
  return 0 unless defined $dp;

  DoSwiss $dp, $repeats, 1, $#{$dp->{'data'}}, 2;
  0;
  }

# Swiss based on round n-2, not n-1
sub CmdPreSwiss ($$) { my($argvp, $args) = @_;
  Error "WARNING: deprecated\n";
  shift @$argvp;
  my $usage = "Usage: PRESWiss repeats division-name\n";
  my $repeats = shift @$argvp;
  unless (defined $repeats && $repeats =~ /^\d+$/) { Error $usage; return 0; }
  my $dp = (ParseDivisionName $argvp, $usage);
  return 0 unless defined $dp;

  DoSwiss $dp, $repeats, 1, $#{$dp->{'data'}}, 1;
  0;
  }

sub CmdRandomScores ($$) { my($argvp, $args) = @_;
  shift @$argvp;
  my $usage = "Usage: RandomScores division-name...\n";
  my $distrib =
    $config::rating_system && $config::rating_system =~ /^absp$/i
      ? 'absp' : 'nsa';
  do { {
    my $dp = (ParseDivisionName $argvp, $usage);
    next unless defined $dp;
    print "Adding random scores to division $dp->{'name'}.\n";
    for my $p ($dp->Players()) {
      next if $#{$p->{'scores'}} >= $#{$p->{'pairings'}};
      push(@{$p->{'scores'}}, 
	$distrib eq 'nsa'
	  # based on the NSA Club #3 2004-2005 season
	  ? int(130*log(($p->{'rating'}||1000)/75)
	    +rand(200)-100)
	  # based on nothing
	  : int(300+rand($p->{'rating'})));
      printf "%s: @{$p->{'scores'}}\n", (TSH::Utility::TaggedName $p);
      }
    $dp->Dirty(1);
    } } while ($#$argvp>=0);
  $gTournament->UpdateDivisions();
  0;
  }

sub CmdRoundRobin ($$) { my ($argvp, $args) = @_;
  my ($dp) = ParseArgs $argvp, [qw(division)];
  return 0 unless defined $dp;
  my $datap = $dp->{'data'};
  my $nplayers = $#$datap;

  # check that division is not partially paired
  {
    my $unpaired = $dp->GetUnpaired('can be empty');
    my $nunpaired = $#$unpaired+1;
    if ($nunpaired > 0 && $nunpaired != $nplayers) {
      Error "Can't add a round robin to a division whose last round is partially paired.\n";
      print "$nunpaired/$nplayers unpaired.\n";
      return 0;
      }
  }
# TSH::Player::SpliceInactive @$unpaired, $nplayers-1;

  print "Calculating round robin pairings for Division $dp->{'name'}.\n";
  # add pairings information one round at a time
  my $schedule_size = $nplayers + ($nplayers % 2);
  for (my $oppi = $schedule_size; $oppi > 1; $oppi--) {
    DoClark %$dp, $oppi;
    }

  $dp->Dirty(1);
  $gTournament->UpdateDivisions();
  0;
  }

sub CmdRoto ($$) { my($argvp, $args) = @_;
  my ($round) = ParseArgs $argvp, [qw(based-on-round)];
  return 0 unless defined $round;
  my $round0 = $round - 1;
  unless ($config'rotofile) {
    Error "No roto file defined.\n";
    return 0;
    }
  MakePlayerMap;
  my $fh = gensym;
  unless (open($fh, "<$config'root_directory/$config'rotofile")) {
    Error "Can't load $config'rotofile.\n";
    return 0;
    }
  local($/) = "\n\n";
  my @teams;
  my $team_size;
  while (<$fh>) {
    s/^\n+//; 
    my (@lines) = split(/\n/);
    my %players;
    my (%data);
    for my $line (@lines) {
      next unless $line =~ /\S/;
      $line =~ s/\s+$//;
      my ($command, $args) = split(/\s+/, $line, 2);
      if ($command eq 'owner') {
	if ($data{'owner'}) {
	  Error "Team $. has two owners ($data{'owner'} and $args)\n";
	  return 0;
	  }
	$data{'owner'} = $args;
        }
      elsif ($command eq 'player') {
	my $did = $global'player_name_to_id{$args};
	if (!defined $did) {
	  Error "Team $. has an unknown player: $args\n";
	  return 0;
	  }
	if ($players{$did}++) {
	  Error "Team $. has a duplicate player: $args\n";
	  return 0;
	  }
	my ($divname, $id) = $did =~ /^(.)(.*)/;
	my $p = $gDivision{lc $divname}{'data'}[$id];
	unless ($p) {
	  Error "Can't find player $did.\n";
	  return 0;
	  }
	push(@{$data{'players'}}, $p);
	$data{'wins'} += $p->{'twins'} = defined $p->{'rwins'}[$round0] 
	  ? $p->{'rwins'}[$round0] : $p->{'wins'} ;
	$data{'spread'} += $p->{'tspread'} = defined $p->{'rspread'}[$round0]
	  ? $p->{'rspread'}[$round0] : $p->{'spread'} ;
        $p->{'twins'} =~ s/\.5/+/;
        }
      else {
	Error "Can't make sense of: $line\n";
	return 0;
        }
      }
    if (defined $team_size) {
      my $this_team_size = scalar(@{$data{'players'}});
      if ($team_size != $this_team_size) {
	warn "$data{'owner'} has $this_team_size player(s), not $team_size.\n";
	$team_size = $this_team_size;
        }
      }
    else {
      $team_size = scalar(@{$data{'players'}});
      }
    push(@teams, \%data ) if %data;
    }
  my $logp = new TSH::Log(undef, 'roto', $round);
  $logp->Write(
    sprintf("%3s "
      . '%5s '
      . "%-30s %s\n", ' W ', 
      ' Sprd',
      'Owner', (' ' x 19) . 'Team'),
    "<th class=wins>Wins</th>"
      . '<th class=spread>Spread</th>'
      . "<th class=owner>Owner</th><th class=team colspan=$team_size>Team</th></tr>\n");
  for my $team (sort { $b->{'wins'} <=> $a->{'wins'} || 
    $b->{'spread'} <=> $a->{'spread'} ||
    lc($a->{'owner'}) cmp lc($b->{'owner'}) } @teams) {
    my $teamwins = $team->{'wins'};
    $teamwins =~ s/\.5/+/ or $teamwins .= ' ';
    $logp->Write(
      sprintf("%3s %+5d %-27.27s" . ('%6s' x $team_size) . "\n",
	$teamwins,
	$team->{'spread'},
	$team->{'owner'},
	map { sprintf("%3s%1s%03d",
	  $_->{'twins'},
	  uc($_->{'division'}{'name'}),
	  $_->{'id'},
	  )
	} @{$team->{'players'}}),
      sprintf("<tr><td class=wins>%s</td>"
        . '<td class=spread>%+d</td>'
	. "<td class=owner>%s</td>\n" 
	. ("<td class=team>%s</td>\n" x $team_size) . "</tr>\n",
	$teamwins,
	$team->{'spread'},
	$team->{'owner'},
	map { 
#	  $_->{'twins'} . ' ' . (TSH::Utility::TaggedName $_) 
          my $s = TSH::Utility::TaggedName $_;
	  if ($s =~ /(.*) \((.*)\)/) {
	    $s = sprintf("%s<br>%s %s %+d\n", $1, $2, $_->{'twins'},
	      $_->{'tspread'});
	    }
	  $s;
	  } @{$team->{'players'}}),
        );
    }
  $logp->Close();

  0;
  }

sub CmdRoundStandings ($$) { my($argvp, $args) = @_;
  my ($round, $dp)
    = ParseArgs $argvp, [qw(based-on-round division)];
  return 0 unless defined $dp;
  my $round0 = $round - 1;
  $dp->CheckRoundHasResults($round0) or return 0;

  my $dname = $dp->Name();
  print "Round $round Standings: Division $dname.\n";
  print "\n";
  print "Rank  Won-Lost Spread Rtng Player\n\n";

  $dp->ComputeRanks($round0);
  my (@sorted) = TSH::Player::SortByStanding $round0, $dp->Players();
  TSH::Player::SpliceInactive(@sorted);
  for my $p (@sorted) {
    my $wins = $p->RoundWins($round0);
    my $games = $p->GamesPlayed();
    $games = $round0 + 1 if $round0 + 1 < $games;
    my $losses = $games - $wins;

    printf "%4d %4.1f-%4.1f %+5d  %4d %s\n", 
      $p->RoundRank($round0),
      $wins, $losses, $p->RoundSpread($round0),
      $p->Rating(), ($p->TaggedName()) unless $p->{name} =~ /^bye /;
    }
  0;
  }

sub CmdStandings ($$) { my($argvp, $args) = @_;
  my ($dp) = ParseArgs $argvp, [qw(division)];
  return 0 unless defined $dp;
  my $datap = $dp->{'data'};

  my $round = $dp->{'maxs'} + 1;
  my $logp = new TSH::Log($dp, 'standings', $round);

  $logp->Write("Rank  Won-Lost Spread Rtng Name\n\n", <<'EOF');
<tr>
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
<th class=rating>Rating</th>
<th class=name>Name</th>
</tr>
EOF

  my $lastw = -1; my $lasts = 0; my $rank = 0; my $i = 0;
  for my $p (TSH::Player::SortByCurrentStanding @$datap[1..$#$datap]) {
    next if exists $p->{'etc'}{'off'};
    my $wins = $p->{'wins'};
    my $spread = $p->{'spread'};
    $i++;
    if ($wins != $lastw || $spread != $lasts) {
      $lastw = $wins;
      $lasts = $spread;
      $rank = $i;
      }
    unless ($p->{name} =~ /^bye /) {
      my (@fields) = ($rank, $wins, $#{$p->{'scores'}}+1-$wins, 
	  $spread, $p->{'rating'}, (TSH::Utility::TaggedName $p));
      $logp->Write(
        sprintf("%4d %4.1f-%4.1f %+5d  %4d %s\n", @fields),
	sprintf(<<'EOF', @fields));
<tr>
<td class=rank>%d</td>
<td class=wl>%4.1f-%4.1f</td>
<td class=spread>%+d</td>
<td class=rating>%d</td>
<td class=name>%s</td>
</tr>
EOF
      }
    }
  $logp->Close();
  0;
  }

# Count how many byes each player in a division has, save as $p->{'byes'}.
# Return the smallest number of byes.
sub CountByes ($) {
  my $dp = shift;

  my $datap = $dp->{'data'};
  my $minbyes = 9999999;

  for my $p (@$datap[1..$#$datap]) {
    next unless defined $p;
    my $byes = 0;
    for my $opp (@{$p->{'pairings'}}) {
      if ((defined $opp) && $opp == 0) {
	$byes++;
        }
      }
    $minbyes = $byes if $byes < $minbyes;
    $p->{'byes'} = $byes;
    }
  return $minbyes;
  }

sub DefineExternal ($$$) {
  my $name = lc shift;
  my $script = shift;
  my $template = shift;

  my $command = new TSH::XCommand("$global'path/$script", [$name], $template);
  $gTournament->AddCommand($command);
# print " $name";
  return 1;
  }

# DoFactor is not yet ready for public use
# sub DoFactor ($$$$) { my ($dp, $repeats, $sr0, $factor) = @_;
#   my $datap = $dp->{'data'};
#   my $theKeyRound = $sr0;
# 
#   my $tobepaired = $dp->GetRegularUnpaired($sr0, 'nobyes');
#   unless (@$tobepaired) {
#     Error "No players can be paired.\n";
#     return 0;
#     }
# # die "Assertion failed" unless @$tobepaired % 2 == 0;
#   my $minbyes = 0;
#   if (@$tobepaired % 2) {
#     $minbyes = CountByes $dp;
#     }
#   my (@ranked) = TSH::Player::SortByStanding $theKeyRound, @$tobepaired;
# 
#   my @pair_list;
#   my $group_number = 0;
#   while (@ranked) {
#     $group_number++;
#     my (@group) = @ranked > $factor + $factor
#       ? splice(@ranked, 0, $factor)
#       : splice(@ranked, 0);
#     my (@group_list) = DoFactorGroup \@group, $repeats, $minbyes;
#     unless (@group_list) {
#       Error "Can't factor group #$group_number. Division is partially paired.\n";
#       last;
#       }
#     push(@pair_list, @group_list);
#     }
# 
#   # store pairings
#   {
#     my $board = 1;
#     while (@pair_list) {
#       my $gp = shift @pair_list;
#       # make sure previous board numbers are set
#       for my $pp (@$gp) {
# 	my $tp = $pp->{'etc'}{'board'};
# 	my $pairingsp = $pp->{'pairings'};
# 	if (!defined $tp) {
# 	  $pp->{'etc'}{'board'} = [ (0) x @$pairingsp ];
# 	  }
# 	elsif ($#$tp < $#$pairingsp) {
# 	  push(@{$pp->{'etc'}{'board'}}, (0) x $#$pairingsp - $#$tp);
# 	  }
#         }
#       if (@$gp == 3) {
# 	# TODO: this somewhat duplicates InitFontes and should use a table
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[1]{'id'}, 0);
# 	push(@{$gp->[1]{'pairings'}},
# 	  0,              $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[1]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, 0);
# 	push(@{$gp->[1]{'etc'}{'board'}}, 0, $board, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board, 0, $board);
# 	$board += 1;
# 	}
#       elsif (@$gp == 4) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[3]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[1]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board, $board+1, $board+1);
# 	$board += 2;
# 	}
#       elsif (@$gp == 5) {
# 	# This table is not the one used in InitFontes
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, 0);
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[4]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  0,              $gp->[1]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board,   $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board,  0);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board,   0,      $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, 0,       $board+1, $board+1);
# 	$board += 2;
#         }
#       elsif (@$gp == 6) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[5]{'id'}, $gp->[3]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[5]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[4]{'id'}, $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[1]{'id'}, $gp->[5]{'id'});
# 	push(@{$gp->[5]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[2]{'id'}, $gp->[4]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board+2, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board+2, $board, $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, $board+2, $board+1, $board+2);
# 	push(@{$gp->[5]{'etc'}{'board'}}, $board, $board+2, $board+2);
# 	$board += 3;
# 	}
#       else { die "Assertion failed"; }
#       }
#       my $p1 = shift @pair_list;
#       my $p2 = shift @pair_list;
#       push(@{$p1->{'pairings'}}, $p2->{'id'});
#       push(@{$p2->{'pairings'}}, $p1->{'id'});
#   } # store pairings
# 
#   print "Done.\n";
#   $dp->Dirty(1);
#   $gTournament->UpdateDivisions();
#   }

# # TODO: this could be more efficient, but was written live at NSC 2005
# sub DoFactorGroup ($$$) {
#   my $psp = shift; # must not modify contents
#   # 0 indicates no repeats allowed, ..., 3 means up to 3 repeats = 4 pairings
#   my $repeats = shift;
#   # TODO: allow for possibility that we have to increase minbytes after 1st plr
#   my $minbyes = shift;
# 
#   print "DFG: " . (1+$#$psp) . ' ' . join(',', map { $_->{'id'} } @$psp) . "\n";
#   if (@$psp == 4 || @$psp == 6) {
#     if (CheckGroupRepeats $psp, $repeats) {
# #     print "DFG: returning $#$psp+1\n";
#       return ([@$psp]);
#       }
#     else {
# #     print "DFG: returning failure\n";
#       return ();
#       }
#     }
#   elsif (@$psp == 3) {
#     for my $p (@$psp) {
#       if ($p->{'byes'} != $minbyes) {
# #	print "DFG: $p->{'name'} already has $p->{'byes'} bye(s).\n";
# 	return ();
#         }
#       }
#     return ([@$psp]);
#     }
#   elsif (@$psp == 5) {
#     my $possible_bye_players = 0;
#     for my $p (@$psp) {
#       if ($p->{'byes'} == $minbyes) {
# 	$possible_bye_players++;
#         }
#       }
#     if ($possible_bye_players < 3) {
#       for my $p (@$psp) {
#         print "DFG5: $p->{'name'} already has $p->{'byes'} bye(s).\n";
#         }
#       return ([
# 	sort { $b->{'byes'} <=> $a->{'byes'} } @$psp
#         ]);
#       }
#     }
#   elsif (@$psp < 7) {
#     die "DoFactorGroup: bad group size: " . scalar(@$psp) . "\n";
#     }
#   my $s = int(@$psp/4);
#   my $p1 = $psp->[0];
#   my $j1 = 0;
#   # first try to pair within quartiles
#   for my $j2 ($s..$s+$s-1) {
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $j3 ($s+$s..$s+$s+$s-1) {
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $j4 ($s+$s+$s..$s+$s+$s+$s-1) {
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
#   # then try to pair anywhere within the group
#   for my $i2 (0..$#$psp) {
#     my $j2 = ($i2 + $s) % @$psp;
#     next if $j2 == $j1;
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $i3 (0..$#$psp) {
#       my $j3 = ($i3 + $s + $s) % @$psp;
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $i4 (0..$#$psp) {
# 	my $j4 = ($i4 + $s + $s + $s) % @$psp;
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
# # print "DFG: returning failure.\n";
#   return ();
#   }

sub DoSwiss ($$$$;$) { my($dp, $repeats, $p1, $p2, $pre) = @_;
  $pre = 0 unless defined $pre;
  my $datap = $dp->{'data'};
  my $theKeyRound = $dp->{maxp} - $pre;

  print("Calculating Swiss pairings for division $dp->{'name'},"
    ." $repeats repeats, players $p1-$p2.\n");
  printf "... using round %d results.\n", $theKeyRound + 1;

  my $tobepaired = $dp->GetUnpaired();
print "tobepaired size: $#$tobepaired.\n";
  @$tobepaired = grep($_->{'id'} >= $p1 && $_->{'id'} <= $p2, @$tobepaired);
  if ($#$tobepaired % 2 == 0) {
    Error "Can't Swiss pair an odd number of players.\n";
    return 0;
    }

print "tobepaired size: $#$tobepaired.\n";
  my ($maxpair, @offsets, $p, @pairing, $wins);

  # generate list of player offsets (see below)
  @offsets = (0); 
  for my $j (1..$#$tobepaired-1) { push(@offsets, $j, -$j); } 

  # sort by wins and spread, arrange tied players randomly
  my (@ranked) = TSH::Player::SortByStanding $theKeyRound, @$tobepaired;

# for my $p (@ranked) { print "$p->{'id'} $p->{'name'}\n"; }

  # calculate pairings for each win group
# print "# calculating pairing preference lists\n";
  { my $n; for (my $i=0; $i<=$#ranked; $i += $n+$n) {
    my $message;

    # this group starts at position $i, how far does it go?
    $wins = $ranked[$i]{'rwins'}[$theKeyRound];
print "$ranked[$i]{'name'} has $wins wins in round $theKeyRound+1.\n";
    if ((!$config'noboys) && $i == 0 && defined $wins) {
      # Dave Boys' idea: first group includes anyone with at least 2/3 leader's wins
#      my $quota = 2*$wins/3;
      # 1.500: changed to 2/3 of total wins
      my $quota = 2*($theKeyRound+1)/3;
      $message = "# at least $quota wins: ";
      $n = 1;
      while ($n%2==1 || (
        $i+$n <= $#ranked && 
        $ranked[$i+$n]{'rwins'}[$theKeyRound] >= $quota
        ))
        { $n++; }
      }
    else {
      $message = "# $wins wins: ";
      $n = 1; # start n at 1 and increment until 
      while ($i+$n<=$#ranked && ( # we hit the end or...
        ((!defined $wins) && (!defined $ranked[$i+$n]{'rwins'}[$theKeyRound])) # number of wins becomes defined or...
        || $ranked[$i+$n]{'rwins'}[$theKeyRound] == $wins || $n%2)) # number of wins changes and n is even
        { $n++; }
      }
    $wins = 0 unless defined $wins;
    print "$message$i to $i+$n\n";
    $n >>= 1;

    # $j indexes how far we are along in the "semigroup"
    for (my $j=0; $j<$n; $j++) {
      # list pairings preferences for player $j in upper semigroup
# TODO: should prefer players in current group first
      { # scoping only
      my $me = $ranked[$i+$j];
      my @pairing = ();
# print "pairings for $me->{'name'} ($me->{'id'})\n";
      for my $k (@offsets) {
        next if $k == -$n; # identity check
        my $opprank = $i+$j+$k+$n;
        next if $opprank < 0 || $opprank > $#ranked; # range check
        my $oppid = $ranked[$opprank]->{'id'};
# print "  $ranked[$opprank]->{'name'} ($ranked[$opprank]->{'id'})\n";
        next if $me->{'repeats'}[$oppid] > $repeats; # re-pairing check
# print "  (ok)\n";
        push(@pairing, $ranked[$opprank]);
        }
      die "$me->{'name'} can't be paired!\n" unless $#pairing >= 0;
      $me->{'pref'} = \@pairing;
# print "$me->{'name'} ($me->{'id'}): @pairing\n";
      } # scoping

      { # scoping only
      # list pairings preferences for player $j in lower semigroup
      my $me = $ranked[$i+$j+$n];
      my @pairing = ();
# print "pairings for $me->{'name'}\n";
      for my $k (@offsets) {
        next if $k == $n; # identity check
        my $opprank = $i+$j+$k;
        next if $opprank < 0 || $opprank > $#ranked; # range check
        my $oppid = $ranked[$opprank]->{'id'};
# print "  $ranked[$opprank]->{'name'}\n";
        next if $me->{'repeats'}[$oppid] > $repeats; # re-pairing check
# print "  (ok)\n";
        push(@pairing, $ranked[$opprank]);
        }
# print "$me->{'name'} ($me->{'id'}): @pairing\n";
      die "$me->{'name'} can't be paired!\n" unless $#pairing >= 0;
      $me->{'pref'} = \@pairing;
      } # scoping
      } # for $j
    } } # for $i

# special check for bye player
byes:
  for my $i (0..$#ranked) {
    my($p) = $ranked[$i];
    if ($p->{'name'} =~ /^bye /i) {
      for my $j (0..$#ranked) {
        if ($i == $j) {
          my($k, @newpref, $repeatp);
          $repeatp = $p->{'repeats'};
          for ($k=$#ranked; $k>=0; $k--) { # bye pairs against bottom of field
            next if $k == $i; # bye doesn't pair against bye
            my $oppid = $ranked[$k]{'id'};
            next if $repeatp->[$oppid] > 0; # bye never repeats
            push(@newpref, $ranked[$k]);
            }
          $ranked[$j]{'pref'} = \@newpref;
          }
        }
      unshift(@ranked, splice(@ranked, $i, 1));
      last byes;
      } # if player is bye
    } # for $i (bye player)

  if (ResolvePairings \@ranked) {
    $dp->Dirty(1);
    $dp->Update();
    }
  }

# Deprecated
sub Error ($) {
  my $message = shift;
  TSH::Utility::Error($message);
  }


sub lint () {
  $config'table_format = '';
  $config'gibson = undef;
  $config'noboys = 0;
  %config::gibson_equivalent = ();
  %config::autopair = ();
# $config'reserved{''} = '';
# $config'tables{''} = '';
  lint;
  }

sub LockFailed ($) {
  my $reason = shift;
  print <<"EOF";
System call failed: $reason

You should not run more than one copy of tsh using the same 
configuration file at the same time.  tsh uses a "lock file" called
tsh.lock to keep track of when it is running.  This copy of tsh
was unable to get access to the lock file.  The most likely reason
for this is that tsh is already in use.
EOF
  exit 1;
  }

sub LockOff () {
  flock($global'lockfh, LOCK_UN)
    or die "Can't unlock tsh.lock - something is seriously wrong.\n";
  close($global'lockfh)
    or die "Can't close tsh.lock - something is seriously wrong.\n";
  }

sub LockOn () {
  my $error;

  $global'lockfh = new FileHandle "$config'root_directory/tsh.lock",
    O_CREAT | O_RDWR
    or die "Can't open tsh.lock - check to make sure tsh isn't already running.\n";
  flock($global'lockfh, LOCK_EX | LOCK_NB) 
    or LockFailed "flock: $!";
  seek($global'lockfh, 0, 0) 
    or die "Can't rewind tsh.lock - something is seriously wrong.\n";
  truncate($global'lockfh, 0) 
    or die "Can't truncate tsh.lock - something is seriously wrong.\n";
  print $global'lockfh "$$\n"
    or die "Can't update tsh.lock - something is seriously wrong.\n";
  } 

sub Main () {
  srand;

# {
#   my $dawg = 'gloria:john:19990925-toronto:twl98.dawg';
#   $dawg = "$ENV{'HOME'}/scrabble/ospd/words.dawg" 
#     unless -f $dawg;
#   &dawg'open(*WORDS, $dawg) || warn "Can't open WORDS file.\n";;
# }
  &dawg'open(*WORDS, 'lib/words/twl98.dwg');

  ReopenConsole if $^O eq 'MacOS';
  my $config_fn = ChooseConfig;
  LockOn;
  $gTournament = new TSH::Tournament;
  $gTournament->TellUser('iwelcome', $gkVersion);
  ReadConfig $config_fn;
  mkdir $config'backup_directory, 0700 unless -d $config'backup_directory;
  ReadDivisions;
  $global'parser = new TSH::ParseArgs;
  Prompt;
  while (<>) {
    next unless /\S/;
    my(@argv) = split;
    # TODO: this should be dispatched through the class mechanism 
    my $sub = $global'commands{lc $argv[0]};
    if (defined $sub) { last if &$sub(\@argv, $_); }
    elsif (!$gTournament->RunCommand(@argv)) 
      { print "Enter 'help' for help.\n"; }
    last if $gTournament->QuittingTime();
    }
  continue {
    Prompt;
    }
  LockOff;
  }

BEGIN {
  %global'ParseArgsDispatch = (
    'based-on-round' => \&ParseBasedOnRoundNumber,
    'division' => \&ParseDivisionName,
    'factor' => \&ParseFactor,
    'nothing' => \&ParseNothing,
    'nrounds' => \&ParseNRounds,
    'player-number-or-0' => \&ParsePlayerNumberOrZero,
    'player-number' => \&ParsePlayerNumber,
    'repeats' => \&ParseRepeats,
    'round' => \&ParseRoundNumber,
    'round-or-0' => \&ParseBasedOnRoundNumber,
    'score' => \&ParseScore,
    );
  }

sub MakePlayerMap () {
  return if %global'player_name_to_id;
  for my $dp (values %gDivision) {
    my $dname = $dp->{'name'};
    my $datap = $dp->{'data'};
    for my $id (1..$#$datap) {
      $global'player_name_to_id{$datap->[$id]{'name'}} = "$dname$id";
      }
    }
  }

sub ParseArgs ($$) {
  my $argvp = shift;
  my $typesp = shift;

  my $arg0 = shift @$argvp;
  my $usage = "Usage: $arg0 ";
  if ($gNDivisions == 1) {
    $usage .= join(' ', grep { $_ ne 'division' } @$typesp);
    }
  else {
    $usage .= join(' ', @$typesp);
    }
  $usage .= "\n";

  my @values;
  $global'parse_error = '';
  for my $type (@$typesp, 'nothing') {
    my $sub = $global'ParseArgsDispatch{$type};
    die "Unknown argument type ($type)." unless $sub;
    my $value = &$sub($argvp, $usage);
    if (defined $value) { push(@values, $value); }
    else { $global'parse_error ||= 'error'; return (); }
    }
  return @values;
  }

sub ParseBasedOnRoundNumber ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'round number', 0,
    ($config'max_rounds || 10000);
  }

sub ParseDivisionName ($$) {
  my $argvp = shift;
  my $usage = shift;

  if ($gNDivisions == 1) { 
    if (defined $gDivision{lc $argvp->[0]}) {
      shift @$argvp;
      }
    return (%gDivision)[1]; 
    }
  elsif ($#$argvp < 0) { 
    Error "You must specify a division name with this command.\n";
    Error $usage; 
    return undef; 
    }
  else {
    my $dn = lc shift @$argvp;
    my $dp = $gDivision{$dn};
    if (!defined $dp) { 
      Error "No such division: \U$dn\E.\n"; 
      Error $usage;
      return undef; 
      }
    else { 
      return $dp;
      }
    }
  }
     
sub ParseFactor ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'factor', 1, 100;
  }

sub ParseInteger ($$$$$) {
  my $argvp = shift;
  my $usage = shift;
  my $typename = shift;
  my $min = shift;
  my $max = shift;

  if ($#$argvp < 0) { 
    Error "Please specify a $typename.\n";
    Error $usage; 
    return undef; 
    }
  my $n = shift @$argvp;
  if ($n !~ /^-?\d+$/) {
    Error "This doesn't look like a $typename to me: $n.\n";
    Error $usage;
    return undef;
    }
  if ($n < $min) {
    Error "$n is too small to be a $typename.\n";
    Error $usage;
    return undef;
    }
  if ($n > $max) {
    Error "$n is too big to be a $typename.\n";
    Error $usage;
    return undef;
    }
  return $n;
  }

sub ParseNothing ($$) {
  my $argvp = shift;
  my $usage = shift;

  if (@$argvp) {
    Error "I don't understand this bit at the end: @$argvp\n";
    Error $usage;
    return undef;
    }
  return 1;
  }

sub ParseNRounds ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'number of rounds', 1,
     ($config'max_rounds || 10000);
  }

sub ParsePlayerNumber ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'player number', 1, 100000;
  }

sub ParsePlayerNumberOrZero ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'player number', 0, 100000;
  }

sub ParseRepeats ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'number of repeat pairings per player',
    0, 
    ($config'max_rounds || 1000);
  }

sub ParseRoundNumber ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'round number', 1,
    ($config'max_rounds || 10000);
  }

sub ParseScore ($$) { my ($argvp, $usage) = @_;
  return ParseInteger $argvp, $usage, 'score', -500, 1500;
  }

sub Prompt () { 
  print ((TSH::Utility::Colour('yellow on blue', 'tsh>')), ' ');
  }

sub ReadConfig ($) {
  my $fn = shift;

  $config'table_format = '%3s';
  open(CONFIG, "<$config'root_directory/$fn") || die "Can't open $fn: $!\n";
  local($_);
  $gTournament->TellUser('iloadcfg', "$config::root_directory/$fn");
  $gNDivisions = 0;
  while (<CONFIG>) { s/#.*//; s/^\s*//; s/\s*$//; next unless /\S/;
    if (/^division?\s+(\S+)\s+(.*)/i) {
      # the new way
      my $dname = $1;
      my $dfile = $2;
      my $dp = new TSH::Division;
      $dp->Name($dname);
      $dp->File($dfile);
      $gTournament->AddDivision($dp);
      # the old way
#     $gDivision{lc $1} = 
#       bless {'file' => $2, 'name' => (uc $1)}, 'TSH::Division';
      $gDivision{lc $dname} = $dp;
      $gNDivisions++;
      }
    elsif (s/^perl\s+//i) { 
      eval $_;
      print "eval: $@\n" if length($@);
      }
    elsif (s/^config\s+//i) { 
      eval q($config') . $_;
      print "eval: $@\n" if length($@);
      }
    elsif (s/^autopair\s+//i) { 
      if (/^(\w+) (\d+) (\d+)\s+(\w+)\s+(.*)/) {
	my ($div, $sr, $round, $command, $args) = ($1, $2, $3, $4, $5);
	my (@args) = split(/\s+/, $args);
	if ($command =~ /^(?:if|pair1324|p1324|koth|ns|newswiss|rr|roundrobin|cp)$/i) {
	  $config::autopair{uc $div}[$round] = [$sr, $command, @args];
	  }
	else {
	  $gTournament->TellUser('ebadapc', $command);
	  exit(1);
	  }
        }
      else {
	chomp;
        $gTournament->TellUser('ebadap', $_, $fn);
	exit(1);
        }
      }
    else {
      $gTournament->TellUser('ebadcfg', $_);
      exit(1);
      }
    }
# print "Configuration file loaded.\n";
  if ($config::external_path) {
#   print "Loading external(s):";
    for $global'path (@$config::external_path) {
      my $config = "$global'path/tshxcfg.txt";
      if (-r "$config") {
	my $rv = do $config;
	if ($@) { print "Can't load externals [\@$@]"; }
	if ($!) { print "Can't load externals [!$!]"; }
	unless ($rv) { print "Can't load externals [X]"; }
        }
      }
#   print "\n";
    }
  for my $dir ($config'html_directory, $config'backup_directory) {
    if ($dir !~ /^(?:\/|[a-zA-Z]:[\/\\])/) 
      { $dir = "$config'root_directory/$dir"; }
    mkdir $dir unless -d $dir;
    }
  $config'split1 = 1000000 unless $config'split1;
  for my $div (keys %config::gibson_groups) {
    my $divp = $config::gibson_groups{$div};
    for my $gibson_group (@$divp) {
      my $first = $gibson_group->[0];
      for my $i (1..$#$gibson_group) {
	$config::gibson_equivalent{$div}[$gibson_group->[$i]] = $first;
#	print "$gibson_group->[$i] equiv $first in $div.\n";
	}
      }
    }
  }

sub ReadDivision (\%) {
  my $dp = shift;
  my $fn = $dp->{'file'};
  $gTournament->TellUser('iloaddiv', $dp->{'name'});
  open(DIV, "<$config'root_directory/$fn") || die "Can't open $fn: $!\n";
  local($_);
  my (@data) = (undef);
  my $id = 1;
  my $name_length = 16;
  while (<DIV>) { s/#.*//; s/^\s*//; s/\s*$//; next unless /\S/;
    s/$/;/ unless /;/;
# TODO: use lib/perl/TFile.pm to do parsing here and elsewhere
# TODO: see if a cached binary file format would speed large file loads
# TODO: consider delaying parsing of some subfields until it's needed, or we're idle
    my($player, $rating, $pairings, $scores, $etc) 
      = /^([^;]+[^;\s\d])\s+(\d+)\s*([\d\s]*);\s*([-\d\s]*)((?:;[^;]*)*)$/;
    die "Can't parse: $_\n" unless defined $scores;
    $name_length = length($player) if $name_length < length($player);
    my(@pairings) = split(/\s+/, $pairings);
    my(@scores) = split(/\s+/, $scores);
    my $pp = { 
      'division' => $dp,
      'id'       => $id,
      'name'     => $player,
      'rating'   => $rating,
#     'rnd'=>rand,
      'rnd'      => ((length($player) * (100+$id) * ord($player)) % 641),
      'pairings' => [split(/\s+/, $pairings)],
      'scores'   => [split(/\s+/, $scores)],
      };
    bless $pp, 'TSH::Player'; # TODO: move this to new() if it doesn't slow things down too much
    for my $extra (split(/;\s*/, $etc)) {
      next unless $extra =~ /\S/;
      my ($tag, @words) = split(/\s+/, $extra);
      if (defined $pp->{$tag}) {
	warn "Overwriting $tag field for $player.\n";
        }
      $pp->{'etc'}{$tag} = \@words;
      }
    push(@data, $pp);
    $id++;
    }
  close(DIV);
  if ($name_length > $config'max_name_length) {
    $config'max_name_length = $name_length;
    $config'name_format = "%-${name_length}s";
    }

  Error "Warning: odd number of players in Division $dp->{'name'}.\n"
    if $#data % 2 == 1;
  $dp->{'data'} = \@data;
  $dp->Synch();
  }

sub ReadDivisions () {
  for my $div (sort keys %gDivision) {
    my $dp = $gDivision{$div};
    ReadDivision %$dp;
    } 
# print "All divisions loaded.\n";
  }

sub ReopenConsole () {
  close(STDOUT);
  close(STDERR);
  close(STDIN);
  open(STDOUT, "+>dev:console:tsh console") || die;
  open(STDERR, "+>dev:console:tsh console") || die;
  open(STDIN, "<dev:console:tsh console") || die;
  $| = 1;
  }

=item $boolean = ResolvePairings $unpairedp[, $just_checking]

Given a division and a list of unpaired players who have their
'pref' field set to a list of opponent preferences, find a reasonable
pairing of all the players.  Return success.  If C<$just_checking>
then run quietly and do not execute pairings, just return status.
In either case, set the 'opp' field of each player paired to the
opponent's ID.

Deprecated in favour of TSH::Player::ResolvePairings.

=cut

sub ResolvePairings ($;$) {
  my $unpairedp = shift;
  my $just_checking = shift;

# { my $p; for $p (@$unpairedp) { print "# $p->{'name'}: @{$p->{'pref'}}\n"; } }
# print "# finding optimal pairing\n";
  # pruning dead branches saves us two orders of magnitude or so
  my %dead;

  # another (slight) speed optimization
  my (@sorted) = @$unpairedp[sort {
    # prefer players with fewer choices
    @{$unpairedp->[$a]{'pref'}} <=> @{$unpairedp->[$b]{'pref'}} ||
    # ties broken according to input ordering
    $a <=> $b;
    } 0..$#$unpairedp
    ];
# my (@sorted) = @$unpairedp;

  { # block for scope isolation only
    my(@choice, $opp, $oppid);

    # mark all players as initially unpaired
    # 'opp' points to the provisional opponent
    for my $p (@sorted) { 
      $p->{'opp'} = -1; 
      # check quickly to see if pairings are impossible
      unless (@{$p->{'pref'}}) {
#	Error "No candidate opponents for " . $p->{'name'};
	return 0;
        }
      }

    # find best opp for each player, favoring top of field
    for (my $i=0; $i<=$#sorted; ) {
      my $p = $sorted[$i];
      if ($p->{'opp'} >= 0)
        { $i++; next; } # player has already been paired - skip
      my $key = join('', grep { $_->{'opp'} < 0 } 
	@sorted[$i..$#sorted]);
      if ($dead{$key}) {
#	print "s\010";
#	print "Skipping known dead: $key.\n";
	# this code is duplicated below and should be merged 
	# when fully debugged
        for ($choice[$i]=undef; $i>=0 && !defined $choice[$i]; $i--) { }
# print "$i.\n";
        if ($i < 0) {
          Error "Walked entire tree, couldn't find acceptable pairing.\n"
	    unless $just_checking;
          return 0;
          }

        # find last paired player's opponent, which now has to be unpaired
        my $opp = $sorted[$i]{'pref'}[$choice[$i]];
        # unpair opponent from that player
        $opp->{'opp'} = -1;
        # unpair that player from the opponent
        $sorted[$i]{'opp'} = -1;
        next;
        }

      # go to head of preference list if visiting player for first time
      $choice[$i] = -1 unless defined $choice[$i];

      # try the next preferred opp for this player.
      $opp = $p->{'pref'}[++$choice[$i]];

      if (!defined $opp) {
# print '.' x $i, "$p->{'name'} can't be paired, climbing back up from i=$i to i=";
#	print "Marking as dead: $key\n";
#	print "m\010";
	$dead{$key}++;
        for ($choice[$i]=undef; $i>=0 && !defined $choice[$i]; $i--) { }
# print "$i.\n";
        if ($i < 0) {
          Error "Walked entire tree, couldn't find acceptable pairing.\n"
	    unless $just_checking;
          return 0;
          }

        # find last paired player's opponent, which now has to be unpaired
        my $opp = $sorted[$i]{'pref'}[$choice[$i]];
        # unpair opponent from that player
        $opp->{'opp'} = -1;
        # unpair that player from the opponent
        $sorted[$i]{'opp'} = -1;
        next;
        } # if (!defined $opp) - we've run out of opps, back up

# print '.' x $i, "$p->{'name'} has pairing vector @{$p->{'pref'}}.\n";
# print ' ' x $i, " trying to pair $p->{'name'}, choice $choice[$i] is ",
# defined $opp ? "$opp->{'name'} ($opp->{'id'})" : 'undef', "\n";

      if ($opp->{'opp'} >= 0) {
# print ' ' x $i, " but $opp->{'name'} has already been paired.\n";
        next;
        }

      # looks good so far, let's try to keep going
      $p->{'opp'} = $opp->{'id'};
      $opp->{'opp'} = $p->{'id'};
      $i++;
      } # for $i
    }
  # copy provisional opponents to pairings
  unless ($just_checking) {
#   my $board = 1;
    my $r0 = $#{$sorted[0]{'pairings'}};
    for my $i (0..$#sorted) {
      my $p = $sorted[$i];
      push(@{$p->{'pairings'}}, $p->{'opp'});
# The following code (which assigns players to boards) was removed
# because ResolvePairings is not always called for the complete set 
# of players, and board numbers were being doubled.
#     my $oid = $p->{'opp'};
#     next unless $oid;
#     my $opp = $p->{'division'}{'data'}[$oid];
#     die "assertion failed" unless $opp;
#     my $ptp = $p->{'etc'}{'board'};
#     if (!defined $ptp) { $p->{'etc'}{'board'} = $ptp = []; }
#     push(@$ptp, (0) x ($r0-$#$ptp)) if $r0 > $#$ptp;
#     my $otp = $opp->{'etc'}{'board'};
#     if (!defined $otp) { $opp->{'etc'}{'board'} = $otp = []; }
#     push(@$otp, (0) x ($r0-$#$otp)) if $r0 > $#$otp;
#     next unless (
#	$p->{'wins'} <=> $opp->{'wins'} ||
#	$p->{'spread'} <=> $opp->{'spread'} ||
#	$opp->{'id'} <=> $p->{'id'}) > 0;
#     push(@$ptp, $board);
#     push(@$otp, $board++);
      }
    }
  1;
  } # sub ResolvePairings

=back

=cut

=head1 TO-DO

Here is the list of planned improvements to C<tsh>, in roughly
descending order of priority.

=over 4

=item *
 don't count inactive players in missing

=item *
 finish adding active/inactive support (P1324, KOTH, RoundRobin, DoClark)

=item *
 mark players active/inactive

=item *
 display firsts/seconds in ratings command

=item *
 finish implementing Chew pairings

=item *
 pair using firsts and seconds

=item *
 team match support (separate module, commands, config file)

=item *
 stability of assignment of tables

=item *
 copy tsh.css from lib if not found where needed

=item *
 Fontes Gibsonization possibility warnings 

=item *
 allow "surname,given" wherever parser expects pn

=item *
 BUG: SC doesn't show final bye

=item *
 do not allow pairing commands to exceed max_rounds

=item *
 webupdater should be configurable and documented

=item *
 more flexible command parser with optional arguments

=item *
 default values for command parameters?

=item *
 prompt for and load or create a config.tsh file if none specified

=item *
 ratings submission command (external)

=item *
 return values from externals

=item *
 some configuration variables to externals

=item *
 a command to unpair selective pairings (maybe a 1-arg form of pair)

=item *
 more contextual help, automatically chosen

=item *
 command-line toggling of debug flags

=item *
 division data complete message and trigger

=item *
 persistent tables through etc/tables

=item *
 bios.txt interface, photos on pairings

=item *
 virtual scorecards on web

=item *
 printing support

=item *
 finish documenting diagnostic messages

=item *
 proofread documentation for typographic style

=item *
 load large divisions in separate threads

=item *
 a report listing last lines of scorecards for all players, 
 so that players can check their results

=item *
 In ResolvePairings, try using a hash to keep track of known unpairable
 sets of players.  

=item *
 ResolvePairings is currently the critical sub, and should be optimized,
 or rewritten in C.

=back

=cut

END { 
  sleep 10 if $^O eq 'MSWin32'; # to prevent error messages from disappearing
  }

## main code
Main;

