#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ChewPair;

use strict;
use warnings;

use TSH::Utility qw(Debug DebugOn DebugOff DebugDumpPairings);
use TSH::Player;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ChewPair - implement the C<tsh> ChewPair command

=head1 SYNOPSIS

  my $command = new TSH::Command::ChewPair;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ChewPair is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub CalculateBPFs ($$$);
sub CalculateBestPossibleFinish ($$$$);
sub CanBePaired ($$);
sub CountContenders ($$$$);
sub CountGibsons ($$);
sub GetPrizeBand($$$);
sub GibsonEquivalent($$);
sub initialise ($$$$);
sub new ($);
sub PairEvenGibsons ($$);
sub PairLeaders ($$$);
sub PairNonLeaders ($$);
sub PairOneGibson ($$$$);
sub Run ($$@);

=item CalculateBPFs($psp, $sr0, $rounds_left)

Calculate best possible finishes for everyone in @$psp.

=cut

sub CalculateBPFs($$$) {
  my $psp = shift;
  my $sr0 = shift;
  my $rounds_left = shift;

  my $toprank = $psp->[0]->RoundRank($sr0);
  for my $i (0..$#$psp) {
    my $r = CalculateBestPossibleFinish $psp, $i, $sr0, $rounds_left;
    $psp->[$i]->MaxRank($r+$toprank-1);
    }
  }

=item CalculateBestPossibleFinish($psp, $index, $sr0, $rounds_left);

Calculate what the highest possible finishing rank is for the
${index}th player in $psp based on results up to round $sr0
with $rounds_left remaining.
The rank is relative to the players in @$psp.

=cut

sub CalculateBestPossibleFinish ($$$$) {
  my $psp = shift;
  my $index = shift;
  my $sr0 = shift;
  my $nrounds = shift;

# DebugOn('CBPF');
  my $round_spread = $nrounds == 1 ? 250 : $nrounds == 2 ? 200 : 150;
  my (@pdata) = map {
    [ $_->RoundWins($sr0), $_->RoundSpread($sr0), $_, 0, 0]
    } @$psp[0..$index];

# for my $i (0..$#pdata) { print "$pdata[$i][0]/$pdata[$i][1] "; } print "\n";
  my ($mywins, $myspread, $p) = @{pop @pdata};
  unless (@pdata) {
#   Debug 'CP', "%3d=>%3d%5.1f%+5d %s #%d", 1, 1, $mywins+$nrounds, $myspread+$nrounds*$round_spread, $p->Name(), $p->ID();
    return 1;
    }
# Debug 'CBPF', "I am %g/%+d.", $mywins, $myspread;
  my $half_of_leaders = int((@pdata-1)/2);
  my $wins_left = $half_of_leaders * $nrounds;
# Debug 'CBPF', "Those above me get %d * %d = %d wins.", $half_of_leaders, $nrounds, $wins_left;
  my $max_spread = $nrounds * $round_spread;
  $mywins += $nrounds;
  $myspread += $max_spread;
# Debug 'CBPF', "I will finish %g/%+d.", $mywins, $myspread;
  # first try to give wins without affecting our rank
  for my $winsprp (@pdata) { my $games_behind = $mywins - $winsprp->[0];
    my $spread_behind = $myspread - $winsprp->[1];
    my $wins_to_give = $spread_behind > 0 ? $games_behind : $games_behind - 1;
    if ($wins_to_give > 0) {
      $wins_to_give = $nrounds if $wins_to_give > $nrounds;
      $winsprp->[0] += $wins_to_give;
      $winsprp->[3] += $wins_to_give;
      $wins_left -= $wins_to_give; # could go negative, but doesn't matter
      last if $wins_left <= 0;
      }
    }
# print "1."; for my $i (0..$#pdata) { print "$pdata[$i][0]/$pdata[$i][1] "; } print "\n";
  # second try to give wins that lower us within our win group
  if ($wins_left > 0) {
    # give wins to players with worst spread
    for my $winsprp (sort { $a->[1] <=> $b->[1] } @pdata) {
      my $wins_to_give = $mywins - $winsprp->[0];
      if ($wins_to_give > 0) {
	my $wins_left_for_this_one = $nrounds - $winsprp->[3];
	$wins_to_give = $wins_left_for_this_one 
	  if $wins_to_give > $wins_left_for_this_one;
	$winsprp->[0] += $wins_to_give;
	$winsprp->[3] += $wins_to_give;
	$wins_left -= $wins_to_give;
	last if $wins_left <= 0;
        }
      }
#   print "2."; for my $i (0..$#pdata) { print "$pdata$i][0]/$pdata$i][1] "; } print "\n";
    }
  # third give remaining wins so as to let as few as possible overtake us
  if ($wins_left > 0) {
    # give wins to players who have received the least wins so far
    for my $winsprp (sort { $a->[3] <=> $b->[3] } @pdata) {
      my $wins_to_give = $nrounds - $winsprp->[3];
      $winsprp->[0] += $wins_to_give;
      $winsprp->[3] = $nrounds;
      $wins_left -= $wins_to_give;
      last if $wins_left <= 0;
      }
#   print "3."; for my $i (0..$#pdata) { print "$pdata[$i][0]/$pdata[$i][1] "; } print "\n";
    }
  # if we still have wins left over, we're in trouble
  if ($wins_left > 0) {
    print "Unexpected error: can't distribute wins.\n";
    return 0;
    }
  
  # now try to move up in rank by adjusting spreads within our win group
  {
    my @group_sempai;
    my $spread_left = $max_spread; # the positive spread we gave me
    if (@pdata % 2 == 0) # pairings involve a big winner below us
      { $spread_left += $max_spread; }
    for my $winsprp (@pdata) {
      if ($winsprp->[0] != $mywins) { # not in our group, can absorb spread
	# assume maximum-point wins and one-point losses
	my $spread = $round_spread * $winsprp->[3] - ($nrounds - $winsprp->[3]);
	if ($spread > 0) {
#	  print "other  $winsprp->[0]/$winsprp->[1] gets +$spread.\n";
	  $winsprp->[1] += $spread;
	  $winsprp->[4] = $spread;
	  $spread_left += $spread;
	  }
        }
      elsif ($winsprp->[1] > $myspread) { # sempai, try to take down in spread
	# assume one-point wins and maximum-point losses
	my $spread = $winsprp->[3] - $round_spread * ($nrounds - $winsprp->[3]);
	if ($spread < 0) {
	  my $spread_needed = $myspread - $winsprp->[1];
	  $spread = $spread_needed if $spread_needed > $spread;
	  if ($spread <= $spread_needed) {
#	    print "sempai $winsprp->[0]/$winsprp->[1] gets $spread.\n";
	    $winsprp->[1] += $spread;
	    $winsprp->[4] = $spread;
	    $spread_left += $spread;
	    }
	  else {
#	    print "Can't bring $winsprp->[0]/$winsprp->[1] down far enough.\n";
	    }
          }
        }
      elsif ($winsprp->[1] < $myspread) { # junior, try giving them spread
	# assume maximum-point wins and one-point losses
	my $spread = $round_spread * $winsprp->[3] - ($nrounds - $winsprp->[3]);
	if ($spread > 0) {
	  my $spread_allowed = $myspread - $winsprp->[1];
	  $spread = $spread_allowed if $spread_allowed < $spread;
#	  print "junior $winsprp->[0]/$winsprp->[1] gets $spread.\n";
	  $winsprp->[1] += $spread;
	  $winsprp->[4] = $spread;
	  $spread_left += $spread;
          }
        }
      }
    # we took away too much spread from sempai
    if ($spread_left < 0) {
      for my $winsprp 
        # look primarily for sempai who were given a lot of negative spread
	# but remember number of wins affects maximum spread awardable
        (sort { $a->[4]-($round_spread-1)*$a->[3] 
	    <=> $b->[4]-($round_spread-1)*$b->[3] } 
	grep { $_->[4] < 0 } @pdata) {
	# assume maximum-point wins and one-point losses
	my $spread = $round_spread * $winsprp->[3] - ($nrounds-$winsprp->[3]);
	my $change = $spread - $winsprp->[4];
	$winsprp->[1] += $change;
	$winsprp->[4] = $spread;
#	print "sempai $winsprp->[0]/$winsprp->[1] now gets $spread.\n";
	$spread_left += $change;
	last if $spread_left >= 0;
	}
      }
    # if we still have spread left over, we're in trouble
    if ($spread_left < 0) {
      print "Unexpected error: can't distribute spread.\n";
      return 0;
      }
#   print "4."; for my $i (0..$#pdata) { print "$pdata[$i][0]/$pdata[$i][1] "; } print "\n";
  }

  my $rank = 1;
  for my $winsprp (sort { $b->[0] <=> $a->[0] || $b->[1] <=> $a->[1] } @pdata) {
    if (($mywins <=> $winsprp->[0] 
      || $myspread <=> $winsprp->[1]) >= 0) {
#     Debug 'CBPF', '%s can finish ranked %d, %g %+d ahead of %s, %g %+d.', $p->TaggedName(), $rank, $mywins, $myspread, $winsprp->[2]->TaggedName(), $winsprp->[0], $winsprp->[1];
      last;
      }
    else { $rank++; }
    }
# Debug 'CP', "%3d=>%3d%5.1f%+5d %s #%d", $index+1, $rank, $mywins, $myspread, $p->Name(), $p->ID();
  return $rank;
  }

=item $boolean = CanBePaired $repeats, \@players

Returns true iff @players can be paired without exceeding C<$repeats>.

=cut

sub CanBePaired ($$) { 
  my $repeats = shift;
  my $psp = shift;

  # using @shuffled improves runtime by two orders of magnitude in bad
  # cases, as looking first at the top and bottom players typically
  # works on the toughest cases first.
  my (@ps) = (@$psp);
  my @shuffled;
# while (@ps) {
#   push(@shuffled, shift @ps);
#   push(@shuffled, pop @ps) if @ps;
#   }
  # the above optimization is currently disabled in favour of sorting
  # the players by number of candidate opponents (and hence difficulty
  # of pairing) in ResolvePairings().
  @shuffled = @ps;
  for my $i (0..$#shuffled) {
# # if we use the commented out version, ResolvePairings always
# # takes a loooong time to run when low-numbered players are in
# # high demand.  Randomly rolling the preferences reduces mean
# # but not worst-case runtime.
#   $shuffled[$i]{'pref'} 
#     = [ grep { $shuffled[$i]->Repeats($_->ID()) <= $repeats }
#     (@shuffled[0..$i-1,$i+1..$#shuffled]) ];
    my @prefs 
      = grep { $shuffled[$i]->Repeats($_->ID()) <= $repeats }
      (@shuffled[0..$i-1,$i+1..$#shuffled]);
    # randomly roll preferences
#   unshift(@prefs, splice(@prefs, rand(@prefs)));
    $shuffled[$i]{'pref'} = \@prefs;
    }
  return main::ResolvePairings \@shuffled, 1;
  }

=item $n = CountContenders($maxrank, $psp, $sr0, $rounds_left)

Count the number of players in contention for a prize band
based on results up to $sr0 with $rounds_left remaining.

=cut

sub CountContenders ($$$$) {
  my $maxrank = shift;
  my $psp = shift;
  my $sr0 = shift;
  my $rounds_left = shift;

  Debug 'CP', 'Counting contenders with maxr=%d np=%d sr0=%d rl=%d\n', $maxrank, scalar(@$psp), $sr0, $rounds_left;
  CalculateBPFs $psp, $sr0, $rounds_left;
  
  my $max_flight_size = 0;
  for my $i (0..$#$psp) {
    my $r = $psp->[$i]->MaxRank();
#   Debug 'CP', '%s maxrank is %d', $psp->[$i]->TaggedName(), $r;
    if ($r <= $maxrank) {
      $max_flight_size = $i;
#     Debug 'CP', 'CC: %d <= %d, mfs=%d', $r, $maxrank, $i;
      }
    }
  TSH::Utility::Error "Assertion failed: only one contender" if $max_flight_size == 0;
  $max_flight_size++; # now one-based
  return $max_flight_size;
  }

=item $n = $parserp->CountGibsons($sr0, $psp)

Count the number of Gibsons in @$psp based on round $sr0 standings.
You must call Division::ComputeRanks($sr0) before calling this sub.

=cut

sub CountGibsons ($$) {
  my $sr0 = shift;
  my $psp = shift;
  my $sr = $sr0 + 1;
  my $rounds_left = $config::max_rounds - $sr;
  my $spread_left = 
  my $round_spread = $rounds_left == 1 ? 500 
    : $rounds_left == 2 ? 400 : 300;
  my $spread_allowed = $rounds_left * $round_spread;

  my $dname = $psp->[0]->Division()->Name();
  my $gibson_equivalent = $config::gibson_equivalent{$dname};
  my $ngibsons = 1;
  if ($gibson_equivalent) {
    # $gibson_equivalent tells us which ranks are equivalent,
    # usually either #1 and #2 or none
    my $rank = $psp->[0]->RoundRank($sr0);
    my $here = $gibson_equivalent->[$rank] || $rank;
    if ($here) {
      while (1) {
	my $next = $gibson_equivalent->[++$rank];
	last unless $next && $next == $here;
	$ngibsons++;
        }
      }
    }
  Debug 'CP', 'Gibson-equivalent ranks: %d starting at %d', $ngibsons, $psp->[0]->RoundRank($sr0);
  while ($ngibsons > 0) {
    my $win_diff = $psp->[$ngibsons-1]->RoundWins($sr0)
      - $psp->[$ngibsons]->RoundWins($sr0);
    if ($win_diff > $rounds_left) { last; }
    elsif ($win_diff < $rounds_left) { next; }
    my $spread_diff = $psp->[$ngibsons-1]->RoundSpread($sr0)
      - $psp->[$ngibsons]->RoundSpread($sr0);
    if ($spread_diff > $spread_allowed) { last; }
    }
  continue { $ngibsons--; }
  Debug 'CP', 'Gibsons found: %d', $ngibsons;
  return $ngibsons;
  }

=item ($minrank,$maxrank) = $parserp->GetPrizeBand($prize_bands, $psp, $sr0)

Find the prize band that contains the given rank.

=cut

sub GetPrizeBand ($$$) {
  my $prize_bands = shift;
  my $psp = shift;
  my $sr0 = shift;
  die "Assertion failed" unless @$psp;
  my $rank = $psp->[0]->RoundRank($sr0);
  my (@splits) = (0, @$prize_bands, $psp->[-1]->RoundRank($sr0));
# Debug 'CP', "GetPB: $rank, @splits";
  for my $i (1..$#splits) {
    my $minrank = $splits[$i-1]+1;
    my $maxrank = $splits[$i];
    if ($rank >= $minrank && $rank <= $maxrank) {
      return ($minrank, $maxrank);
      }
    }
  die "Assertion failed with rank=$rank, splits=@splits";
  }

=item $parserp->GibsonEquivalent($dname, $rank)

Return the highest rank equivalent for Gibson purposes to $rank.

=cut

sub GibsonEquivalent($$) {
  my $dname = shift;
  my $rank = shift;

  return ($config::gibson_equivalent{$dname}[$rank] || $rank);
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to compute Chew pairings for a division.
EOF
  $this->{'names'} = [qw(cp chewpair)];
  $this->{'argtypes'} = [qw(BasedOnRound Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

# DebugOn('CP');
  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

=item PairEvenGibsons($psp, $ngibsons)

Pair an even number of gibsons with each other,
with decreasing priority given to

  - minimizing rematches
  - avoiding consecutive rematches (where possible)
  - pairing each player with the one ranked closest possible

=cut

sub PairEvenGibsons ($$) {
  my $psp = shift;
  my $ngibsons = shift;
  return unless $ngibsons;
  Debug 'CP', 'Pairing %d gibsons.', scalar($ngibsons);
  if (
    TSH::Player::PairGRT([@$psp[0..$ngibsons-1]],
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $distance = abs($_[1]-$_[2]);
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d svr=%d dist=%d', $pid, $oid, $repeats, $sameopp, $pairsvr, $distance;
	pack('NCCNN',
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $pairsvr, # pair those due to start vs those due to reply
	  $distance, # prefer closer-ranked opponents
	  $_[2], # to recover player ID
	  )
        },
      # allowed opponent filter
      sub {
	# allow any
	1,
        },
      [],
      )
    ) # if
    {
    DebugDumpPairings 'CP', $psp->[0]->CountOpponents()-1, 
      [splice(@$psp, 0, $ngibsons)];
    }
  else
    {
    TSH::Utility::Error "Assertion failed: can't resolve even gibson pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairLeaders($$$)

Pair an even number of flight leaders with decreasing priority given to
with decreasing priority given to

  - not exceeding minrep (ever)
  - minimizing repeats
  - avoiding consecutive rematches (where possible)
# - matching players due to go first with those due to go second (w.p.)
  - pairing the highest unpaired with the lowest who can catch up 
    (or as close as can be managed)

=cut

sub PairLeaders ($$$) {
  my $psp = shift;
  my $minrep = shift;
  my $sr0 = shift;
  Debug 'CP', 'Pairing %d leaders.', scalar(@$psp);
  unless (
    TSH::Player::PairGRT($psp,
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $ketchup = ($o->MaxRank() > $p->RoundRank($sr0));
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d can=%d (svr=%d)', $pid, $oid, $repeats, $sameopp, $ketchup, $pairsvr;
	pack('NCCN',
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $ketchup, # prefer those who can catch up
#	  $pairsvr, # pair those due to start vs those due to reply
	  $_[2], # prefer lower-ranked opponents
	  )
        },
      # allowed opponent filter
      sub {
 	Debug 'GRT', 'filter: %s vs %s %d ?<= %d', $_[0][$_[1]]->TaggedName(), $_[0][$_[2]]->TaggedName(), $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()), $minrep;
	# do not exceed minrep
        $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $minrep
        },
      [],
      )
    ) # unless
    {
    TSH::Utility::Error "Assertion failed: can't resolve leader pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairNonLeaders($$)

This routine is not currently used, as nonleaders are now paired Swiss.

Pair an even number of flight non-leaders with decreasing priority given to
with decreasing priority given to

  - not exceeding minrep (ever)
  - minimizing rematches
  - avoiding consecutive rematches (where possible)
  - matching players due to go first with those due to go second (w.p.)
  - keeping rank differences close to half the group size

=cut

sub PairNonLeaders ($$) {
  my $psp = shift;
  my $minrep = shift;
  Debug 'CP', 'Pairing %d non-leaders.', scalar(@$psp);
  unless (
    TSH::Player::PairGRT($psp,
      # opponent preference ranking
      # $psp is arg 0
      # $pindex is arg 1
      # $oppindex is arg 2
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1)||-1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $distance = abs(@{$_[0]}-abs(2*($_[1]-$_[2])));
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref %d-%d rep=%d prev=%d dist=%d svr=%d', $pid, $oid, $repeats, $sameopp, $distance, $pairsvr;
	pack('NCCNN',
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $pairsvr, # pair those due to start vs those due to reply
	  $distance,# prefer opponents close to half the group size away in rank
	  # provide index for GRT to extract, effectively prefer lower opps
	  $_[2],
	  )
        },
      # allowed opponent filter
      sub {
#	Debug 'GRT', 'filter %d %d', $_[1], $_[2];
  	Debug 'GRT', 'filter: %s vs %s %d ?<= %d', $_[0][$_[1]]->TaggedName(), $_[0][$_[2]]->TaggedName(), $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()), $minrep;
	# do not exceed minrep
        $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $minrep
        },
      [],
      )
    ) # unless
    {
    TSH::Utility::Error "Assertion failed: can't resolve non-leader pairings for: " 
      . join(', ', map { $_->TaggedName() } @$psp);
    }
  return;
  }

=item PairOneGibson($psp, $last_prize_rank)

Pair one gibson.  If the bottom prize band (of players who have 
nothing at stake but ratings and pride) is occupied, the victim is
the highest ranked among those who have played the gibson least often.
If not, then it's the lowest ranked overall (including higher ranked
players who may still be concerned with reaching prize money or
qualification status) among those who have played the gibson least often.

=cut

sub PairOneGibson ($$$$) {
  my $psp = shift;
  my $last_prize_rank = shift;
  my $sr0 = shift;
  my $rounds_left = shift;
  Debug 'CP', 'Pairing one gibson from %d players, lpr=%d.', scalar(@$psp), $last_prize_rank;
  my $gibson = $psp->[0];
  my $gid = $gibson->ID();
  
  if (@$psp % 2) {
    Debug 'CP', 'Gibson gets the bye.';
    $gibson->Division()->Pair($gid, 0, $gibson->CountOpponents());
    splice(@$psp, 0, 1);
    return;
    }

  CalculateBPFs $psp, $sr0, $rounds_left;

  my $victim = undef;
  my $minrep = undef;
  my $hopeless_victim = undef;
  for (my $i = $#$psp; $i > 0; $i--) {
    my $poss_victim = $psp->[$i];
    my $rep = $poss_victim->Repeats($gid);
    my $is_hopeless = $poss_victim->MaxRank() > $last_prize_rank;
    if (!defined $minrep) {
      Debug 'CP', '... victim (rep=%d) = %s', $rep, $poss_victim->TaggedName();
      $minrep = $rep;
      $victim = $i;
      $hopeless_victim = $victim if $is_hopeless;
      }
    elsif ($minrep > $rep) {
      Debug 'CP', '... better victim (rep %d<%d) = %s', $minrep, $rep, $poss_victim->TaggedName();
      $minrep = $rep;
      $victim = $i;
      $hopeless_victim = $victim if $is_hopeless;
      }
    elsif ($is_hopeless && $minrep == $rep) {
      Debug 'CP', '... better victim (rep %d=%d) = %s', $minrep, $rep, $poss_victim->TaggedName();
      $hopeless_victim = $i;
      }
    }
   
  $victim = $hopeless_victim if defined $hopeless_victim;
  $gibson->Division()->Pair($gid, $psp->[$victim]->ID(),
    $gibson->CountOpponents());
  splice(@$psp, $victim, 1);
  splice(@$psp, 0, 1);
  return;
  }

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($sr, $dp) = @_;
  my $sr0 = $sr-1;
  my $start = time;

  $dp->CheckRoundHasResults($sr0) or return;
  unless ($config::max_rounds) 
    { $tournament->TellUser('eneedmxr'); return; }
  my $prize_bands = $config::prize_bands{$dp->Name()};
  unless (defined $prize_bands) 
    { $tournament->TellUser('wneedpzb'); $prize_bands = [1]; }
  print "Calculating Chew pairings for Division ", $dp->Name(), ".\n";
  $dp->ComputeRanks($sr0);

  my $rounds_left = $config::max_rounds - $sr;

  my $round0 = $dp->FirstUnpairedRound0();
  my $tobepaired = $dp->GetUnpairedRound($round0);
  unless (@$tobepaired) { 
    Debug 'CP', "assertion failed at " . __FILE__ . ' line ' . __LINE__ . ": r0=$round0";
    $tournament->TellUser('ealpaird'); 
    return; 
    }
  @$tobepaired = TSH::Player::SortByStanding($sr0, @$tobepaired);

  # while we still have players left to pair
  while (@$tobepaired) {
    Debug 'CP', '%d left to be paired', scalar(@$tobepaired);
    # while there are gibsons
    while (my $ngibsons = CountGibsons($sr0, $tobepaired)) {
#     Debug 'CP', '%d gibson%s.', $ngibsons, ($ngibsons == 1 ? '' : 's');
      # if there is an even number of gibsons
      if ($ngibsons % 2 == 0) {
	# then pair them KOTH but minimize rematches
	PairEvenGibsons($tobepaired, $ngibsons); 
        }
      # else there is an odd number of gibsons
      else { 
	# pair all but the lowest ranked gibson KOTH minimizing rematches
	PairEvenGibsons($tobepaired, $ngibsons-1);
	# pair the last one with a low-ranked victim
	PairOneGibson($tobepaired, $prize_bands->[-1], $sr0, $rounds_left);
        }
      }
    # if the number of remaining players is odd
    if (@$tobepaired % 2) {
      Debug 'CP', 'need a bye';
      # assign bye to lowest ranked among players with fewest byes
      $dp->ChooseBye($sr0, $round0, $tobepaired);
      }
    last unless @$tobepaired;

    # find the highest remaining prize band
    my ($minrank, $maxrank) = GetPrizeBand($prize_bands, $tobepaired, $sr0);
    Debug 'CP', 'prize_band: [%d,%d]', $minrank, $maxrank;
    # count prize band contenders, round up to an even number
    my $ncontenders 
      = CountContenders($maxrank,
	$tobepaired, $sr0, $rounds_left); 
    $ncontenders++ if $ncontenders % 2;
    Debug 'CP', 'There are %d contenders.', $ncontenders;
    # if contenders are more than four times rounds left, cap to that amount
    my $cap = 4 * $rounds_left; # as used by NSA at NSC
#   my $cap = 2 + 2 * $rounds_left; # possible alternate formula
    if ($minrank == 1 && $ncontenders > $cap && $sr > 0) {
      $ncontenders = 4 * $rounds_left;
      Debug 'CP', 'Capping to %d contenders.', $ncontenders;
      }
    die "assertion failed" if $ncontenders > @$tobepaired;
    # compute the smallest number of repeats 'minrep' needed to pair band
    my $minrep = 0;
    my (@flight) = @$tobepaired[0..$ncontenders-1];
    until (CanBePaired $minrep, \@flight) { $minrep++; }
    Debug 'CP', 'minrep is %d.', $minrep;
    # if everyone is still a contender, Swiss-pair them
    if ($ncontenders == @$tobepaired) {
      Debug 'CP', 'everyone is a contender';
      my (@pair_list) = TSH::Division::PairSomeSwiss(\@flight, $minrep, $sr0);
      die "assertion failed" unless @pair_list;
      while (@pair_list) {
	my $p1 = shift @pair_list;
	my $p2 = shift @pair_list;
	$dp->Pair($p1->ID(), $p2->ID(), $round0);
        }
      }
    # else there are some contenders and some noncontenders
    else {
      # find the highest split into two groups pairable in minrep
      my $split = 0;
      while (($split += 2) < $ncontenders) {
	if (CanBePaired $minrep, [@flight[0..$split-1]]) {
	  Debug 'CP', 'Top %d can be paired with rep=%d.', $split, $minrep;
	  }
	else {
	  Debug 'CP', 'Top %d cannot be paired with rep=%d.', $split, $minrep;
	  next;
	  }
	if (CanBePaired $minrep, [@flight[$split..$#flight]]) {
	  Debug 'CP', 'Rest can also be paired with rep=%d.', $minrep;
	  last;
	  }
	else {
	  Debug 'CP', 'Rest cannot be paired with rep=%d.', $minrep;
	  next;
	  }
	}
      Debug 'CP', 'split is %d.', $split;
#     DebugOn('GRT');
      # pair the flight leaders 
      PairLeaders [@flight[0..$split-1]], $minrep, $sr0;
      # pair the rest Swiss
# the old way
#     PairNonLeaders [@flight[$split..$#flight]], $minrep 
#	if $split <= $#flight;
      if ($split <= $#flight) {
	my (@pair_list) = TSH::Division::PairSomeSwiss(
	  [@flight[$split..$#flight]], $minrep, $sr0);
	die "assertion failed" unless @pair_list;
#	die "assertion failed: " . join(',', @pair_list) if @pair_list % 2;
	while (@pair_list) {
	  my $p1 = shift @pair_list;
	  my $p2 = shift @pair_list;
#	  die "assertion failed for $p1" unless UNIVERSAL::isa($p1, 'TSH::Player');
	  die "assertion failed for $p2" unless UNIVERSAL::isa($p2, 'TSH::Player');
	  $dp->Pair($p1->ID(), $p2->ID(), $round0);
	  }
	}
      DebugOff('GRT');
      }
    splice(@$tobepaired, 0, $ncontenders);
    }

  $dp->Dirty(1);
  $dp->Update();
  $tournament->TellUser('idone');
  Debug('CP', '%d second(s) runtime', time - $start);
  0;
  }

=back

=cut

=bugs

=head1 BUGS

Gibsonization should consider results from partially scored rounds.

C<CalculateBestPossibleFinish()> should just distribute wins kalah-style,
then spread points too
constrained by number of wins scored and total spread.
e.g. 0-4: [-1200,-4], 1-3: [-899,299].

We should not be calling C<main::ResolvePairings()>, which should
be moved to C<Division.pm>.

The current algorithm assumes (e.g.) that if the top eight players
are paired with each other to contend for the top two places, then
the ninth-ranked player cannot finish higher than ninth place, and
is paired with other such players.

Check to see if inactive players and odd divisions are handled
correctly.  Add a command to inactivate players.

CountGibsons should perhaps be repeatedly called until it returns zero.

=cut


1;
