#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::HELP;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::HELP - implement the C<tsh> HELP command

=head1 SYNOPSIS

  my $command = new TSH::Command::HELP;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::HELP is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to view built-in documentation.  For fuller details,
please consult the HTML reference manual.  Enter 'help index' to see a
list of available commands, and 'help' followed by a command name
to find out more about that command.
EOF
  $this->{'names'} = [qw(help)];
  $this->{'argtypes'} = [qw(Topic)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $topic = shift;
  if ($topic !~ /^index$/i) {
    my $command = $tournament->GetCommandByName($topic);
    unless ($command) {
      $tournament->TellUser('enohelp', $topic);
      return;
      }
    print $command->Usage();
    print "\n";
    print TSH::Utility::Wrap(0, $command->Help());
    return;
    }

  print TSH::Utility::Wrap(0, <<'EOS');
The help system is being redesigned.  Detailed online help
is available for the following commands:
EOS
  my (@commands) = $tournament->CommandNames();
  print TSH::Utility::Wrap(2, @commands);
  print <<'EOS';

Here is the older help text for commands not listed below.

Commands Available:

Look word                    look up 'word' in dictionary
Pair1324 rpt sr d            pair 1-3, 2-4, 5-7, 6-8 etc.
PairMany r d                 make several pairings changes at once
RoundStandings r d           show standings after given round
STandings d                  show current standings

Notes:
- You can type the whole command name or just the part in caps
- 'd' stands for a division name, 'r' for a round number
- 'sr' stands for a 'source round' on which pairings are to be based
- 'p1' and 'p2' are player numbers, 's1' and 's2' are scores
- 'rpt' specifies how many times two players can repeat pairings
EOS
  }

=back

=cut

1;
