#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ShowPairings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ShowPairings - implement the C<tsh> ShowPairings command

=head1 SYNOPSIS

  my $command = new TSH::Command::ShowPairings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ShowPairings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub CheckAutoPair ($$$);
sub CheckChewPair ($$$);
sub initialise ($$$$);
sub new ($);
sub Run ($$@);
sub ShowBye ($$$);
sub ShowHeader ($$$);
sub ShowUnpaired ($$$);

=item $success = CheckAutoPair($tournament, $div, $round)

Check to see if we are ready to generate pairings for division $div 
in round $round.

=cut

sub CheckAutoPair ($$$) {
  my $tournament = shift;
  my $dp = shift;
  my $round = shift;
  my $round0 = $round - 1; # internally, rounds are zero-based
# print "Checking AutoPair.\n";
  # first look to see if there are any unpaired 
  # TODO: check to see if this duplicates one of the sub Get...s
  my (@unpaired) = @{$dp->GetUnpairedRound($round0)};
  return 0 unless @unpaired;
  return 0 if $config::manual_pairings;
  if (!%config::autopair) {
    return CheckChewPair $tournament, $dp, $round0;
    }
  my $apdp = $config::autopair{uc $dp->Name()}[$round];
  return 0 unless $apdp;
  my (@apd) = @{$apdp};

  my $sr = shift @apd;
  my $sr0 = $sr - 1;
  if ($sr0 > $dp->LeastScores()-1) {
    $tournament->TellUser('emisss2', $dp->Name(), $sr, $dp->LeastScoresPlayer()->TaggedName());
    return 0;
    }
  # check to see if all results are in for the source round
  my $system = $apd[0];
  # check to see we aren't going too far ahead
  if ($round0 != $dp->FirstUnpairedRound0()) {
    $tournament->TellUser('eapwrr', $round, $dp->FirstUnpairedRound0()+1);
    return 0;
    }
  $tournament->TellUser('iautopr');
  if ($system =~ /^(?:cp|if|koth|ns|newswiss|p1324|pair1324|roundrobin|rr)$/i) {
    if ($tournament->RunCommand(@apd)) {
      return 1;
      }
    else {
      $tournament->TellUser('eapfail', "@apd");
      return 0;
      }
    }
  else { 
    $tournament->TellUser('ebadapc', $system);
    return 0;
    }
  }
  
=item $success = CheckChewPair($tournament, $div, $round0)

Check to see if we should generate Chew pairings for the given div/round.
Return 1 if we computed them.

=cut

sub CheckChewPair ($$$) {
  my $tournament = shift;
  my $dp = shift;
  my $round0 = shift;

  if ($round0 != $dp->FirstUnpairedRound0()) {
    $tournament->TellUser('eapwrr', $round0+1, $dp->FirstUnpairedRound0()+1);
    return 0;
    }
  # check to see if the source round seems reasonable
  my $sr0 = $dp->LeastScores() - 1;
  # sr0 = previous round is always reasonable, else...
  if ($sr0 != $round0 - 1) {
    # sr0 neither of previous two rounds is never reasonable
    if ($sr0 != $round0 - 2) {
      $tournament->TellUser('eacpbadr', $round0+1, $sr0+1);
      return 0;
      }
    # sr0 = second previous round is always ok if not after a session break
    if ($config::session_breaks) {
      for my $sb (@$config::session_breaks) {
	if ($round0 == $sb) {
	  $tournament->TellUser('eacprnsb', $round0+1, $round0);
	  return 0;
	  }
        }
      }
    else {
      $tournament->TellUser('eacpnsb', $round0+1, $round0-1);
      return 0;
      }
    }
  $tournament->RunCommand('chewpair', $sr0+1, $dp->Name());
  return 1;
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display the pairings for the specified round
and division.
EOF
  $this->{'names'} = [qw(sp showpairings)];
  $this->{'argtypes'} = [qw(Round Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($round, $dp) = @_;
# my $opt_p = 0;
# if (@$argvp && $argvp->[0] eq '-p') {
#   shift @$argvp;
#   $opt_p = 1;
#   }
  my $round0 = $round-1;

  CheckAutoPair($tournament, $dp, $round);
  if ($round0 > $dp->LastPairedRound0()) {
    $tournament->TellUser('enopryet', $dp->Name(), $round);
    return;
    }
# # a kludge formerly used with make-rr.pl
# if ($opt_p) {
#   print '[';
#   print join(',', map { $_->{'pairings'}[$round0]-1 } @$datap[1..$#$datap]);
#   print "]\n";
#   return 0;
#   }
  my $logp = new TSH::Log($dp, 'pairings', $round);
  # sample config line: perl $config'tables{'A'} = [1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10]; # (for 20 2-board tables)
  my $tables = $config::tables{$dp->Name()};
  # sample config line: perl $config'reserved{'P'}[13] = 4; # (to permanently station disabled player #13 in division P at board 4) 
  my $reserved = $config::reserved{$dp->Name()};
  ShowHeader $tournament, $logp, $tables;
  {
    my %done;
    my %reserved;
    my @unreserved;
    
    my @sorted;
    {
      my $sr0 = $round - 2;
      $sr0 = 0 if $sr0 < 0;
      $sr0 = $dp->MostScores() if $sr0 > $dp->MostScores()-1;
      @sorted = TSH::Player::SortByStanding $sr0, $dp->Players();
    }
    for my $p (@sorted) {
      next unless $p->Active();
      my $oppid = $p->OpponentID($round0, 'undef for unpaired');
      my $pid = $p->ID();
      if (!defined $oppid) {
	ShowUnpaired $logp, $tables, $p;
        }
      elsif ($oppid == 0) {
	ShowBye $logp, $tables, $p;
        }
      elsif (!$done{$pid}++) {
	my $opp = $dp->Player($oppid);
	if ($pid != $opp->OpponentID($round0)) {
	  my $ooid = $opp->OpponentID($round0);
	  my $oppoppname = $ooid ? $opp->Opponent($round0)->Name() : 'bye';
	  $tournament->TellUser('ebadpair', 
	    $pid, $p->Name(),
	    $oppid, $opp->Name(),
	    $ooid, $oppoppname);
	  }
	elsif ($pid == $oppid) {
	  $tournament->TellUser('eselpair', $pid, $p->Name());
	  }
	else {
	  $done{$oppid}++;
	  }
#	Debug 'SP', 'checking board for %s', $p->TaggedName();
	if ($p->Board($round0)) { 
#	  Debug 'SP', '%s has assigned board: %d', $p->TaggedName(), $p->Board($round0);
	  my $board = $p->Board($round0);
	  if (exists $reserved{$board}) {
	    printf STDERR "%s is assigned to board %d, which is already occupied by %s and %s.\n", $p->TaggedName(), $board, $reserved{$board}[0]->TaggedName(), $reserved{$board}[1]->TaggedName();
	    }
	  else {
	    $reserved{$board} = [$p, $opp]; 
	    $reserved->[$pid] = $board;
	    $reserved->[$oppid] = $board;
	    next;
	    }
	  }
	elsif (defined $reserved && defined $reserved->[$pid]) { 
	  my $board = $reserved->[$pid];
	  if (exists $reserved{$board}) {
	    printf STDERR "%s is stationed at board %d, which is already occupied by %s and %s.\n", $p->TaggedName(), $board, $reserved{$board}[0]->TaggedName(), $reserved{$board}[1]->TaggedName();
	    }
	  else {
	    $reserved{$board} = [$p, $opp]; 
	    next;
	    }
	  }
        elsif (defined $reserved && defined $reserved->[$oppid]) {
	  my $board = $reserved->[$oppid];
	  if (exists $reserved{$board}) {
	    printf STDERR "%s is stationed at board %d, which is already occupied by %s and %s.\n", $opp->TaggedName(), $board, $reserved{$board}[0]->TaggedName(), $reserved{$board}[1]->TaggedName();
	    }
	  else {
	    $reserved{$board} = [$opp, $p]; 
	    next;
	    }
	  }
#	Debug 'SP', 'unreserved seating: %s %s', $p->TaggedName(), $opp->TaggedName();
	push (@unreserved, [$p, $opp]); 
        }
      } # for $p
    {
      # $tables is zero-indexed, but players prefer one-indexed tables
      for (my $board=0; ; $board++) {
	my $p1;
	my $p2;
	if ($reserved{$board+1}) {
	  ($p1, $p2) = @{$reserved{$board+1}};
	  }
	elsif (@unreserved) { # take next pair from unreserved queue
	  ($p1, $p2) = @{shift @unreserved};
	  }
	else { last; }
	$logp->Write('', '<tr>');
        $logp->Write(sprintf(" $config'table_format  ", $tables->[$board]),
	  "<td class=table>$tables->[$board]</td>")
	  if defined $tables;
	$p1->Board($round0, $board+1) unless $p1->Board($round0);
	$p2->Board($round0, $board+1) unless $p2->Board($round0);
	my $vs = $dp->FormatPairing($round0, $p1->ID());
	unless ($vs) {
	  my $pname = $p1->Name();
	  $logp->Write("Lost track of $pname", "Lost track of $pname");
	  next;
	  }
	my $vshtml = $vs;
	$vshtml =~ s/\*(?:starts|draws)\*/<span class=starts>$&<\/span>/;
        $logp->Write(sprintf(" %3d  %s.\n", $board+1, $vs),
	  '<td class=board>' . ($board+1) . "</td><td class=name>$vshtml</td></tr>\n"
	  );
	}
    }
  $logp->Close();
  }
  $dp->Update(); # in case table numbers were changed
  }

=item ShowBye $log, $tables, $p

Used internally to show a bye player

=cut

sub ShowBye ($$$) {
  my $logp = shift;
  my $tables = shift;
  my $p = shift;

  $logp->Write('', '<tr>');
  $logp->Write(sprintf("  $config'table_format ", ''), 
    '<td class=notable>&nbsp;</td>') if defined $tables;
  $logp->Write(sprintf("      %s: BYE.\n", ($p->TaggedName())),
    '<td class=bye>BYE</td><td class=name>' . 
    ($p->TaggedName()) . '</td>');
  $logp->Write('', '</tr>');
  }

=item ShowHeader $tournament, $log, $tables

Used internally to show the header of the pairings listings

=cut

sub ShowHeader ($$$) {
  my $tournament = shift;
  my $logp = shift;
  my $tables = shift;

  if (defined $tables) {
    $logp->Write('Table ', '<th class=table>Table');
    my $shortage = length(sprintf($config::table_format, ''))-3;
    if ($shortage < 0) {
      $tournament->TellUser('wsmtfmt');
      $config::table_format = '%3s';
      $shortage = 0;
      }
    $logp->Write((' ' x $shortage), '</th>');
    }
  $logp->Write("Board Players\n", <<'EOF');
<th class=board>Board</th>
<th class=name>Who Plays Whom</th>
</tr>
EOF
  }

=item ShowUnpaired $log, $tables, $p

Used internally to show an unpaired player

=cut

sub ShowUnpaired ($$$) {
  my $logp = shift;
  my $tables = shift;
  my $p = shift;

  $logp->Write('', '<tr>');
  $logp->Write(sprintf("  $config'table_format ", ''), 
    '<td class=notable>&nbsp;</td>') if defined $tables;
  $logp->Write(sprintf("      %s: UNPAIRED.\n", ($p->TaggedName())),
    '<td class=unpaired>UNPAIRED</td><td class=name>' . 
    ($p->TaggedName()) . '</td>');
  $logp->Write('', '</tr>');
  }

=back

=cut

=head1 BUGS

Auto-pairing commands should be indicated through subclassing,
not pattern matching on names.

A reasonable guess should be made when there are not enough tables
configured to accommodate the boards needed.

enopryet should be reported even when only inactive players are paired

=cut

1;
