#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::ParseArgs::Division;

use strict;
use warnings;

use TSH::Division;
use TSH::Utility;

=pod

=head1 NAME

TSH::ParseArgs::Division - handle a command-line argument that should be a tournament division name

=head1 SYNOPSIS

  my $parser = new TSH::ParseArgs::Division;

=head1 ABSTRACT

This Perl module is used by C<ParseArgs.pm> to ignore command-line arguments.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($);
sub new ($);
sub Parse ($$);
sub Usage ($);

=item $parserp->initialise();

Used internally to (re)initialise the object.

=cut

sub initialise ($) {
  my $this = shift;
  return $this;
  }

=item $parserp = new ParserArgs::Division

Create a new instance

=cut

sub new ($) { TSH::Utility::new(@_); }

=item $argument_description = $this->Parse($line_parser)

Return the checked value of the command-line argument,
or () if the argument is invalid.

=cut

sub Parse ($$) {
  my $this = shift;
  my $line_parser = shift;
  my $value = $line_parser->GetArg();
  my $dp;
  if (defined $value) {
    $dp = $main::gTournament->GetDivisionByName($value);
    return $dp if defined $dp;
    }
  if ($main::gTournament->CountDivisions() == 1) {
    $line_parser->UnGetArg($value) if defined $value;
    return ($main::gTournament->Divisions())[0];
    }
  elsif (defined $value) {
    $line_parser->Error("I don't know a Division '$value'.");
    return ();
    }
  else {
    $line_parser->Error("You need to specify a division with this command.");
    return ();
    }
  }

=item $argument_description = $this->Usage()

Briefly describe this argument's usage (in a word or hyphenated phrase)

=cut

sub Usage ($) {
  my $this = shift;
  if ($main::gTournament->CountDivisions() == 1) {
    return '';
    }
  else {
    return 'division';
    }
  }

=back

=head1 BUGS

This module relies on C<tsh> internals to an unhealthy extent.

=cut

1;
