#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::MISSing;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::MISSing - implement the C<tsh> MISSing command

=head1 SYNOPSIS

  my $command = new TSH::Command::MISSing;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::MISSing is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to see which player scores have yet to be entered
in a given division.
EOF
  $this->{'names'} = [qw(miss missing)];
  $this->{'argtypes'} = [qw(Round)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($round) = @_;
  my $round0 = $round-1;
  
  for my $dp (sort { $a->Name() cmp $b->Name() } $tournament->Divisions()) {
    my @done = ();
    for my $p ($dp->Players()) {
      next unless $p->Active();
      my $id = $p->ID();
      next if $done[$id];
      next if defined $p->Score($round0);
      if (my $board = $p->Board($round0)) { 
	printf "Board $config::table_format. ", $board;
        }
      my $opp = $p->Opponent($round0);
      if ($opp) {
	print $dp->FormatPairing($round0, $id);
	$done[$opp->ID()] = 1;
	}
      else {
	print $p->TaggedName();
	}
      print "\n";
      }
    }
  0;
  }
 
=back

=cut

=head1 BUGS

=cut

1;
