#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundStandings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RoundStandings - implement the C<tsh> RoundStandings command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundStandings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundStandings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display standings as of a specific round.
EOF
  $this->{'names'} = [qw(rs roundstandings)];
  $this->{'argtypes'} = [qw(BasedOnRound Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($round, $dp) = @_;

  my $round0 = $round - 1;
  $dp->CheckRoundHasResults($round0) or return 0;

  my $dname = $dp->Name();
  print "Round $round Standings: Division $dname.\n";
  print "\n";
  print "Rank  Won-Lost Spread Rtng Player\n\n";

  $dp->ComputeRanks($round0);
  my (@sorted) = TSH::Player::SortByStanding $round0, $dp->Players();
  TSH::Player::SpliceInactive(@sorted, 1, $round0);
  for my $p (@sorted) {
    my $wins = $p->RoundWins($round0);
    my $games = $p->GamesPlayed();
    $games = $round0 + 1 if $round0 + 1 < $games;
    my $losses = $games - $wins;

    printf "%4d %4.1f-%4.1f %+5d  %4d %s\n", 
      $p->RoundRank($round0),
      $wins, $losses, $p->RoundSpread($round0),
      $p->Rating(), ($p->TaggedName()) unless $p->{name} =~ /^bye /;
    }
  }

=back

=cut

=head1 BUGS

Makes unauthorized use of TSH::Player internals.

=cut

1;

