#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Utility;

use strict;
use warnings;

use Symbol;

our(@ISA) = 'Exporter';
our(@EXPORT_OK) = qw(Debug DebugOn DebugOff DebugDumpPairings);

=pod

=head1 NAME

TSH::Utility- miscellaneous Perl utilities

=head1 SYNOPSIS

  sub new { Util::new(@_); }
  PrintColour $colour_name, $text;
  Error "You goofed.\n";
  print TaggedName($player);
  print Wrap($indent, @text);
  $old_value = GetOrSet($object, $field, $new_value);
  $current_value = GetOrSet($object, $field);
  DebugOn($code);
  DebugOff($code);
  Debug($code, $format, @args);
  DebugDumpPairings($code, $round0, $psp);
  Prompt('subtsh>');

=head1 ABSTRACT

This library contains miscellaneous bits of code used by in more than 
one C<tsh> source file.

=cut

sub Colour ($$);
sub Debug ($$@);
sub DebugDumpPairings($$$);
sub DebugOff ($);
sub DebugOn ($);
sub Error ($);
sub GetOrSet ($@);
sub new (@);
sub PrintColour ($$);
sub Prompt ($);
sub TaggedName ($);
sub Wrap ($@);

my %Colours = (
  'red' => { 'ansi' => "\e[31m", 'win' => $::FG_RED },
  'green' => { 'ansi' => "\e[32m", 'win' => $::FG_GREEN },
  'blue' => { 'ansi' => "\e[34m", 'win' => ($::FG_BLUE||0)|($::BG_WHITE||0) },
  'yellow on blue' => { 'ansi' => "\e[44;33m", 'win' => (($::FG_YELLOW||0) | ($::BG_BLUE||0)) },
  'plain' => { 'ansi' => "\e[30;47;0m", 'win' => $::FG_GRAY },
  );
my %gDebug;
my $gDebugFH;
my $WinConsole; 
BEGIN { 
  if ($^O eq 'MSWin32') { 
    require Win32::Console; import Win32::Console;
    $WinConsole = new Win32::Console &STD_OUTPUT_HANDLE;
    };
  }

=head1 DESCRIPTION

=over 4

=cut
 
=item Colour($colour, $text)

Add escape sequences to add colour to text.
Deprecated in favour of PrintColour because Windows console
doesn't use escape sequences.

=cut

sub Colour ($$) {
  my $colour = shift;
  my $text = shift;
  return $text if (defined $config'colour) && $config'colour =~ /^(?:no|0)$/i;
  return $text if $^O eq 'MSWin32' &&
    ((!defined $config'colour) || $config'colour !~ /^yes$/i);
  my $escape = $Colours{$colour}{'ansi'};
  die "Unknown colour: $colour\n" unless defined $escape;
  return "$escape$text$Colours{'plain'}{'ansi'}";
  }

=item Debug($code, $format, @args)

Display and log debug text.

=cut

sub Debug ($$@) {
  my $code = shift;
  return unless $gDebug{$code};
  my $format = shift;
  my $s = "[$code] " .  sprintf($format, @_);
  $s .= "\n" unless $s =~ /\n$/;
  print "Debug: " . $s;
  unless ($gDebugFH) {
    $gDebugFH = gensym;
    open($gDebugFH, ">debug.txt");
    }
  if ($gDebugFH) {
    print $gDebugFH $s;
    }
  }

=item DebugDumpPairings($code, $round0, $psp)

Dump the $round0 pairings for the players in $psp if DebugOn($code);

=cut

sub DebugDumpPairings($$$) {
  my $code = shift;
  my $round0 = shift;
  my $psp = shift;
  return unless $gDebug{$code};
  Debug $code, 'Pairings:';
  my %done;
  for my $i (0..$#$psp) {
    my $p = $psp->[$i];
    my $opp = $p->Opponent(-1);
    next if $done{$p->ID()};
    $done{$opp->ID()}++;
    Debug $code, '... %s vs %s.', $p->TaggedName(), $opp->TaggedName();
    }
  }

=item DebugOff($code)

Turn off debugging of type $code.

=cut

sub DebugOff ($) {
  my $code = shift;
  $gDebug{$code} = 0;
  }

=item DebugOn($code)

Turn on debugging of type $code.

=cut

sub DebugOn ($) {
  my $code = shift;
  $gDebug{$code} = 1;
  }

=item Error($text)

Display an error message.

=cut

sub Error ($) {
  my $message = shift;
  $message .= "\n" unless $message =~ /\n$/;
  print Colour 'red', $message;
  }

=item GetOrSet()

Boilerplate code for get/set methods.

=cut

sub GetOrSet ($@) {
  my $field = shift;
  my $object = shift;
  my $value = shift;
  my $old = $object->{$field};
  if (defined $value) {
    $object->{$field} = $value;
    }
  return $old;
  }

=item new()

Boilerplate code for creating Perl objects.

=cut

sub new (@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $this = { };
  bless($this, $class);
  $this->initialise(@_);
  return $this;
  }

=item PrintColour($colour, $text)

Print text in specified colour.

=cut

sub PrintColour ($$) {
  my $colour = shift;
  my $text = shift;
  return $text if (defined $config'colour) && $config'colour =~ /^(?:no|0)$/i;
  if ($^O eq 'MSWin32') { 
    my $attr = $Colours{$colour}{'win'};
    die "Unknown colour: $colour\n" unless defined $attr;
    $WinConsole->Attr($attr);
    print $text;
    $WinConsole->Attr($Colours{'plain'}{'win'});
    }
  else {
    my $escape = $Colours{$colour}{'ansi'};
    die "Unknown colour: $colour\n" unless defined $escape;
    print "$escape$text$Colours{'plain'}{'ansi'}";
    }
  }

=item Prompt($text)

Displays the given prompt in blue, followed by a plain space.

=cut

sub Prompt ($) {
  my $text = shift;
  PrintColour 'blue', $text;
  print ' ';
  }

=item TaggedName($player)

Safe wrapper for TSH::Player::TaggedName

=cut

sub TaggedName ($) {
  my $p = shift;
  if (UNIVERSAL::isa($p,'TSH::Player')) {
    return $p->TaggedName();
    }
  else {
    return 'nobody';
    }
  }

=item Wrap($indent, @text)

Justify text right-ragged.

=cut

sub Wrap ($@) {
  my $indent = shift;
  my $indent_space = ' ' x $indent;
  my $width = 78;
  my $s = $indent_space;
  my $hpos = $indent;
  my $atsol = 1;
  while (@_) {
    my (@words) = split(/\s+/, shift);
    for my $word (@words) {
      $word =~ s/\.$/. / unless $word =~ /\../;
      my $l = length($word);
      if ($hpos + $l > $width) {
	$s .= "\n$indent_space$word";
	$hpos = $indent + $l;
        }
      else {
	unless ($atsol) {
	  $word = " $word";
	  $l++;
	  }
	$s .= $word;
	$hpos += $l;
	$atsol = 0;
        }
      }
    }
  $s .= "\n";
  return $s;
  }

=back

=cut

=head1 BUGS

C<Colour>,
C<PrintColour>
and
C<Wrap>
should be in something called 
C<TSH::TTY>.

C<Wrap> should dynamically determine the width of the
current console.

=cut

1;
