#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::STandings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::STandings - implement the C<tsh> STandings command

=head1 SYNOPSIS

  my $command = new TSH::Command::STandings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::STandings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display standings as of a specific round.
EOF
  $this->{'names'} = [qw(st standings)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;

  my $round = $dp->MostScores();
  my $logp = new TSH::Log($dp, 'standings', $round);

  $logp->Write("Rank  Won-Lost Spread Rtng Name\n\n", <<'EOF');
<tr>
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
<th class=rating>Rating</th>
<th class=name>Name</th>
</tr>
EOF

  my $lastw = -1; my $lasts = 0; my $rank = 0; my $i = 0;
  for my $p (TSH::Player::SortByCurrentStanding $dp->Players()) {
    next unless $p->Active();
    my $wins = $p->Wins();
    my $spread = $p->Spread();
    $i++;
    if ($wins != $lastw || $spread != $lasts) {
      $lastw = $wins;
      $lasts = $spread;
      $rank = $i;
      }
    unless ($p->Name() =~ /^bye /) {
      my (@fields) = ($rank, $wins, $p->CountScores()-$wins, 
	  $spread, $p->Rating(), $p->TaggedName());
      $logp->Write(
        sprintf("%4d %4.1f-%4.1f %+5d  %4d %s\n", @fields),
	sprintf(<<'EOF', @fields));
<tr>
<td class=rank>%d</td>
<td class=wl>%4.1f-%4.1f</td>
<td class=spread>%+d</td>
<td class=rating>%d</td>
<td class=name>%s</td>
</tr>
EOF
      }
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

May be deprecated in future in favour of RATings command,
with a configuration option to suppress ratings estimates.

=cut

1;
