#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::DELETEscore;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::DELETEscore - implement the C<tsh> DELETEscore command

=head1 SYNOPSIS

  my $command = new TSH::Command::DELETEscore;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::DELETEscore is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to delete entered scores.
Deleting scores is rarely necessary, and
the EditScore command should generally be used instead to 
correct scores.
EOF
  $this->{'names'} = [qw(delete deletescore)];
  $this->{'argtypes'} = [qw(Player Score Player Score Round Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($pn1, $s1, $pn2, $s2, $round, $dp) = @_;
  my $round0 = $round - 1;

  my $pp1 = $dp->Player($pn1);
  unless ($pp1) { $tournament->TellUser('enosuchp', $pn1); return; }
  my $pp2 = $dp->Player($pn2);
  unless ($pp2) { $tournament->TellUser('enosuchp', $pn2); return; }
  my $pname1 = $pp1->Name();
  my $pname2 = $pp2->Name();
  my $ps1 = $pp1->Score($round0);
  my $ps2 = $pp2->Score($round0);
  if (($pp1->OpponentID($round0)||-1) ne $pn2) {
    TSH::Utility::Error "$pname1 and $pname2 did not play each other in round $round.\n";
    return;
    }
  if ($ps1 != $s1) {
    TSH::Utility::Error "${pname1}'s score was $ps1, not $s1.\n";
    return;
    }
  if ($ps2 != $s2) {
    TSH::Utility::Error "${pname2}'s score was $ps2, not $s2.\n";
    return;
    }
  if ($dp->DeleteScores($pn1, $pn2, $round0)) {
    $tournament->TellUser('idone');
    $dp->Dirty(1);
    $dp->Update();
    }
  }

=back

=cut

=head1 BUGS

You should be able to delete a bye, either with this command or
another similar one, but you can't.

=cut

1;
