#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::UPSETs;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::UPSETs - implement the C<tsh> UPSETs command

=head1 SYNOPSIS

  my $command = new TSH::Command::UPSETs;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::UPSETs is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the UPSETs command to list biggest ratings upsets in a division.
EOF
  $this->{'names'} = [qw(upset upsets)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;

  my (@upsets);
  for my $p ($dp->Players()) {
    for my $r (1..$p->CountScores()) {
      my $r0 = $r-1;
      my $ms = $p->Score($r0);
      my $oid = $p->OpponentID($r0);
      next unless $oid;
      my $os = $p->OpponentScore($r0);
      next unless $ms > $os;
      my $o = $p->Opponent($r0);
      my $diff = $o->Rating() - $p->Rating();
      next unless $diff > 0;
      push(@upsets, [$diff, $r0, $o, $p]);
      if (@upsets > 100) {
	@upsets = (sort { $b->[0] <=> $a->[0] } @upsets)[0..19];
        }
      }
    }
  if (@upsets > 20) {
    @upsets = (sort { $b->[0] <=> $a->[0] } @upsets)[0..19];
    }

  my $logp = new TSH::Log($dp, 'upsets');
  $logp->Write(
    sprintf("%4s %4s %4s %2s  %5s  %-25s %-25s\n",
       qw(Diff Rtg1 Rtg2 Rd Score Player1 Player2)),
    "<th>Ratings Difference</th>"
    . "<th>Rating 1</th>"
    . "<th>Rating 2</th>"
    . "<th>Round</th>"
    . "<th>Score 1</th>"
    . "<th>Score 2</th>"
    . "<th>Player 1</th>"
    . "<th>Player 2</th>"
    . "</tr>\n"
  );
  for my $upset (@upsets) {
    my ($diff, $r0, @p) = @$upset;
    my (@rating, @score, @name);
    for my $i (0..1) { 
      my $p = $p[$i];
      $rating[$i] = $p->Rating();
      $score[$i] = $p->Score($r0);
      $name[$i] = $p->Name();
      }
    $logp->Write(
      sprintf("%4d %4d %4d %2d %3d-%3d %-25s %-25s\n",
	$diff, $rating[0], $rating[1], $r0+1, $score[0], $score[1], 
	$name[0], $name[1]),
      "<tr>"
      . "<td>$diff</td>"
      . "<td>$rating[0]</td>"
      . "<td>$rating[1]</td>"
      . "<td>" . ($r0+1) . "</td>"
      . "<td>$score[0]</td>"
      . "<td>$score[1]</td>"
      . "<td>$name[0]</td>"
      . "<td>$name[1]</td>"
      . "</tr>"
      );
    }
  $logp->Close();
  }

=back

=cut

1;
