#!/usr/bin/perl -w

## tsh - tournament shell
## Copyright (C) 1998-2003 by John J. Chew, III.

=head1 NAME

B<tsh> - Scrabble tournament management shell

=head1 SYNOPSIS

B<tsh> [directory|configuration-file]

=head1 DESCRIPTION

For user information, see the accompanying HTML documentation.
This builtin (pod) documentation is intended for program maintenance 
and development.

=cut

## Version history: moved to doc/news.html

## public libraries

BEGIN { unshift(@::INC, "$::ENV{'HOME'}/lib/perl") if defined $::ENV{'HOME'}; }
use strict;
use lib './lib/perl';
use Fcntl ':flock';
use FileHandle;
use Symbol;
use UNIVERSAL qw(isa);

# use warnings FATAL => 'uninitialized';

## private libraries

use TSH::Command;
use TSH::Division;
use TSH::PairingCommand;
use TSH::ParseArgs;
use TSH::Log;
use TSH::Player;
use TSH::Tournament;
use TSH::XCommand;
use UserMessage;

## global constants
if ($^O eq 'MacOS') { 
  $config'backup_directory = ':old:'; 
  $config'html_directory = ':html'; 
  }
else { 
  $config'backup_directory = './old/'; 
  $config'html_directory = './html/'; 
  }
$config'max_name_length = 22;
$config'name_format = '%-22s';
$config'external_path = [qw(./bin)];
our $gkVersion = '3.070';

## prototypes

# sub CheckGroupRepeats ($$);
sub ChooseConfig ();
# sub CmdFactorPairQuads ($$);
sub DefineExternal ($$$);
# sub DoFactor ($$$$);
# sub DoFactorGroup ($$$);
sub lint ();
sub LockFailed ($);
sub LockOff ();
sub LockOn ();
sub Main ();
sub Prompt ();
sub ReadConfig ($);
sub ReadDivision (\%);
sub ReadDivisions ();
sub ReopenConsole ();
sub ResolvePairings ($;$);

## global variables

my %gDivision;
my $gNDivisions;
# $gTournament should gradually subsume %gDivision, $gNDivisions and others
our $gTournament; 

=head1 SUBROUTINES

=over 4

=cut

# Part of the not yet ready CmdFactorPairQuads
# sub CheckGroupRepeats ($$) {
#   my $psp = shift;
#   my $repeats = shift;
#   for my $i (0..$#$psp) {
#     my $repeatsp = $psp->[$i]{'repeats'};
#     die "Player $psp->[$i]{'name'} has no repeats information.\n"
#       unless ref($repeatsp) eq 'ARRAY';
#     for my $j ($i+1..$#$psp) {
#       if ((my $this_repeats = $repeatsp->[$psp->[$j]{'id'}]) > $repeats) {
# #	TSH::Utility::Error "Warning: $psp->[$i]{'name'} and $psp->[$j]{'name'} have played each other $this_repeats time(s).\n";
# 	return 0;
#         }
#       }
#     }
#   return 1;
#   }

sub ChooseConfig () {
  $config'root_directory = '.';
  # first choice: something specified on the command line
  if (@::ARGV) {
    my $argv = shift @::ARGV;
    # if it's a directory
    if (-d $argv) {
      for my $try (qw(config.tsh tsh.config)) {
	if (-e "$argv/$try") {
	  $config'root_directory = $argv;
	  return $try;
	  }
        }
      die "$argv is a directory but has neither config.tsh nor tsh.config\n";
      }
    # else it's a configuration file
    elsif (-f $argv) {
      if ($argv =~ /\//) {
	($config'root_directory, $argv) = $argv =~ /^(.*)\/(.*)/;
	return $argv;
        }
      else { return $argv; }
      }
    else { die "No such file or directory: $argv\n"; }
    }
  # second choice: the directory containing the newest of */config.tsh
  if (my (@glob) = glob('*/config.tsh')) {
    # if more than one, choose newest
    if (@glob > 1) {
      my $newest_age = -M $glob[0];
      while (@glob > 1) {
	my $this_age = -M $glob[1];
	if ($this_age < $newest_age) {
	  $newest_age = $this_age;
	  shift @glob;
	  }
	else {
	  splice(@glob, 1, 1);
	  }
        }
      }
    ($config'root_directory, $glob[0]) = $glob[0] =~ /^(.*)\/(.*)/;
#   print "Directory: $config'root_directory\n";
    if ($config'root_directory =~ /^sample\d$/) {
      print "If you didn't mean to use one of the sample directories,\n";
      print "you need to create a subdirectory of files for your own event.\n";
      }
    return $glob[0];
    }
  # third choice: ./tsh.config
  if (-f 'tsh.config') {
    # don't use colour here, as colour configuration hasn't been found
    print "Warning: use of tsh.config is deprecated.\n";
    print "Please place your event files in a subdirectory, and save your\n";
    print "configuration in that subdirectory under the name config.tsh.\n";
    }
  return 'tsh.config';
  }

# FactorPairQuads is not yet ready for public use
# sub CmdFactorPairQuads ($$) { my($argvp, $args) = @_;
#   my ($factor, $repeats, $sr, $dp) 
#     = ParseArgs $argvp, [qw(factor repeats based-on-round division)];
#   return 0 unless defined $dp;
#   my $sr0 = $sr-1;
#   $dp->CheckRoundHasResults($sr0) or return 0;
#   print "Calculating Factored Pairings for Division $dp->{'name'} based on round $sr, $repeats repeats allowed, factoring by $factor.\n";
#   DoFactor $dp, $repeats, $sr0, $factor;
#   return 0;
#   }
# 
  
sub DefineExternal ($$$) {
  my $name = lc shift;
  my $script = shift;
  my $template = shift;

  my $command = new TSH::XCommand("$global'path/$script", [$name], $template);
  $gTournament->AddCommand($command);
# print " $name";
  return 1;
  }

# DoFactor is not yet ready for public use
# sub DoFactor ($$$$) { my ($dp, $repeats, $sr0, $factor) = @_;
#   my $datap = $dp->{'data'};
#   my $theKeyRound = $sr0;
# 
#   my $tobepaired = $dp->GetRegularUnpaired($sr0, 'nobyes');
#   unless (@$tobepaired) {
#     TSH::Utility::Error "No players can be paired.\n";
#     return 0;
#     }
# # die "Assertion failed" unless @$tobepaired % 2 == 0;
#   my $minbyes = 0;
#   if (@$tobepaired % 2) {
#     $minbyes = CountByes $dp;
#     }
#   my (@ranked) = TSH::Player::SortByStanding $theKeyRound, @$tobepaired;
# 
#   my @pair_list;
#   my $group_number = 0;
#   while (@ranked) {
#     $group_number++;
#     my (@group) = @ranked > $factor + $factor
#       ? splice(@ranked, 0, $factor)
#       : splice(@ranked, 0);
#     my (@group_list) = DoFactorGroup \@group, $repeats, $minbyes;
#     unless (@group_list) {
#       TSH::Utility::Error "Can't factor group #$group_number. Division is partially paired.\n";
#       last;
#       }
#     push(@pair_list, @group_list);
#     }
# 
#   # store pairings
#   {
#     my $board = 1;
#     while (@pair_list) {
#       my $gp = shift @pair_list;
#       # make sure previous board numbers are set
#       for my $pp (@$gp) {
# 	my $tp = $pp->{'etc'}{'board'};
# 	my $pairingsp = $pp->{'pairings'};
# 	if (!defined $tp) {
# 	  $pp->{'etc'}{'board'} = [ (0) x @$pairingsp ];
# 	  }
# 	elsif ($#$tp < $#$pairingsp) {
# 	  push(@{$pp->{'etc'}{'board'}}, (0) x $#$pairingsp - $#$tp);
# 	  }
#         }
#       if (@$gp == 3) {
# 	# TODO: this somewhat duplicates InitFontes and should use a table
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[1]{'id'}, 0);
# 	push(@{$gp->[1]{'pairings'}},
# 	  0,              $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[1]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, 0);
# 	push(@{$gp->[1]{'etc'}{'board'}}, 0, $board, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board, 0, $board);
# 	$board += 1;
# 	}
#       elsif (@$gp == 4) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[3]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[1]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board, $board+1, $board+1);
# 	$board += 2;
# 	}
#       elsif (@$gp == 5) {
# 	# This table is not the one used in InitFontes
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, 0);
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[4]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  0,              $gp->[1]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board,   $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board,  0);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board,   0,      $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, 0,       $board+1, $board+1);
# 	$board += 2;
#         }
#       elsif (@$gp == 6) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[5]{'id'}, $gp->[3]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[5]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[4]{'id'}, $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[1]{'id'}, $gp->[5]{'id'});
# 	push(@{$gp->[5]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[2]{'id'}, $gp->[4]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board+2, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board+2, $board, $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, $board+2, $board+1, $board+2);
# 	push(@{$gp->[5]{'etc'}{'board'}}, $board, $board+2, $board+2);
# 	$board += 3;
# 	}
#       else { die "Assertion failed"; }
#       }
#       my $p1 = shift @pair_list;
#       my $p2 = shift @pair_list;
#       push(@{$p1->{'pairings'}}, $p2->{'id'});
#       push(@{$p2->{'pairings'}}, $p1->{'id'});
#   } # store pairings
# 
#   print "Done.\n";
#   $dp->Dirty(1);
#   $gTournament->UpdateDivisions();
#   }

# # TODO: this could be more efficient, but was written live at NSC 2005
# sub DoFactorGroup ($$$) {
#   my $psp = shift; # must not modify contents
#   # 0 indicates no repeats allowed, ..., 3 means up to 3 repeats = 4 pairings
#   my $repeats = shift;
#   # TODO: allow for possibility that we have to increase minbytes after 1st plr
#   my $minbyes = shift;
# 
#   print "DFG: " . (1+$#$psp) . ' ' . join(',', map { $_->{'id'} } @$psp) . "\n";
#   if (@$psp == 4 || @$psp == 6) {
#     if (CheckGroupRepeats $psp, $repeats) {
# #     print "DFG: returning $#$psp+1\n";
#       return ([@$psp]);
#       }
#     else {
# #     print "DFG: returning failure\n";
#       return ();
#       }
#     }
#   elsif (@$psp == 3) {
#     for my $p (@$psp) {
#       if ($p->{'byes'} != $minbyes) {
# #	print "DFG: $p->{'name'} already has $p->{'byes'} bye(s).\n";
# 	return ();
#         }
#       }
#     return ([@$psp]);
#     }
#   elsif (@$psp == 5) {
#     my $possible_bye_players = 0;
#     for my $p (@$psp) {
#       if ($p->{'byes'} == $minbyes) {
# 	$possible_bye_players++;
#         }
#       }
#     if ($possible_bye_players < 3) {
#       for my $p (@$psp) {
#         print "DFG5: $p->{'name'} already has $p->{'byes'} bye(s).\n";
#         }
#       return ([
# 	sort { $b->{'byes'} <=> $a->{'byes'} } @$psp
#         ]);
#       }
#     }
#   elsif (@$psp < 7) {
#     die "DoFactorGroup: bad group size: " . scalar(@$psp) . "\n";
#     }
#   my $s = int(@$psp/4);
#   my $p1 = $psp->[0];
#   my $j1 = 0;
#   # first try to pair within quartiles
#   for my $j2 ($s..$s+$s-1) {
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $j3 ($s+$s..$s+$s+$s-1) {
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $j4 ($s+$s+$s..$s+$s+$s+$s-1) {
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
#   # then try to pair anywhere within the group
#   for my $i2 (0..$#$psp) {
#     my $j2 = ($i2 + $s) % @$psp;
#     next if $j2 == $j1;
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $i3 (0..$#$psp) {
#       my $j3 = ($i3 + $s + $s) % @$psp;
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $i4 (0..$#$psp) {
# 	my $j4 = ($i4 + $s + $s + $s) % @$psp;
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
# # print "DFG: returning failure.\n";
#   return ();
#   }

sub lint () {
  $config'table_format = '';
  $config'gibson = undef;
  %config::gibson_equivalent = ();
  %config::autopair = ();
# $config'reserved{''} = '';
# $config'tables{''} = '';
  lint;
  }

sub LockFailed ($) {
  my $reason = shift;
  print <<"EOF";
System call failed: $reason

You should not run more than one copy of tsh using the same 
configuration file at the same time.  tsh uses a "lock file" called
tsh.lock to keep track of when it is running.  This copy of tsh
was unable to get access to the lock file.  The most likely reason
for this is that tsh is already in use.
EOF
  exit 1;
  }

sub LockOff () {
  flock($global'lockfh, LOCK_UN)
    or die "Can't unlock tsh.lock - something is seriously wrong.\n";
  close($global'lockfh)
    or die "Can't close tsh.lock - something is seriously wrong.\n";
  }

sub LockOn () {
  my $error;

  $global'lockfh = new FileHandle "$config'root_directory/tsh.lock",
    O_CREAT | O_RDWR
    or die "Can't open tsh.lock - check to make sure tsh isn't already running.\n";
  flock($global'lockfh, LOCK_EX | LOCK_NB) 
    or LockFailed "flock: $!";
  seek($global'lockfh, 0, 0) 
    or die "Can't rewind tsh.lock - something is seriously wrong.\n";
  truncate($global'lockfh, 0) 
    or die "Can't truncate tsh.lock - something is seriously wrong.\n";
  print $global'lockfh "$$\n"
    or die "Can't update tsh.lock - something is seriously wrong.\n";
  } 

sub Main () {
  srand;


  ReopenConsole if $^O eq 'MacOS';
  my $config_fn = ChooseConfig;
  LockOn;
  $gTournament = new TSH::Tournament;
  $gTournament->TellUser('iwelcome', $gkVersion);
  ReadConfig $config_fn;
  mkdir $config'backup_directory, 0700 unless -d $config'backup_directory;
  ReadDivisions;
  $global'parser = new TSH::ParseArgs;
  if (defined $config::event_name) {
    $gTournament->TellUser('ievtname', $config::event_name);
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/) {
      print "\e]0;tsh $gkVersion - $config::event_name\a";
      }
    }
  Prompt;
  while (<>) {
    next unless /\S/;
    s/^\s+//;
    my(@argv) = split;
    if (!$gTournament->RunCommand(@argv)) 
      { print "Enter 'help' for help.\n"; }
    last if $gTournament->QuittingTime();
    }
  continue {
    Prompt;
    }
  LockOff;
  if (defined $config::event_name) {
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/) {
      print "\e]0;\a";
      }
    }
  }

sub Prompt () { 
  TSH::Utility::PrintColour 'yellow on blue', 'tsh>';
  print ' ';
  }

sub ReadConfig ($) {
  my $fn = shift;

  $config'table_format = '%3s';
  open(CONFIG, "<$config'root_directory/$fn") || die "Can't open $fn: $!\n";
  local($_);
  $gTournament->TellUser('iloadcfg', "$config::root_directory/$fn");
  $gNDivisions = 0;
  while (<CONFIG>) { s/#.*//; s/^\s*//; s/\s*$//; next unless /\S/;
    if (/^division?\s+(\S+)\s+(.*)/i) {
      # the new way
      my $dname = $1;
      my $dfile = $2;
      my $dp = new TSH::Division;
      $dp->Name($dname);
      $dp->File($dfile);
      $gTournament->AddDivision($dp);
      # the old way
#     $gDivision{lc $1} = 
#       bless {'file' => $2, 'name' => (uc $1)}, 'TSH::Division';
      $gDivision{lc $dname} = $dp;
      $gNDivisions++;
      }
    elsif (s/^perl\s+//i) { 
      eval $_;
      print "eval: $@\n" if length($@);
      }
    elsif (s/^config\s+//i) { 
      eval q($config') . $_;
      print "eval: $@\n" if length($@);
      }
    elsif (s/^autopair\s+//i) { 
      if (/^(\w+) (\d+) (\d+)\s+(\w+)\s+(.*)/) {
	my ($div, $sr, $round, $command, $args) = ($1, $2, $3, $4, $5);
	my (@args) = split(/\s+/, $args);
	my $commandp = $gTournament->GetCommandByName($command);
	if (isa $commandp, 'TSH::VCommand') {
	  $commandp = $commandp->Load();
	  }
	if ($command =~ /^(?:if|pair1324|p1324|koth|ns|newswiss|rr|roundrobin|cp)$/i || ($commandp && (isa $commandp, 'TSH::PairingCommand'))) {
	  $config::autopair{uc $div}[$round] = [$sr, $command, @args];
	  }
	else {
	  print "'$commandp'\n";
	  $gTournament->TellUser('ebadapc', $command);
	  exit(1);
	  }
        }
      else {
	chomp;
        $gTournament->TellUser('ebadap', $_, $fn);
	exit(1);
        }
      }
    else {
      $gTournament->TellUser('ebadcfg', $_);
      exit(1);
      }
    }
# print "Configuration file loaded.\n";
  if ($config::external_path) {
#   print "Loading external(s):";
    for $global'path (@$config::external_path) {
      my $config = "$global'path/tshxcfg.txt";
      if (-r "$config") {
	my $rv = do $config;
	if ($@) { print "Can't load externals [\@$@]"; }
	if ($!) { print "Can't load externals [!$!]"; }
	unless ($rv) { print "Can't load externals [X]"; }
        }
      }
#   print "\n";
    }
  for my $dir ($config'html_directory, $config'backup_directory) {
    if ($dir !~ /^(?:\/|[a-zA-Z]:[\/\\])/) 
      { $dir = "$config'root_directory/$dir"; }
    }
  unless (-d $config::html_directory) {
    mkdir $config::html_directory;
    my $fh;
    $fh = new FileHandle("lib/tsh.css", "<");
    local($/) = undef;
    my $css = <$fh>;
    $fh->close();
    $fh = new FileHandle("$config::html_directory/tsh.css", ">");
    print $fh $css;
    $fh->close();
    }
  mkdir $config::backup_directory unless -d $config::backup_directory;
  unless ($config::player_number_format) {
    $config'player_number_format = 
      $gTournament->CountDivisions() == 1 ? '#%s' : '%s';
    }
  $config'split1 = 1000000 unless $config'split1;
  for my $div (keys %config::gibson_groups) {
    my $divp = $config::gibson_groups{$div};
    for my $gibson_group (@$divp) {
      my $first = $gibson_group->[0];
      for my $i (1..$#$gibson_group) {
	$config::gibson_equivalent{$div}[$gibson_group->[$i]] = $first;
#	print "$gibson_group->[$i] equiv $first in $div.\n";
	}
      }
    }
  }

sub ReadDivision (\%) {
  my $dp = shift;
  my $fn = $dp->{'file'};
  $gTournament->TellUser('iloaddiv', $dp->{'name'});
  open(DIV, "<$config'root_directory/$fn") || die "Can't open $fn: $!\n";
  local($_);
  my (@data) = (undef);
  my $id = 1;
  my $name_length = 16;
  while (<DIV>) { s/#.*//; s/^\s*//; s/\s*$//; next unless /\S/;
    s/$/;/ unless /;/;
# TODO: use lib/perl/TFile.pm to do parsing here and elsewhere
# TODO: see if a cached binary file format would speed large file loads
# TODO: consider delaying parsing of some subfields until it's needed, or we're idle
    my($player, $rating, $pairings, $scores, $etc) 
      = /^([^;]+[^;\s\d])\s+(\d+)\s*([\d\s]*);\s*([-\d\s]*)((?:;[^;]*)*)$/;
    die "Can't parse: $_\n" unless defined $scores;
    $name_length = length($player) if $name_length < length($player);
    my(@pairings) = split(/\s+/, $pairings);
    my(@scores) = split(/\s+/, $scores);
    my $pp = { 
      'division' => $dp,
      'id'       => $id,
      'name'     => $player,
      'rating'   => $rating,
#     'rnd'=>rand,
      'rnd'      => ((length($player) * (100+$id) * ord($player)) % 641),
      'pairings' => [split(/\s+/, $pairings)],
      'scores'   => [split(/\s+/, $scores)],
      };
    bless $pp, 'TSH::Player'; # TODO: move this to new() if it doesn't slow things down too much
    for my $extra (split(/;\s*/, $etc)) {
      next unless $extra =~ /\S/;
      my ($tag, @words) = split(/\s+/, $extra);
      if (defined $pp->{$tag}) {
	warn "Overwriting $tag field for $player.\n";
        }
      $pp->{'etc'}{$tag} = \@words;
      }
    push(@data, $pp);
    $gTournament->RegisterPlayer($pp);
    $id++;
    }
  close(DIV);
  if ($name_length > $config'max_name_length) {
    $config'max_name_length = $name_length;
    $config'name_format = "%-${name_length}s";
    }

  TSH::Utility::Error "Warning: odd number of players in Division $dp->{'name'}.\n"
    if $#data % 2 == 1;
  $dp->{'data'} = \@data;
  $dp->Synch();
  }

sub ReadDivisions () {
  for my $div (sort keys %gDivision) {
    my $dp = $gDivision{$div};
    ReadDivision %$dp;
    } 
# print "All divisions loaded.\n";
  }

sub ReopenConsole () {
  close(STDOUT);
  close(STDERR);
  close(STDIN);
  open(STDOUT, "+>dev:console:tsh console") || die;
  open(STDERR, "+>dev:console:tsh console") || die;
  open(STDIN, "<dev:console:tsh console") || die;
  $| = 1;
  }

=item $boolean = ResolvePairings $unpairedp[, $just_checking]

Given a division and a list of unpaired players who have their
'pref' field set to a list of opponent preferences, find a reasonable
pairing of all the players.  Return success.  If C<$just_checking>
then run quietly and do not execute pairings, just return status.
In either case, set the 'opp' field of each player paired to the
opponent's ID.

Should be moved to TSH::Player::ResolvePairings.

=cut

sub ResolvePairings ($;$) {
  my $unpairedp = shift;
  my $just_checking = shift;

# { my $p; for $p (@$unpairedp) { print "# $p->{'name'}: @{$p->{'pref'}}\n"; } }
# print "# finding optimal pairing\n";
  # pruning dead branches saves us two orders of magnitude or so
  my %dead;

  my @sorted;
  # another (slight) speed optimization
  if (@$unpairedp > 12) {
    @sorted = @$unpairedp[sort {
      # prefer players with fewer choices
      @{$unpairedp->[$a]{'pref'}} <=> @{$unpairedp->[$b]{'pref'}} ||
      # ties broken according to input ordering
      $a <=> $b;
      } 0..$#$unpairedp
      ];
    }
  else { @sorted = @$unpairedp; }

  { # block for scope isolation only
    my(@choice, $opp, $oppid);

    # mark all players as initially unpaired
    # 'opp' points to the provisional opponent
    for my $p (@sorted) { 
      $p->{'opp'} = -1; 
      # check quickly to see if pairings are impossible
      unless (@{$p->{'pref'}}) {
#	TSH::Utility::Error "No candidate opponents for " . $p->{'name'};
	return 0;
        }
      }

    # find best opp for each player, favoring top of field
    for (my $i=0; $i<=$#sorted; ) {
      my $p = $sorted[$i];
      if ($p->{'opp'} >= 0)
        { $i++; next; } # player has already been paired - skip
      my $key = join('', grep { $_->{'opp'} < 0 } 
	@sorted[$i..$#sorted]);
      if ($dead{$key}) {
#	print "s\010";
#	print "Skipping known dead: $key.\n";
	# this code is duplicated below and should be merged 
	# when fully debugged
        for ($choice[$i]=undef; $i>=0 && !defined $choice[$i]; $i--) { }
# print "$i.\n";
        if ($i < 0) {
	  TSH::Utility::Error "Walked entire tree, couldn't find acceptable pairing.\n"
	    unless $just_checking;
          return 0;
          }

        # find last paired player's opponent, which now has to be unpaired
        my $opp = $sorted[$i]{'pref'}[$choice[$i]];
        # unpair opponent from that player
        $opp->{'opp'} = -1;
        # unpair that player from the opponent
        $sorted[$i]{'opp'} = -1;
        next;
        }

      # go to head of preference list if visiting player for first time
      $choice[$i] = -1 unless defined $choice[$i];

      # try the next preferred opp for this player.
      $opp = $p->{'pref'}[++$choice[$i]];

      if (!defined $opp) {
# print '.' x $i, "$p->{'name'} can't be paired, climbing back up from i=$i to i=";
#	print "Marking as dead: $key\n";
#	print "m\010";
	$dead{$key}++;
        for ($choice[$i]=undef; $i>=0 && !defined $choice[$i]; $i--) { }
# print "$i.\n";
        if ($i < 0) {
	  TSH::Utility::Error "Walked entire tree, couldn't find acceptable pairing.\n"
	    unless $just_checking;
          return 0;
          }

        # find last paired player's opponent, which now has to be unpaired
        my $opp = $sorted[$i]{'pref'}[$choice[$i]];
        # unpair opponent from that player
        $opp->{'opp'} = -1;
        # unpair that player from the opponent
        $sorted[$i]{'opp'} = -1;
        next;
        } # if (!defined $opp) - we've run out of opps, back up

# print '.' x $i, "$p->{'name'} has pairing vector @{$p->{'pref'}}.\n";
# print ' ' x $i, " trying to pair $p->{'name'}, choice $choice[$i] is ",
# defined $opp ? "$opp->{'name'} ($opp->{'id'})" : 'undef', "\n";

      if ($opp->{'opp'} >= 0) {
# print ' ' x $i, " but $opp->{'name'} has already been paired.\n";
        next;
        }

      # looks good so far, let's try to keep going
      $p->{'opp'} = $opp->{'id'};
      $opp->{'opp'} = $p->{'id'};
      $i++;
      } # for $i
    }
  # copy provisional opponents to pairings
  unless ($just_checking) {
#   my $board = 1;
    my $r0 = $#{$sorted[0]{'pairings'}};
    for my $i (0..$#sorted) {
      my $p = $sorted[$i];
      push(@{$p->{'pairings'}}, $p->{'opp'});
# The following code (which assigns players to boards) was removed
# because ResolvePairings is not always called for the complete set 
# of players, and board numbers were being doubled.
#     my $oid = $p->{'opp'};
#     next unless $oid;
#     my $opp = $p->{'division'}{'data'}[$oid];
#     die "assertion failed" unless $opp;
#     my $ptp = $p->{'etc'}{'board'};
#     if (!defined $ptp) { $p->{'etc'}{'board'} = $ptp = []; }
#     push(@$ptp, (0) x ($r0-$#$ptp)) if $r0 > $#$ptp;
#     my $otp = $opp->{'etc'}{'board'};
#     if (!defined $otp) { $opp->{'etc'}{'board'} = $otp = []; }
#     push(@$otp, (0) x ($r0-$#$otp)) if $r0 > $#$otp;
#     next unless (
#	$p->{'wins'} <=> $opp->{'wins'} ||
#	$p->{'spread'} <=> $opp->{'spread'} ||
#	$opp->{'id'} <=> $p->{'id'}) > 0;
#     push(@$ptp, $board);
#     push(@$otp, $board++);
      }
    }
  1;
  } # sub ResolvePairings

=back

=cut

=head1 BUGS

Here is the list of planned improvements to C<tsh>, in roughly
descending order of priority.

=over 4

=item *
 default pairings should do IF for first three rounds of a tournament
 lasting at least six rounds; RR if number of players is <=
 number of remaining rounds - 1 (and last round is not partially paired)

=item *
 MANual and WEB commands to open HTML files; config browser

=item *
 team match support (separate module, commands, config file)

=item *
 Fontes Gibsonization possibility warnings 

=item *
 allow "surname,given" wherever parser expects pn

=item *
 do not allow pairing commands to exceed max_rounds

=item *
 webupdater should be configurable and documented

=item *
 default values for command parameters?

=item *
 prompt for and load or create a config.tsh file if none specified

=item *
 ratings submission command (external)

=item *
 return values from externals

=item *
 some configuration variables to externals

=item *
 a command to unpair selective pairings (maybe a 1-arg form of pair)

=item *
 division data complete message and trigger

=item *
 bios.txt interface, photos on pairings

=item *
 virtual scorecards on web

=item *
 printing support

=item *
 add more internal cross-references in the documentation

=item *
 proofread documentation for typographic style

=item *
 load large divisions in separate threads

=item *
 a report listing last lines of scorecards for all players, 
 so that players can check their results

=item *
 ResolvePairings is currently the critical sub, and should be optimized,
 or rewritten in C.

=item *
 FindPlayer command.

=item *
 choose random seed for firsts/seconds in a way that can't be jiggered
 by a director

=item *
 should not create a ratings-0.html file

=item *
 optionally run a html/http gui in a separate thread

=item *
 touch the config.tsh file after opening it so that it's the new default

=item *
 detect malformed config.tsh file due to bad line breaks, report

=back

=cut

END { 
  sleep 10 if $^O eq 'MSWin32'; # to prevent error messages from disappearing
  }

## main code
Main;

