#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TFile;

=pod

=head1 NAME

TFile - manipulate John Chew's Scrabble tournament .t files 

=head1 SYNOPSIS

  my $tf = new TFile 'a.t';
  while (my $datap = $tf->ReadLine()) {
    print "$datap->{'name'}\n";
    }
  $tf->Close();

  
=head1 ABSTRACT

This Perl module is used to read tournament data files in '.t' file format.

=head1 DESCRIPTION

=over 4

=cut

use strict;
use warnings;
use Symbol;

sub new ($$);
sub Close($);
sub FormatLine($);
sub ParseLine($);
sub ReadLine($);

=item $tf = new TFile($filename);

Create a new TFile object.  
Opens file, returns undef on failure.

=cut

sub new ($$) {
  my $proto = shift;
  my $filename = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $fh = gensym;
  my $this = {
    'filename' => $filename,
    'id' => 0,
    };
  return undef unless open($fh, "<$filename");
  $this->{'handle'} = $fh;
  bless($this, $class);
  return $this;
  }

=item $success = $fh->Close();

Explicitly closes the .t file.

=cut

sub Close ($) {
  my $this = shift;
  
  close($this->{'handle'});
  delete $this->{'handle'};
  return 1;
  }

=item $line = FormatLine($datap);

Recreate a formatted $line from its parsed data.

=cut

sub FormatLine ($) {
  my $p = shift;
  my $s = sprintf("%-22s %4d %s; %s",
    $p->{'name'},
    $p->{'rating'},
    join(' ', @{$p->{'pairings'}}),
    join(' ', @{$p->{'scores'}}));
  if ($p->{'etc'}) {
    my $etcp = $p->{'etc'};
    for my $key (sort keys %$etcp) {
      if (my $wordsp = $etcp->{$key}) {
	$s .= "; $key @$wordsp";
        }
      }
    }
  $s .= "\n";
  return $s;
  }

=item $datap = $fh->ReadLine();

Read and parse one line from the file.
Returns a hash whose keys are C<id>, C<name>, C<rating>, C<pairings>,
C<scores>, C<rnd> and C<etc>.

=cut

sub ReadLine ($) {
  my $this = shift;
  my %data;

  my $fh = $this->{'handle'};
  while (1) {
    local($_);
    $_ = <$fh>;
    return undef unless defined $_;
    s/#.*//; 
    s/^\s*//;
    s/\s*$//; 
    next unless /\S/;
    s/$/;/ unless /;/;
    if (my $datap = ParseLine($_)) {
      my $id = ++($this->{'id'});
      $datap->{'id'} = $id;
      $datap->{'rnd'} = ((length($_) * (100+$id) * ord($_)) % 641),
      return $datap;
      }
    else {
      die "Can't parse in $this->{'filename'}: $_\n";
      }
    }
  }

=item $datap = ParseLine($line) 

Parse a received input line into a data structure.

=cut

sub ParseLine ($) {
  my $s = shift;
  my($player, $rating, $pairings, $scores, $etc) 
    = $s =~ /^([^;]+[^;\s\d])\s+(\d+)\s*([\d\s]*);\s*([-\d\s]*)((?:;[^;]*)*)$/;
  return undef unless defined $scores;
  my (@pairings) = split(/\s+/, $pairings);
  my (@scores) = split(/\s+/, $scores);
  my %data = (
#     'division' => $dp,
    'name'     => $player,
    'rating'   => $rating,
#     'rnd'=>rand,
    'pairings' => [split(/\s+/, $pairings)],
    'scores'   => [split(/\s+/, $scores)],
    );
  for my $extra (split(/;\s*/, $etc)) {
    next unless $extra =~ /\S/;
    my ($tag, @words) = split(/\s+/, $extra);
    if (exists $data{$tag}) {
      die "Duplicate $tag field for $player.\n";
      }
    $data{'etc'}{$tag} = \@words;
    }
  return \%data;
  }

=back

=cut

1;
