#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::MANual;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::MANual - implement the C<tsh> MANual command

=head1 SYNOPSIS

  my $command = new TSH::Command::MANual;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::MANual is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to view the tsh reference manual.
At present this command only works with OS/X.
If someone can tell me (John Chew) how to provide the same
functionality for other operating systems, I'll be happy to provide it.
EOF
  $this->{'names'} = [qw(man manual)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $topic = shift;
  if ($topic !~ /^index$/i) {
    my $command = $tournament->GetCommandByName($topic);
    unless ($command) {
      $tournament->TellUser('enohelp', $topic);
      return;
      }
    print $command->Usage();
    print "\n";
    print TSH::Utility::Wrap(0, $command->Help());
    return;
    }

  print TSH::Utility::Wrap(0, <<'EOS');
Detailed online help is available for the following commands
(type "help" followed by one of these command names):
EOS
  my (@commands) = $tournament->CommandNames();
  print TSH::Utility::Wrap(2, @commands);
  }

=back

=cut

1;
