#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package HTTP::Message;

=pod

=head1 Name

HTTPMessage -  HTTP Message Class

=head1 SYNOPSIS

  $mp = new HTTP::Message ('handle' => $this->{'socket'});
  $te = $mp->MessageHeader('transfer-encoding');
  $body = $mp->Body();


=head1 ABSTRACT

This Perl library is an abstraction of an HTTP message.  HTTP messages
are used to communicate between a web client and web server.  

=head1 DESCRIPTION

=over 4

=cut

sub new ($@);
sub Body ($);
sub Die ($$);
sub Error ($;$);
sub GetFromHandle ($$);
sub MessageHeader ($$;$);
sub ParseHeader ($$);
sub StatusCode ($;$);

=item $mp = new HTTPMessage($key1 => $value1, ...);

Create a new HTTPMessage object.  
All parameter keys are optional.
C<body> gives the message body as a string.
C<error> sets the current module error message.
C<handle> gives a handle from which to read the message.
C<message-headers> gives the message headers as a hash.

=cut

sub new ($@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $this = {
    'body' => '',
    'error' => '',
    'message-headers' => {},
    };
  my %param = @_;
  for my $key (qw(body message-headers)) {
    if (exists $param{$key}) { $this->{$key} = $param{$key}; }
    elsif (!exists $this->{$key}) {
      die "Missing required parameter ($key).";
      }
    }
  bless($this, $class);

  if ($param{'handle'}) {
    $this->GetFromHandle($param{'handle'});
    }
  
  return $this;
  }

=item $body = $mp->Body();

Returns the body of an HTTP message.

=cut

sub Body ($) { my $this = shift; return $this->{'body'}; }

=item $mp->Die($error);

Sets the module error message and returns undef.
Could be overridden to invoke the builtin C<die>.

=cut

sub Die ($$) {
  my $this = shift;
  my $message = shift;

  $this->{'error'} = $message;
  return undef;
  }

=item $mp->Error($error);

=item $error = $mp->Error();

Sets and/or gets the module error message.

=cut

sub Error ($;$) {
  my $this = shift;
  my $message = shift;
  if (defined $message) 
    { return $this->{'error'} = $message; }
  else 
    { return $this->{'error'}; }
  }

=item $mp = $mp->GetFromHandle($fh);

Reads a message from a file handle using the C<sysread()>.
Sets error on error.

=cut

sub GetFromHandle ($$) {
  my $this = shift;
  my $handle = shift;

  my $buffer = '';
  my $count;
  my $rv;

  while ($buffer !~ /\015\012\015\012/) {
    $rv = sysread($handle, $buffer, 1024, length($buffer));
    next unless defined $rv; # likely no data available yet
    last unless $rv; # EOF
    }
  # parse header
  $this->ParseHeader($buffer)
    or $this->Die("Web server returned malformed header") or return $this;
# print substr($buffer, 0, $this->{'header-size'});
  if ($this->{'status-code'} != 200) {
    $this->Die("Web server request failed: $this->{'status-code'} $this->{'reason-phrase'}");
    return $this;
    }
  # remove header from buffer
  substr($buffer, 0, $this->{'header-size'}) = '';
  # if the message body length is explicitly specified
  if (my $content_length = $this->MessageHeader('content-length')) {
#   warn "content-length: $content_length\n";
    while (length($buffer) < $content_length) {
      $rv = sysread($handle, $buffer, 
	$content_length-length($buffer), length($buffer));
      next unless defined $rv; # likely no data available yet
      last unless $rv; # EOF
      }
    if (length($buffer) < $content_length) {
      $this->Die("Web server response was truncated");
      return $this;
      }
    $this->{'body'} = $buffer;
    }
  # else if a transfer encoding is given
  elsif (my $transfer_encoding = $this->MessageHeader('transfer-encoding')) {
#   warn "transfer-encoding: $transfer_encoding\n";
    if ($transfer_encoding eq 'chunked') {
      my $body = '';
      while (1) {
	# TODO: process transfer encoding extensions
#	print substr($buffer, 0, 80);
	my $chunk_size;
	while (1) {
          if ($buffer =~ s/^([0-9a-f]+)[^\015\012]*\015\012//i) {
	    $chunk_size = $1;
	    last;
	    }
#while ($buffer !~ s/^([0-9a-f]+)[^\015\012]*\015\012//i) {
#	  print "going back for more.\n";
#	  print "buffer: $buffer\n";
#	  sleep 1;
	  $rv = sysread($handle, $buffer, 1024, length($buffer));
	  next unless defined $rv; # likely no data available yet
	  last unless $rv; # EOF
	  }
	if (length($buffer) && !defined $chunk_size) {
	  $this->Die("Web server response was badly chunked\n$buffer");
	  return $this;
	  }
#	print "chunk_size: $chunk_size\n";
	last unless $chunk_size;
	$chunk_size = hex($chunk_size);
	while (length($buffer) < $chunk_size + 2) {
	  $rv = sysread($handle, $buffer, 
	    $chunk_size + 2 - length($buffer), length($buffer));
	  next unless defined $rv; # likely no data available yet
	  last unless $rv; # EOF
	  }
#        print "chunk: $buffer\n";
	$body .= substr($buffer, 0, $chunk_size);
	if (length($buffer) < $chunk_size + 2) {
	  $this->Die("Web server response was truncated");
	  return $this;
	  }
	else {
	  substr($buffer, 0, $chunk_size + 2) = '';
	  }
	}
      $this->{'body'} = $body;
      }
    else {
      $this->Die("Unsupported transfer encoding: $transfer_encoding");
      return $this;
      }
    }
  # no message body will be present
  else {
#   warn "no body\n";
    $this->{'body'} = '';
    }
  return $this;
  }

=item $mp->MessageHeader($field_name, $field_value);

=item $field_value = $mp->MessageHeader($field_name);

Sets and/or gets an individual message header.

=cut

sub MessageHeader ($$;$) {
  my $this = shift;
  my $field_name = shift;
  my $field_value = shift;
  if (defined $field_value) 
    { return $this->{'message-headers'}{$field_name} = $field_value; }
  else 
    { return $this->{'message-headers'}{$field_name}; }
  }

=item $mp->ParseHeader($header)

Intended for internal use, this method parses a message header
and updates internal variables.

=cut

# intended for internal use
sub ParseHeader ($$) {
  my $this = shift;
  my $header = shift;

  $header =~ s/^(?:\015\012)+//;
  my $header_size = index $header, "\015\012\015\012";
  if ($header_size > 0) { $this->{'header-size'} = $header_size + 4; }
  else 
    { $this->Die("Web server did not return a complete header"); return undef; }
  my ($start_line, @message_headers) 
    = split(/\015\012/, substr($header, 0, $header_size));
  if ($start_line =~ /^HTTP\S+\s+(\d\d\d)\s+([^\015\012]*)/) {
    $this->{'status-code'} = $1;
    $this->{'reason-phrase'} = $2;
    }
  else { 
    $this->Die("Web server returned a bad start line: $start_line");
    return undef; 
    }

  for my $message_header (@message_headers) {
    my ($field_name, $field_value) = split(/:\s*/, $message_header, 2);
    $field_name = lc $field_name;
    if (defined $field_value) {
      if (exists $this->{'message-headers'}{$field_name}) {
	$this->{'message-headers'}{$field_name} .= ',' . $field_value;
        }
      else {
	$this->{'message-headers'}{$field_name} = $field_value;
        }
      }
    else {
      $this->Die("Web server returned a bad message header: $header"); 
      return undef; 
      }
    }
  return $this;
  }

=item $mp->StatusCode($status_code);

=item $status_code = $mp->StatusCode();

Sets and/or gets the message's status code.

=cut

sub StatusCode ($;$) {
  my $this = shift;
  my $message = shift;
  if (defined $message) 
    { return $this->{'status-code'} = $message; }
  else 
    { return $this->{'status-code'}; }
  }

=back

=head1 SEE ALSO

The current HTTP standard at www.w3.org.

=cut

1;

