#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundRobin;

use strict;
use warnings;

use TSH::Utility;
use TSH::Division::Pairing::Clark;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RoundRobin - implement the C<tsh> RoundRobin command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundRobin;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundRobin is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the RoundRobin command to manually add a set of full round robin pairings
to a division.  When you use this command, the last paired round must
be fully, not partially paired.
EOF
  $this->{'names'} = [qw(rr roundrobin)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($dp) = @_;

  my $datap = $dp->{'data'};
  my $nplayers = $#$datap;

  # check that division is not partially paired
  my $round0 = $dp->FirstUnpairedRound0();
  if ($round0 != $dp->LastPairedRound0()+1) {
    $tournament->TellUser('errpartp');
    return;
    }
# TSH::Player::SpliceInactive @$unpaired, $nplayers-1;

  print "Calculating round robin pairings for Division $dp->{'name'}.\n";
  my $psp = $dp->GetUnpairedRound($round0);
  # add pairings information one round at a time
  my $schedule_size = @$psp + (@$psp % 2);
  for (my $oppi = $schedule_size; $oppi > 1; $oppi--) {
    TSH::Division::Pairing::Clark::PairGroup 
      $dp->GetUnpairedRound($round0++), $oppi; 
    }
  $tournament->TellUser('idone');

  $dp->Dirty(1);
  $dp->Update();
  }

=back

=cut

=head1 BUGS

None known.

=cut

1;
