#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package HTTP;

=pod

=head1 NAME

HTTP - HTTP Class

=head1 SYNOPSIS

  my $http = new HTTP;
  $http->Connect($server) or die;
  my ($boundary, $entity_body) = HTTP::EncodeMultipartFormData (
    'plain_field_name' => $field_value,
    'file_field_name' => { 'type' => 'file',
      'filename' => $filename,
      'data' => $data,
      },
    );
  $http->PutRaw(''
    ."POST $path HTTP/1.1\015\012"
    ."Host: $server\015\012"
    ."Accept-Charset: iso-8859-1\015\012"
    ."Accept-Encoding:\015\012"
    ."User-Agent: poslfit\015\012"
    ."Content-Length: " . length($entity_body) . "\015\012"
    ."Content-Type: multipart/form-data; boundary=$boundary\015\012"
    ."\015\012"
    . $entity_body
    );
  my $response = $http->GetResponse();
  $http->Close();
  if (my $error = $response->Error()) {
    print "$error\n";
    }
  my $html = $response->Body();
  
=head1 ABSTRACT

This Perl library is an interface class for the HyperText Transfer
Protocol, usually used to communicate between a web client and web server.  

=head1 DESCRIPTION

=over 4

=cut

use Fcntl qw(F_GETFL F_SETFL O_NONBLOCK);
use Socket;
use Symbol;

use HTTP::Message;

sub new ($@);
sub Close($);
sub Connect ($$);
sub Die ($$);
sub EncodeMultipartFormData (@);
sub GetResponse($);
sub PutRaw ($$);

=item $hp = new HTTP($key1 => $value1, ...);

Create a new HTTP object.  
All parameter keys are optional.
C<exit_on_error> gives a boolean value specifying whether or not to
exit on any error.

=cut

sub new ($@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $this = {
    'state' => 'init',
    'exit_on_error' => 1,
    };
  my %param = @_;
  for my $key (qw(exit_on_error)) {
    if (exists $param{$key}) { $this->{$key} = $param{$key}; }
    elsif (!exists $this->{$key}) {
      die "Missing required parameter ($key).";
      }
    }
  
  bless($this, $class);
  return $this;
  }

=item $success = $mp->Close();

Closes the HTTP connection, which cannot be used again until it is
reopened.  
Fails only if the connection was not open to begin with.

=cut

sub Close ($) {
  my $this = shift;
  
  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'open') 
    { $this->Die("Connection is not open"); return undef; }
  $this->{'state'} = 'init';
  return 1;
  }

=item $success = $hp->Connect($hostname);

Opens a connection to the named host.

=cut

sub Connect ($$) {
  my $this = shift;
  my $hostname = shift;

  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'init') 
    { $this->Die("Connection is already open"); return undef; }

  my $serveraddress = gethostbyname($hostname)
    or $this->Die("Can't find $hostname: $!");
  my $sfh = $this->{'socket'} = gensym;
  socket($sfh, PF_INET, SOCK_STREAM, getprotobyname('tcp')) 
    or $this->Die("Couldn't create socket for web connection: $!");
  # bind($sfh, sockaddr_in(0, "\0\0\0\0")) 
  #   or $this->Die("Couldn't bind socket for web connection: $!");
  connect($sfh, sockaddr_in(80, $serveraddress)) 
    or $this->Die("Couldn't connect to web server: $!");
  {
    my $flags;
    $flags = fcntl($sfh, F_GETFL, 0)
      or $this->Die("Can't get socket flags for web connection: $!");
    fcntl($sfh, F_SETFL, $flags | O_NONBLOCK) 
      or $this->Die("Can't set socket flags for web connection: $!");
  }
  $this->{'state'} = 'open';
  return 1;
  }

=item $hp->Die($message);

Sets the module error message and exits if C<exit_on_error> was set.

=cut

sub Die ($$) { 
  my $this = shift;
  my $message = shift;
  $this->{'error'} = $message;
  $this->{'state'} = 'error';
  print STDERR $message;
  exit 1 if $this->{'exit_on_error'};
  }

=item ($boundary, $entity_body) = $hp->EncodeMultipartFormData($key1 => $value1, ...);

Encodes a hash as multipart/form-data.
Used in creating a message to POST to a web server,
using the contents of a form that includes a file upload.

=cut

sub EncodeMultipartFormData (@) {
  my @data;
  my (%data) = @_;
  my $boundary = 'posl--';
  while (my ($key, $value) = each %data) {
    if (ref($value) eq 'HASH') {
      if ($value->{'type'} eq 'file') {
	push(@data, 
	  qq(Content-Disposition: form-data; name="$key"; filename="$value->{'filename'}\015\012\015\012)
	  . "$value->{'data'}\015\012");
        }
      else {
	die "Unknown form data type: $value->{'type'}\n";
        }
      }
    else {
      while ($value =~ /--$boundary/) {
	$boundary .= chr(97 + int(rand(26)));
        }
      push(@data,
	qq(Content-Disposition: form-data; name="$key"\015\012\015\012)
	. "$value\015\012");
      }
    }
  return ($boundary, join("--$boundary\015\012", '', @data, ''));
  }

=item $message = $mp->GetResponse();

Reads an HTTP message (as a C<HTTP::Message>) from an open 
connection, returns undef or dies on failure.

=cut

sub GetResponse ($) {
  my $this = shift;

  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'open') 
    { $this->Die("Connection is not open"); return undef; }

  return HTTP::Message->new('handle' => $this->{'socket'});
  }

=item $success = $mp->PutRaw($request);

Writes text to an open connection.

=cut

sub PutRaw ($$) {
  my $this = shift;
  my $request = shift;

  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'open') 
    { $this->Die("Connection is not open"); return undef; }

  my $old = select($this->{'socket'});
  $| = 1;
  print $request;
  select $old;
# print "[begin request]\n$request\n[end request]\n";

  return 1;
  }

=back

=cut

1;
