#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package HTTP::Server;

use strict;
use warnings;

use Socket;
use Symbol;

=pod

=head1 NAME

HTTP::Server - HTTP Server Class

=head1 SYNOPSIS

  my $server = new HTTP::Server({port => 7777, queue => 5, content => 
    sub { my $url = shift; return { 'code' => 200, 'body' => "No such URL: $url" }}
    ) or die;
  $server->Start() or die;
  while ($server->Run()) {}
  $server->Stop();
  
=head1 ABSTRACT

This Perl module implements a minimal HTTP server.

=head1 DESCRIPTION

=over 4

=cut

sub new ($@);

=item $server = new HTTP::Server({$key1 => $value1, ...});

Create a new HTTP::Server object.  
All keys are optional.

C<port>: port to listen on for new connections, defaults to 7777.

C<queue>: length of listener connection queue, defaults to 5.

C<content>: reference to sub that generates content, defaults to generating
an error message complaining of the lack of such a sub.

=cut

sub new ($@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $this = {
    'content' => sub { 'code' => 500, 'body' => 'No content code has been configured.' },
    'port' => 7777,
    'queue' => 5,
    };
  my %param = @_;
  for my $key (qw(content port queue)) {
    if (exists $param{$key}) { $this->{$key} = $param{$key}; }
    elsif (!exists $this->{$key}) {
      die "Missing required parameter ($key).";
      }
    }
  bless($this, $class);
  return $this;
  }

=item while ($server->Run()) { }

Checks to see if an HTTP request has been received.
If one has, it is processed.

=cut

sub Run ($) {
  my $this = shift;
  return 1;
  }

=item $server->Start() or die;

Prepares the server to accept new connections, returns boolean indicating success.

=cut

sub Start ($) {
  my $this = shift;
  my $sh = gensym;
  unless (socket($sh, PF_INET, SOCK_STREAM, getprotobyname('tcp'))
    or ;
  setsockopt($sh, SOL_SOCKET, SO_REUSE_ADDR, pack('l',1))
    or die "
  return 1;
  }

=item $server->Stop();

Shuts down the server and stops it from accepting new connections.

=cut

sub Stop ($) {
  my $this = shift;
  }

=back

=cut

1;
