#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Division;

use strict;
use warnings;

use Carp;
use Symbol;
use TSH::Player;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('RSw');

=pod

=head1 NAME

TSH::Division - abstraction of a Scrabble tournament division within C<tsh>

=head1 SYNOPSIS

  $d = new Division;
  $s = $d->Name();
  print "Yes.\n" if $s eq TSH::Division::CanonicaliseName('whatever');
  $d->Name($s);
  $d->Dirty(1); 
  print "Division has unsaved data.\n" if $d->Dirty();
  $s = $d->Tournament();
  $d->Tournament($s);

  $n = $d->CheckGibson($sr0, $round0);
  $p = $d->ChooseBye($sr0, $round0, \@psp);
  $n = $d->CountByes();
  $psp = $d->GetRegularUnpaired($sr0, $nobyes);
  print $d->FormatPairing($round0, $pn1, $style);
  $d->Pair($pn1, $pn2, $round0);
  $d->PairSwiss($repeats, $round0);
  PairSomeSwiss($psp, $repeats, $sr0);

  $d->DeleteScores($pn1, $pn2, $round0);
  $d->DeleteByeScore($pn1, $round0);
  $s = $d->LeastScores();
  $p = $d->LeastScoresPlayer();
  $s = $d->MostScores();
  $p = $d->MostScoresPlayer();

  $pp = $d->Player($pn);
  @pp = $d->Players;
  $round0 = $d->LastPairedRound0();

  $d->ComputeRanks($sr0);
  # $p->RoundRank($sr0); # see TSH::Player.pm

  $d->Update(); # do both of the following
  $d->Synch(); # update internal statistics
  $d->Write(); # save to disk

=head1 ABSTRACT

This Perl module is used to manipulate divisions within C<tsh>.

=head1 DESCRIPTION

A Division has (at least) the following member fields, none of which
ought to be accessed directly from outside the class.

  data        array of player data, 1-based indexing
  file        filename of .t file
  minp        least number of rounds of pairing data for any player (0-based)
  maxp        highest round number that has pairings data (0-based)
  mins        least number of scores registered for any player (0-based)
  mins_player a pointer to a player who has no scores past round mins
  maxs        highest round number that has score data (0-based)
  maxs_player a pointer to a player who has a score in round maxs
  name        division name
  tournament  a pointer to the division's associated tournament

The following member functions are currently defined.

=over 4

=cut

sub CanonicaliseName ($);
sub CheckRoundHasResults ($$);
sub CheckGibson ($$$);
sub ComputeRanks ($$);
sub CountByes ($);
sub CountPlayers ($);
sub DeleteByeScore ($$$);
sub DeleteScores ($$$$);
sub Dirty ($;$);
sub File ($;$);
sub FirstUnpairedRound ($);
sub FormatPairing ($$$;$);
sub GetRegularUnpaired ($$;$);
sub GetUnpaired ($;$);
sub GetUnpairedRound ($$);
sub initialise ($);
sub LastPairedRound0 ($);
sub LastPairedScoreRound0 ($);
sub LastPairedScorePlayer ($);
sub LeastScores ($);
sub LeastScoresPlayer ($);
sub MostScores ($);
sub MostScoresPlayer ($);
sub Name ($;$);
sub new ($);
sub Pair ($$$$;$);
sub PairSomeSwiss($$$);
sub PairSwiss ($$$);
sub Player ($$);
sub Players ($);
sub RecursiveSwiss ($$);
sub RecursiveSwissBottom ($$);
sub RecursiveSwissOne ($$);
sub RecursiveSwissTop ($$);
sub Synch ($);
sub Tournament ($;$);
sub Update ($);
sub Write ($);

=item $s1 = CanonicaliseName($s)

Canonicalise a division name

=cut

sub CanonicaliseName ($) {
  my $s = shift;
  $s =~ s/\W//g;
  $s = uc $s;
  return $s;
  }

=item $n = CheckGibson($dp, $sr0, $round0)

Returns the number of players in division $dp who are unpaired in
round $round0 and must be Gibsonized as of their standing in
round $sr0.
Does not check for Gibsonization on spread, and if $sr0 is not 
$round0-1 will miss some Gibsonizations. 
See TSH::Command::ChewPair for a better implementation.

=cut

sub CheckGibson ($$$) {
  my $dp = shift;
  my $sr0 = shift;
  my $round0 = shift;

  unless ($config'max_rounds) {
    TSH::Utility::Error "Can't do Gibson check without 'config max_rounds = ?'.\n";
    return -1;
    }
  my (@sorted) = TSH::Player::SortByStanding $sr0, @{$dp->GetUnpairedRound($round0)};
  # TODO: handle triple Gibsons

  # Note that this does not catch all Gibson situations.  In particular,
  # if you are using Fontes pairings, players may complain that a Gibson
  # situation in Round N has arisen as a result of a Round N-1 game,
  # when Round N pairings have been computed based on Round N-2 standings
  my $rounds_left = $config'max_rounds - ($sr0+1);

  my (@spread, @wins);
  for my $i (0..2) {
    my $pp = $sorted[$i];
    $spread[$i] = defined $pp->{'rspread'}[$sr0] ?
      $pp->{'rspread'}[$sr0] : $pp->{'spread'};
    $wins[$i] = defined $pp->{'rwins'}[$sr0] ?
      $pp->{'rwins'}[$sr0] : $pp->{'wins'};
    }

  # Note that we do not yet support Gibsoning on spread.
  if ($wins[0] - $wins[1] > $rounds_left) {
    printf "%s (%d %+d) needs to be Gibsonized with respect to %s (%d %+d).\n",
      (TSH::Utility::TaggedName $sorted[0]), $wins[0], $spread[0],
      (TSH::Utility::TaggedName $sorted[1]), $wins[1], $spread[1],
      ;
    return 1;
    }
  elsif ($wins[1] - $wins[2] > $rounds_left) {
    printf "%s (%d %+d) and %s (%d %+d) need to be Gibsonized with respect to %s (%d %+d).\n",
      (TSH::Utility::TaggedName $sorted[0]), $wins[0], $spread[0],
      (TSH::Utility::TaggedName $sorted[1]), $wins[1], $spread[1],
      (TSH::Utility::TaggedName $sorted[2]), $wins[2], $spread[2],
      ;
    return 2;
    }
  return 0;
  }

=item $boolean = $dp->CheckRoundHasResults($round0):

Return true if the assertion that division has at least some results in
zero-based round $round0 is true.

=cut

sub CheckRoundHasResults ($$) {
  my $this = shift;
  my $sr0 = shift;
  if ($sr0 > $this->{'maxs'}) {
    my $sr = $sr0+1;
    TSH::Utility::Error "You don't have round $sr results yet.\n";
    return 0;
    }
  return 1;
  }

=item $p = $d->ChooseBye($sr0, $round0, \@psp)

Assign a bye in $round0 to the player who was lowest ranked in round
$sr0 among those players in @psp who had the fewest byes.
Splice player from @psp, return player (or undef if no byes
because @psp is even).

=cut

sub ChooseBye($$$$) {
  my $this = shift;
  my $sr0 = shift;
  my $round0 = shift;
  my $psp = shift;

  return undef unless @$psp % 2;

  my $minbyes = $this->CountByes();
  my $p = (TSH::Player::SortByStanding $sr0, grep { $_->{'byes'} == $minbyes } 
    @$psp)[-1] or die "Assertion failed";
  # only assign the bye pairing, don't register the +50, as some routines
  # (and operators) may get confused by having early score data present
  my $pid = $p->ID();
  $this->Pair(0, $pid, $round0);
  # TODO: test to see how slow this is, fix if necessary
  my $found = 0;
  for my $i (0..$#$psp) {
    if ($psp->[$i]->ID() eq $pid) {
      splice(@$psp, $i, 1);
      $found = 1;
      last;
      }
    }
  die "Assertion failed" unless $found;
  print "Gave a bye to ", $p->Name(), ".\n";
  # We have to call Synch the division to update maxp
  # and we do have to Write it out because later parsing might fail
  $this->Dirty(1);
  $this->Update();
  }

=item $d->ComputeRanks($sr0);

Compute rankings of players as of zero-based round $sr0,
which should be a fully scored round.
Computed rankings may be obtained using Player::RoundRank().

=cut

sub ComputeRanks ($$) {
  my $this = shift;
  my $sr0 = shift;

  my (@sorted) = TSH::Player::SortByStanding $sr0, $this->Players();
  TSH::Player::SpliceInactive @sorted, 1, $sr0;
  my $lastw = -1;
  my $lasts = 0;
  my $rank = 0;
  for my $i (0..$#sorted) {
    my $p = $sorted[$i];
    my $wins = $p->RoundWins($sr0);
    my $spread = $p->RoundSpread($sr0);
    if ($wins != $lastw || $spread != $lasts) {
      $lastw = $wins;
      $lasts = $spread;
      $rank = $i+1;
      }
    $p->RoundRank($sr0, $rank);
    }
  }

=item $count = $d->CountByes();

Counts how many byes each player has had, returns least number.

=cut

sub CountByes ($) {
  my $dp = shift;

  my $datap = $dp->{'data'};
  my $minbyes = 9999999;

  for my $p (@$datap[1..$#$datap]) {
    next unless defined $p;
    my $byes = 0;
    for my $opp (@{$p->{'pairings'}}) {
      if ((defined $opp) && $opp == 0) {
	$byes++;
        }
      }
    $minbyes = $byes if $byes < $minbyes;
    $p->{'byes'} = $byes;
    }
  return $minbyes;
  }

=item $n = $d->CountPlayers();

Return the number of players registered in the division.

=cut

sub CountPlayers ($) { 
  my $this = shift;
  my $datap = $this->{'data'};
  return $#$datap; # players start at 1
  }

=item $success = $d->DeleteByeScore($pn1, $round0);

Delete a bye score for the given player in the given (0-based) round.
Player must have had a bye that round.

=cut

sub DeleteByeScore($$$) {
  my $this = shift;
  my $pn1 = shift;
  my $round0 = shift;

  if (!$pn1) {
    return 0;
    }
  my $p1 = $this->Player($pn1);
  if ($p1->OpponentID($round0)) {
    TSH::Utility::Error "Can't delete bye score: "
      . $p1->Name() 
      . ' did not have a bye in Round ' . ($round0+1) . '.';
    return 0;
    }
  $p1->DeleteLastScore();
  return 1;
  }

=item $success = $d->DeleteScores($pn1, $pn2, $round0);

Delete scores for the given players in the given (0-based) round.
Players must have been paired with each other in that round.
Byes may be deleted by giving one of the player numbers as 0.

=cut

sub DeleteScores($$$$) {
  my $this = shift;
  my $pn1 = shift;
  my $pn2 = shift;
  my $round0 = shift;

  if (!$pn1) {
    return $this->DeleteByeScore($pn2, $round0);
    }
  elsif (!$pn2) {
    return $this->DeleteByeScore($pn1, $round0);
    }
  my $p1 = $this->Player($pn1);
  my $p2 = $this->Player($pn2);
  if ($p1->OpponentID($round0) != $pn2) {
    TSH::Utility::Error "Can't delete scores: "
      . $p1->Name() . ' and ' . $p2->Name()
      . ' did not play each other in Round ' . ($round0+1) . '.';
    return 0;
    }
  if ($round0 != $p1->CountScores() - 1) {
    TSH::Utility::Error "Can't delete scores: "
      . $p1->Name() . ' has a score in Round ' . ($round0+2) . '.';
    return 0;
    }
  if ($round0 != $p2->CountScores() - 1) {
    TSH::Utility::Error "Can't delete scores: "
      . $p2->Name() . ' has a score in Round ' . ($round0+2) . '.';
    return 0;
    }
  $p1->DeleteLastScore();
  $p2->DeleteLastScore();
  return 1;
  }

=item $n = $d->Dirty();
=item $d->Dirty($boolean);

Get/set a division's dirtiness.

=cut

sub Dirty ($;$) { TSH::Utility::GetOrSet('dirty', @_); }

=item $n = $d->File();
=item $d->File($n);

Get/set a division's file(name).

=cut

sub File ($;$) { TSH::Utility::GetOrSet('file', @_); }

=item $r0 = $d->FirstUnpairedRound0()

Returns the (0-based) number of the first round that
is missing pairing information.

=cut

sub FirstUnpairedRound0($) {
  my $this = shift;
  return $this->{'minp'}+1;
  }

=item $dp->FormatPairing($round0, $pn1, $style)

Return a string describing the pairing for player $pn1 in 
zero-based round $round0 in division $dp.
$style may be 'normal', 'half' or 'brief'.

=cut

sub FormatPairing ($$$;$) {
  my $this = shift;
  my $round0 = shift;
  my $pn1 = shift;
  my $style = shift || 'normal';
  my $datap = $this->{'data'};

  my $pn2 = $datap->[$pn1]{'pairings'}[$round0]
    or return $style eq 'brief' ? 'bye' : '';
  my $p = $datap->[$pn1];
  my $opp = $datap->[$pn2];
  my $p121 = $p->{'etc'}{'p12'}[$round0] || 0;
  my $p122 = $opp->{'etc'}{'p12'}[$round0] || 0;
  if ($p121 == 2 && $p122 == 1) { 
    if ($style eq 'half') { return 'second vs. ' . (TSH::Utility::TaggedName $opp); }
    elsif ($style eq 'brief') { return "2nd vs. $pn2"; }
    ($p, $opp) = ($opp, $p);
    }
  elsif ($p121 == 3 && $p122 == 3) {
    if ($style eq 'half') { return 'draws vs. ' . (TSH::Utility::TaggedName $opp); }
    elsif ($style eq 'brief') { return "? vs. $pn2"; }
    else { return (TSH::Utility::TaggedName $p) . ' *draws* vs. ' . (TSH::Utility::TaggedName $opp); }
    }
  elsif (!($p121 == 1 && $p122 == 2)) {
    if ($style eq 'half') { return 'vs. ' . (TSH::Utility::TaggedName $opp); }
    elsif ($style eq 'brief') { return "? vs. $pn2"; }
    else { return (TSH::Utility::TaggedName $p) . ' vs. ' . (TSH::Utility::TaggedName $opp); }
    }
  if ($style eq 'half') { return 'first vs. ' . (TSH::Utility::TaggedName $opp); }
  elsif ($style eq 'brief') { return "1st vs. $pn2"; }
  else { return (TSH::Utility::TaggedName $p) . ' *starts* vs. ' . (TSH::Utility::TaggedName $opp); }
  }

=item $psp = $d->GetRegularUnpaired($sr0, $nobyes)

Return a vector of players that need to be paired in the given division.
Make adjustments if necessary, such as assigning a bye to make the
vector even, or (still to do) Gibsonizing a clincher. Inactive players
will be assigned their bye score.

=cut

sub GetRegularUnpaired ($$;$) {
  my $this = shift;
  my $sr0 = shift; # used in assigning byes
  my $nobyes = shift;
  my $datap = $this->{'data'};

  my $round0 = $this->LastPairedRound0();
  $round0 = 0 if $round0 < 0;
  my $psp = $this->GetUnpaired('empty is ok');
  # if last round paired is complete
  if (!@$psp) {
    # use next round
    $round0++; 
    $psp = [@$datap];
    shift @$psp;
    TSH::Player::SpliceInactive @$psp, 1, $round0;
    }

  if ($config::gibson) {
    my $gibson_count = $this->CheckGibson($sr0, $round0);
    if ($gibson_count < 0) { return []; } # error
    elsif ($gibson_count == 1) {
      # assign bye if possible
      $psp = [TSH::Player::SortByStanding $sr0, @$psp];
      my $p = shift @$psp;
      if (@$psp % 2) {
	print "Please manually pair $p->{'name'}.\n";
	return [];
        }
      else {
	$p->{'pairings'}[$round0] = 0;
	print "Assigned a bye to $p->{'name'}.\n";
        }
      }
    elsif ($gibson_count == 2) {
      # pair with each other
      $psp = [TSH::Player::SortByStanding $sr0, @$psp];
      my $p1 = shift @$psp;
      my $p2 = shift @$psp;
      $this->Pair($p1->ID(), $p2->ID(), $round0);
      print "Paired $p1->{'name'} and $p2->{'name'}.\n";
      }
    }

  # check for byes
  if (!$nobyes) {
    $this->ChooseBye($sr0, $round0, $psp);
    }

  return $psp;
  }

=item $psp = $d->GetUnpaired($emptyok)

Return a vector of active
players that need to be paired in the last round in the given division.
If the last round is fully paired, the return value will be
the empty vector if C<$emptyok> is true
and a vector of all active players if it is false.

This subroutine should generally only be called by GetRegularUnpaired
and similar low-level routines,
which further filter results by Gibsonization
and assignment of byes.

=cut

sub GetUnpaired ($;$) {
  my $dp = shift;
  my $emptyok = shift;

  my @unpaired = $emptyok 
    ? @{$dp->GetUnpairedRound($dp->LastPairedRound0())}
    : @{$dp->GetUnpairedRound($dp->FirstUnpairedRound0())};
# print 'y:', scalar(@unpaired), ',', $dp->{'maxp'}, "\n";
  TSH::Player::SpliceInactive @unpaired, 1, $dp->FirstUnpairedRound0();

  return \@unpaired;
  } 

=item $psp = $d->GetUnpairedRound($sr0)

Return a vector of active players that need to be paired in the given
zero-based round.

=cut

sub GetUnpairedRound ($$) {
  my $dp = shift;
  my $round0 = shift;

  my @unpaired = ();

  for my $p ($dp->Players()) {
    next if defined $p->OpponentID($round0);
    push(@unpaired, $p);
    } # for $p

  TSH::Player::SpliceInactive @unpaired, 1, $round0;
  
  return \@unpaired;
  } # sub GetUnpairedRound

=item $d->initialise();

(Re)initialise a Division object, for internal use.

=cut

sub initialise ($) {
  my $this = shift;
  $this->{'name'} = '';
  }

=item $d->LastPairedRound0();

Returns the (0-based) number of the last round that
contains pairing information, not including inactive players.

=cut

sub LastPairedRound0 ($) {
  my $this = shift;
  return $this->{'maxp'};
  }

=item $d->LastPairedScoreRound0();

Returns the (0-based) number of the last round that
contains a paired (nonbye) score.

=cut

sub LastPairedScoreRound0 ($) {
  my $this = shift;
  return $this->{'maxps'};
  }

=item $n = $d->LastPairedScorePlayer();

Returns a player who has a paired score in  $this->LastPairedScoreRound0().

=cut

sub LastPairedScorePlayer ($) {
  my $this = shift;
  return $this->{'maxps_player'};
  }

=item $n = $d->LeastScores();

Returns the smallest number of scores that any player in the division
has recorded.

=cut

sub LeastScores ($) {
  my $this = shift;
  return $this->{'mins'} + 1;
  }

=item $n = $d->LeastScoresPlayer();

Returns a player who has only $this->LeastScores() scores.

=cut

sub LeastScoresPlayer ($) {
  my $this = shift;
  return $this->{'mins_player'};
  }

=item $n = $d->MostScores();

Returns the largest number of scores that any player in the division
has recorded.

=cut

sub MostScores ($) {
  my $this = shift;
  return $this->{'maxs'} + 1;
  }

=item $n = $d->MostScoresPlayer();

Returns a player who has $this->MostScores() scores.

=cut

sub MostScoresPlayer ($) {
  my $this = shift;
  return $this->{'maxs_player'};
  }

=item $n = $d->Name();
=item $d->Name($n);

Get/set a division's name.

=cut

sub Name ($;$) { 
  my $this = shift;
  my $name = shift;
  my $old = $this->{'name'};
  if (defined $name) { 
    $this->{'name'} = CanonicaliseName($name); 
    }
  return $old;
  }

=item $d = new Division;

Create a new Division object.  

=cut

sub new ($) { return TSH::Utility::new(@_); }

=item $d->Pair($pn1, $pn2, $round0, $repair);

Pair two players in a given round.  Set one player number to 0 for a bye.

=cut

sub Pair ($$$$;$) { 
  my $this = shift;
  my (@p) = (shift, shift);
  my $round0 = shift;
  my $repair = shift;
  my $datap = $this->{'data'};
  my $round = $round0 + 1;
  return unless ($p[0] || $p[1]); # wise guy, huh?
  for my $p (@p) {
    if ($p < 0 || $p > $#$datap) {
      $this->Tournament->TellUser('enosuchp', $p);
      return;
      }
    next unless $p;
    my $nopps = $datap->[$p]->CountOpponents();
    if ($repair ? $nopps< $round0 : $nopps != $round0) {
      $this->Tournament->TellUser('eprbadr', $p, $nopps+1, $round);
      return;
      }
    }
  for my $i (0..1) {
    my $p = $p[$i];
    my $o = $p[1-$i];
    next unless $p;
    my $pp = $datap->[$p];
    my $opp = $pp->Opponent($round0);
    if ($opp && $opp->ID() != $o) {
      $this->Tournament->TellUser('iprwasp',
        $pp->TaggedName(), $opp->TaggedName());
      if ($round == $opp->CountOpponents()) {
	pop @{$opp->{'pairings'}};
	if ($round == $opp->CountScores()) {
	  pop @{$opp->{'scores'}};
	  }
        }
      else {
	$opp->{'pairings'}[$round0] = 0;
	if ($round < $opp->CountScores()) {
	  $opp->{'scores'}[$round0] = 50;
	  }
        }
      }
    $pp->{'pairings'}[$round0] = $o;
    }
  $this->Dirty(1);
  }

=item PairSomeSwiss($psp, $repeats, $sr0)

Swiss-pair the players in $psp (who should be in ranking order
as of round $sr0) without exceeding $repeats.

This routine is a wrapper for RecursiveSwiss, which expects
the player list to be divided into win groups.

=cut

sub PairSomeSwiss ($$$) {
  my $psp = shift;
  my $repeats = shift;
  my $sr0 = shift;
  my @win_groups = ();
  my $wins = -1;

  for my $p (@$psp) {
    my $this_wins = $p->RoundWins($sr0);
    if ($this_wins == $wins) {
      push(@{$win_groups[-1]}, $p);
      }
    else {
      push(@win_groups, [$p]);
      $wins = $this_wins;
      }
    } # for my $p

  return RecursiveSwiss \@win_groups, $repeats;
  }

=item $d->PairSwiss($repeats, $round0)

Add Swiss pairings to the last unpaired round in the division,
not exceeding $repeats, and using standings from $round0.

=cut

sub PairSwiss ($$$) {
  my $dp = shift;
  my $repeats = shift;
  my $sr0 = shift;

  my $datap = $dp->{'data'};

  my $tobepaired = $dp->GetRegularUnpaired($sr0);
  unless (@$tobepaired) {
    $dp->Tournament->TellUser('ealpaird');
    return;
    }
  die "Assertion failed" unless @$tobepaired % 2 == 0;
  my (@ranked) = TSH::Player::SortByStanding $sr0, @$tobepaired;

  my @pair_list = PairSomeSwiss \@ranked, $repeats, $sr0;
  unless (@pair_list) {
    $dp->Tournament->TellUser('ensfail');
    return;
    }
  # store pairings
  {
    while (@pair_list) {
      my $p1 = shift @pair_list;
      my $p2 = shift @pair_list;
      push(@{$p1->{'pairings'}}, $p2->{'id'});
      push(@{$p2->{'pairings'}}, $p1->{'id'});
      }
  } # store pairings
  $dp->Tournament->TellUser('idone');
  $dp->Dirty(1);
  $dp->Update();
  }

=item $pp = $d->Player($pn);

Look up a player by number.

=cut

sub Player ($$) { 
  my $this = shift;
  my $pn = shift;
  return undef if $pn < 1; # mostly in case $pn is negative
  my $p = $this->{'data'}[$pn];
  # direct access to member fields could have caused corruption
  $p = undef unless UNIVERSAL::isa($p, 'TSH::Player');
  return $p;
  }

=item @pp = $d->Players();

Return a list of all the players in the division.
The first player in the list (whose index will be 0 if you
are using 0-based subscripting) will be player #1.

=cut

sub Players ($) { 
  my $this = shift;
  my $datap = $this->{'data'};
  return @$datap[1..$#$datap];
  }

=item RecursiveSwiss($wgsp, $repeats)

Recursively compute Swiss pairings as follows.

If we have one group, try all pairings until we find one that works
or have exhausted all possibilities.  preference order for opponents
is based on distance from ideal Swiss opponent (1 plays ceil(n/2)).

If we have more than one group, try pairing the top and the bottom
groups as follows.  Make the group even by promoting (demoting)
a player if necessary.  If the group can't be paired without
exceeding the repeat threshold, promote (demote) two more players
until we succeed or run out of players and fail.

=cut

sub RecursiveSwiss ($$) {
  # [[p1,p2],[p3],[p4,p5,p6],...] divided into pairings groups
  # number of players must be even (assign bye before calling)
  my $win_groupsp = shift; # passed as reference, but shouldn't be modified
  # 0 indicates no repeats allowed, ..., 3 means up to 3 repeats = 4 pairings
  my $repeats = shift;

  Debug 'RSw', 'Main(%d): %s', scalar(@$win_groupsp), join(',',map { '[' . join(',', map { $_->ID() } @$_) . ']' } @$win_groupsp);
  confess "no groups\n" unless @$win_groupsp;

  # if we're down to one group, try all possibilities
  if (@$win_groupsp == 1) {
    return RecursiveSwissOne $win_groupsp->[0], $repeats;
    }
  # else we have more than one group
  else {
    my ($top_paired, $rest) = RecursiveSwissTop $win_groupsp, $repeats;
    # could not pair top group
    if (@$top_paired == 0) { return (); }
    # used up all players trying to pair top group
    elsif (@$rest == 0) { return @$top_paired; }
    # pairing top group left only one other group
    elsif (@$rest == 1) { 
      my (@rest) = RecursiveSwissOne $rest->[0], $repeats;
      # if that group can be paired, we're done.
      if (@rest) { return (@$top_paired, @rest); }
      # if the other group can't be paired, try pairing everyone together
      else { 
	return RecursiveSwissOne [ map { @$_ } @$win_groupsp], $repeats; 
        }
      }
    # pairing top group left more than one group
    else {
      # try pairing the bottom group
      my ($bottom_paired, $middle) = RecursiveSwissBottom $rest, $repeats;
      # if we failed, try pairing everything together
      if (@$bottom_paired == 0) { 
#	warn "Had to pair all groups together.\n";
	return RecursiveSwissOne [ map { @$_ } @$win_groupsp], $repeats; 
        }
      # if we used up the players in the middle and succeeded, return
      elsif (@$middle == 0) { return (@$top_paired, @$bottom_paired); }
      # if we left one group in the middle, try pairing it
      elsif (@$middle == 1) { 
	my (@middle) = RecursiveSwissOne $middle->[0], $repeats;
	# if that group can be paired, we're done.
	if (@middle) { return (@$top_paired, @middle, @$bottom_paired); }
	# else fall through to X
        }
      # if we left more than one group in the middle, recurse fully
      else {
	my (@middle) = RecursiveSwiss $middle, $repeats;
	if (@middle) { return (@$top_paired, @middle, @$bottom_paired); }
	# else fall through to X
        }
      # X: we couldn't pair the middle, so first try combining with bottom
      my (@midbot) = 
        RecursiveSwissOne [ map { @$_ } @$rest ], $repeats;
      # if middle and bottom paired together, return what we have
      if (@midbot) { return (@$top_paired, @midbot); }
      # else try pairing everyone together, or give up
      else { 
#	warn "Had to pair all groups together.\n";
	return RecursiveSwissOne [ map { @$_ } @$win_groupsp], $repeats; 
	}
      }
    }
  }

=item ($bottom_paired, $rest) = RecursiveSwissBottom($wgsp, $repeats)

Called by RecursiveSwiss when it wants to pair the bottom players in $wgsp.

=cut

sub RecursiveSwissBottom($$) {
  my $win_groupsp = shift;
  my $repeats = shift;

  Debug 'RSw', 'Bottom(%d): %s', scalar(@$win_groupsp), join(',',map { '[' . join(',', map { $_->ID() } @$_) . ']' } @$win_groupsp);

  # car/cdr nomenclature is upside down here, as we copied this code
  # from RecursiveSwissTop.

  # make sure we have an even number of players
  my $carp = $win_groupsp->[-1];
  my (@cdr) = @$win_groupsp[0..$#$win_groupsp-1];

  my (@car) = (@$carp); # copy so as not to change master copy
  my @cadr;
  unless ($#$carp % 2) {
    # [1] get (copy, so we don't mess things up) the next group
    (@cadr) = (@{pop @cdr});
    # move its top player to top group
      # change this and the other two push(@car)s below to unshift()s
      # and watch the runtime increase by two orders of magnitude
    push(@car, pop @cadr);
    # if that didn't empty the next group, put it back
    if (@cadr) { push(@cdr, \@cadr); }
    }
  # pair within the group, then keep demoting pairs
  while (1) {
    my (@car_paired) = RecursiveSwiss [\@car], $repeats;
    if (@car_paired) {
      return \@car_paired, \@cdr;
      }
    # did we run out of players?
    last unless @cdr;
    # demote one, see [1] for comments
    (@cadr) = (@{pop @cdr}); 
    push(@car, pop @cadr);
    if (@cadr) { push(@cdr, \@cadr); }
    die "Ark error - did you forget to assign a bye?\n" unless @cdr;
    # demote the other, see [1] for comments
    (@cadr) = (@{pop @cdr}); 
    push(@car, pop @cadr);
    if (@cadr) { push(@cdr, \@cadr); }
  }
  # ran out of players - d'oh!
  Debug 'RSw', 'failed: no more players to demote';
  return ([], []);
  }

=item $success = RecursiveSwissOne($wgsp, $repeats)

Called by RecursiveSwiss when it knows that it has only one group to pair.

=cut

sub RecursiveSwissOne($$) {
  my $win_groupp = shift;
  my $repeats = shift;
  Debug 'RSw', 'One(1:%d)', scalar(@$win_groupp);
  
  my $group_size = scalar(@$win_groupp);
  # odd number of players - oops
  if ($group_size % 2) {
    die "Odd number of players in last win group:" 
      . join(',', map { $_->{'id'}} @$win_groupp)
      . "\nAborting";
    }
  # no more players - shouldn't happen
  elsif ($group_size == 0) {
    die "Ran out of players?!\n";
    }
  # one pair left - almost as easy
  elsif ($group_size == 2) {
    my ($p1, $p2) = @$win_groupp;
    if ($p1->Repeats($p2->ID()) <= $repeats
      && ((!$config::exagony) || $p1->Team() ne $p2->Team())) {
      Debug 'RSw', 'success: %s', join(',', map { $_->ID() } ($p1, $p2));
      return ($p1, $p2);
      }
    else {
      Debug 'RSw', 'cannot pair %s and %s', $p1->TaggedName(), $p2->TaggedName();
      return ();
      }
    }
  # more than one pair - try each possible opp for first player, recurse
  else {
    my (@ps) = @$win_groupp;
    # preferred opponent is halfway down the group
    if (TSH::Player::PairGRT(\@ps,
      sub {
	my $p = $_[0][$_[1]];
	my $pid = $p->ID();
	my $o = $_[0][$_[2]];
	my $oid = $o->ID();
	my $lastoid = ($p->OpponentID(-1) || -1);
	my $repeats = $p->Repeats($oid); 
	my $sameopp = ($oid == $lastoid);
	my $distance = abs(@{$_[0]}-abs(2*($_[1]-$_[2])));
	my $pairsvr = $config::track_firsts ? 2-abs(($p->{'p1'}-$p->{'p2'} <=> 0)  -($o->{'p1'}-$o->{'p2'} <=> 0)) : 0;

 	Debug 'GRT', 'pref RS1 %d-%d rep=%d prev=%d svr=%d rnk=%d', $pid, $oid, $repeats, $sameopp, $pairsvr, $distance;
	pack('NCCNN', 
	  $repeats, # minimize repeats
	  $sameopp, # avoid previous opponent
	  $pairsvr, # pair those due to start vs those due to reply
	  $distance, # optimize rank
	  $_ # index for GRT to extract
	  );
        },
      sub {
      ($config::exagony ? $_[0][$_[1]]->Team() ne $_[0][$_[2]]->Team() : 1) && 
        $_[0][$_[1]]->Repeats($_[0][$_[2]]->ID()) <= $repeats
        },
      [],
      'just checking'
      )) {
      my (@pairings);
      my $debug = '';
      for my $p (@ps) {
	if ($p->ID() < $p->{'opp'}) # should fix use of 'opp'
	  { 
	    push(@pairings, $p, $p->Division->Player($p->{'opp'})); 
	  $debug .= $p->ID() . '-' . $p->{'opp'} . ' ';
	  }
        }
      Debug 'RSw', $debug;
      return @pairings;
      }
    else {
      Debug 'RSw', 'cannot pair group led by %s', $ps[0]->TaggedName();
      return ();
      }
    }
  }

=item ($top_paired, $rest) = RecursiveSwissTop($wgsp, $repeats)

Called by RecursiveSwiss when it wants to pair the top players in $wgsp.

=cut

sub RecursiveSwissTop($$) {
  my $win_groupsp = shift;
  my $repeats = shift;

  Debug 'RSw', 'Top(%d)', scalar(@$win_groupsp);
  # make sure we have an even number of players
  my ($carp, @cdr) = @$win_groupsp;
  my (@car) = (@$carp); # copy so as not to change master copy
  my @cadr;
  unless ($#$carp % 2) {
    # [1] get (copy, so we don't mess things up) the next group
    (@cadr) = (@{shift @cdr});
    # move its top player to top group
    push(@car, shift @cadr);
    # if that didn't empty the next group, put it back
    if (@cadr) { unshift(@cdr, \@cadr); }
    }
  # pair within the group, then keep promoting pairs
  while (1) {
    my (@car_paired) = RecursiveSwiss [\@car], $repeats;
    if (@car_paired) {
      return \@car_paired, \@cdr;
      }
    # did we run out of players?
    last unless @cdr;
    # promote one, see [1] for comments
    (@cadr) = (@{shift @cdr}); 
    push(@car, shift @cadr);
    if (@cadr) { unshift(@cdr, \@cadr); }
    die "Ark error - did you forget to assign a bye?\n" unless @cdr;
    # promote the other, see [1] for comments
    (@cadr) = (@{shift @cdr}); 
    push(@car, shift @cadr);
    if (@cadr) { unshift(@cdr, \@cadr); }
  }
  # ran out of players - d'oh!
  Debug 'RSw', 'failed: no more players to promote';
  return ([], []);
  }

=item $n = $d->Synch();

Update internal statistics.

=cut

sub Synch ($) { 
  my $this = shift;
  my $datap = $this->{'data'};

  my $minpairs = 999999;
  my $maxpairs = -1;
  my $minscores = 999999;
  my $maxscores = -1;
  my $maxps = -1;
  my $maxps_player = undef;
  my $mins_player = undef;
  my $maxs_player = undef;

  for my $i (1..$#$datap) {
    my $p = $datap->[$i];
    my $pairingsp = $p->{'pairings'};
    my $npairs = $#$pairingsp;
    my $nscores = $#{$p->{'scores'}};
    my $spread = 0;
    my (@spread) = ();
    my $ratedgames = 0; 
    my $ratedwins = 0; 
    my $wins = 0;
    my $losses = 0;
    my (@losses) = ();
    my (@wins) = ();
    $p->{'ewins1'} = $p->{'ewins2'} = 0;

# print "$p->{id} has $nscores+1 scores.\n" if $nscores > $maxscores;
# print "$p->{id} has $npairs+1 opponents.\n" if $npairs > $maxpairs;
    my $active = !$p->{'etc'}{'off'};
    $minpairs = $npairs if $npairs < $minpairs && $active;
    $maxpairs = $npairs if $npairs > $maxpairs && $active;
    if ($nscores < $minscores && $active) {
      $minscores = $nscores;
      $mins_player = $p;
      }
    if ($nscores > $maxscores && $active) {
      $maxscores = $nscores;
      $maxs_player = $p;
#     print join(' ', $p->{'name'}, $nscores, @{$p->{'scores'}}), "\n";
      }

    my $last_ps = -1;
    for my $j (0..$nscores) { # number of scores
      my $opp = $pairingsp->[$j];
      if ($opp && $active) { $last_ps = $j; }
      if (!defined $opp) {
	my $name = $p->TaggedName();
	my $r1 = $j + 1;
	TSH::Utility::Error "Oops: $name has a score but no opp in round $r1.\n";
        }
      else {
	my $myscore = $p->{'scores'}[$j];
	my $oppscore;
	if ($opp) {
	  $oppscore = $datap->[$opp]{'scores'}[$j];
	  if ((!defined $oppscore) && defined $myscore) {
	    my $name = $p->TaggedName();
	    my $r1 = $j + 1;
	    TSH::Utility::Error "Oops: $name is missing an opponent score in round $r1.\n";
	    $oppscore = 0;
	    }
	  else {
	    $ratedgames++;
	    }
	  }
	else {
	  $oppscore = 0;
	  }
	unless (defined $myscore) {
	  my $name = $p->TaggedName();
	  my $r1 = $j + 1;
	  TSH::Utility::Error "Oops: $name is missing a score vs. #$opp in round $r1.\n";
	  }
        my $thisSpread = $p->{'scores'}[$j] - $oppscore;
        $spread += $thisSpread;
        push(@spread, $spread);
	my $result = (1 + ($thisSpread <=> 0))/2;
	if ($opp || $thisSpread) {
	  $wins += $result;
	  $losses += 1 - $result; 
	  }
	if ($opp) {
	  $ratedwins += $result;
	  $p->{$j < $config'split1 ? 'ewins1' : 'ewins2'} += $result;
	  }
        push(@losses, $losses);
        push(@wins, $wins);
        }
      } # for $j
    if ($last_ps > $maxps) { 
      $maxps = $last_ps;
      $maxps_player = $p;
      }
    $p->{'rspread'} = \@spread;
    $p->{'rlosses'} = \@losses;
    $p->{'rwins'} = \@wins;
    $p->{'spread'} = $spread;
    $p->{'ratedwins'} = $ratedwins;
    $p->{'ratedgames'} = $ratedgames;
    $p->{'losses'} = $losses;
    $p->{'wins'} = $wins;

    { 
      my (@repeats) = (0) x @$datap;
      for my $j (@$pairingsp) { $repeats[$j]++; }
      $p->{'repeats'} = \@repeats;
    }

    }

  $this->{'mins'} = $minscores;
  $this->{'mins_player'} = $mins_player;
  $this->{'maxps'} = $maxps;
  $this->{'maxps_player'} = $maxps_player;
  $this->{'maxs'} = $maxscores;
  $this->{'maxs_player'} = $maxs_player;
  $this->{'maxp'} = $maxpairs;
  $this->{'minp'} = $minpairs;

  if ($config::track_firsts) { # must come after maxp computation
    $this->SynchFirsts();
    }
  }

=item $n = $d->SynchFirsts();

Make sure that firsts and seconds as recorded in 
$p->{'etc'}{'p12'} are consistent, and make any
possible inferences.  Firsts and seconds are encoded as follows:

  0: bye
  1: first
  2: second
  3: must draw
  4: indeterminate pending prior draws

Recalculate $p->{'p1'},...,$p->{'p4'}

=cut

sub SynchFirsts ($) {
  my $this = shift;
  my $datap = $this->{'data'};

  for my $p (@$datap[1..$#$datap]) {
    $p->{'p1'} = $p->{'p2'} = $p->{'p3'} = $p->{'p4'} = 0;
    }
  # If we are tracking but not assigning firsts we should recalculate
  # inferred future firsts in case past firsts have been edited.
  # We force this to happen by truncating firsts back to the last 
  # entered score.
  if (!$config::assign_firsts) {
    for my $p (@$datap[1..$#$datap]) {
      my $scoresp = $p->{'scores'};
      my $p12p = $p->{'etc'}{'p12'};
      if ($#$p12p > $#$scoresp) {
        $#$p12p = $#$scoresp;
#	print "Truncating starts/replies for ", $p->TaggedName(), "\n";
        }
      }
    }

  # check consistency of firsts, see if we can make future inferences
  for my $round0 (0..$this->{'maxp'}) {
    my $o12;
    my $oppp;
    my $p12;
    for my $p (@$datap[1..$#$datap]) {
      my $oppid = $p->{'pairings'}[$round0];
      last unless defined $oppid;
      my $p12p = $p->{'etc'}{'p12'};
      if ($oppid == 0) { $p12 = 0; $oppp = undef; next; }
      $oppp = undef;
      $p12 = $p12p->[$round0];
      if ($oppid < $p->{'id'}) { # in theory, we already did this one
	# if the pairings are inconsistent, though...
	if (!defined $p12p->[$round0]) {
	  my $round = $round0 + 1;
	  TSH::Utility::Error((TSH::Utility::TaggedName $p) . " has no opponent in round $round.\n");
	  $p12 = 4;
	  }
	next;
        }
      $oppp = $datap->[$oppid];
      my $o12p = $oppp->{'etc'}{'p12'};
      my $exists = 1;
      $o12 = $o12p->[$round0];
      my $p12known = $p12 && $p12 < 4;
      my $o12known = $o12 && $o12 < 4;
      if ($p12known) {
	if ($o12known) { # both set: check for consistency
	  if ($o12 != (0,2,1,3)[$p12]) {
	    $this->Tournament->TellUser('edivbad12', 
              (TSH::Utility::TaggedName $p),
              (TSH::Utility::TaggedName $oppp),
	      $round0+1);
	    }
	  }
	else # we are set but opp is not: set opp
	  { $o12p->[$round0] = (0, 2, 1, 3)[$p12]; }
        }
      else {
        if ($o12known) {
	  # opp is set but we are not: set us
	  $p12p->[$round0] = (0, 2, 1, 3)[$o12];
	  }
	else { $exists = 0; }
        }
      if ($exists) { 
	$p->{"p$p12p->[$round0]"}++;
	$oppp->{"p$o12p->[$round0]"}++;
	next;
        }
      # otherwise, see if we can deduce first/second
      my $ofuzz = $oppp->{'p3'} + $oppp->{'p4'};
      my $pfuzz = $p->{'p3'} + $p->{'p4'};
      if ($pfuzz + $ofuzz == 0 || $round0 == 0) {
	my $which = 1 +
	  ($p->{'p1'} <=> $oppp->{'p1'} || $oppp->{'p2'} <=> $p->{'p2'});
#  print "$p->{'name'} $p->{'p1'} $p->{'p2'} $oppp->{'name'} $oppp->{'p1'} $oppp->{'p2'} $which\n";
	if ($which == 1 && $config'assign_firsts) {
	  $which = 2 * int(rand(2));
	  }
        $p12 = (1, 3, 2)[$which];
        $o12 = (2, 3, 1)[$which];
        }
      # else there's fuzz, so we could get 4s
      else {
	my $diff1 = $p->{'p1'} - $oppp->{'p1'};
	my $diff2 = $p->{'p2'} - $oppp->{'p2'};
	if (($diff1 <=> $ofuzz || -$diff2 <=> $pfuzz) > 0) 
	  { $p12 = 2; $o12 = 1; }
	elsif ((-$diff1 <=> $pfuzz || $diff2 <=> $ofuzz) > 0) 
  	  { $p12 = 1; $o12 = 2; }
	elsif ($config'assign_firsts) {
	  if (rand(1) > 0.5) { $p12 = 1; $o12 = 2; }
	  else { $p12 = 2; $o12 = 1; }
	  }
	else 
	  { $p12 = $o12 = 4; } 
        }
      }
    continue
      {
      $p->{'etc'}{'p12'}[$round0] = $p12;
      warn "p=$p name=$p->{'name'} p12=$p12\n" unless defined $p12;
      $p->{"p$p12"}++;
      if ($oppp) {
	$oppp->{'etc'}{'p12'}[$round0] = $o12;
	$oppp->{"p$o12"}++;
        }
      }
    }
  }

=item $t = $d->Tournament();
=item $d->Tournament($t);

Get/set a division's associated tournament.

=cut

sub Tournament ($;$) { TSH::Utility::GetOrSet('tournament', @_); }

=item $d->Update();

$d->Synch(), then $d->Write().

=cut

sub Update ($) { 
  my $this = shift;
  return unless $this->Dirty();
  $this->Synch();
  $this->Write();
  $this->Dirty(0);
  }

=item $d->Write();

Save changes to disk.

=cut

sub Write ($) { 
  my $this = shift;
  my $fn = $this->File();
  my $fh = gensym;

  rename "$config'root_directory/$fn", ("$config'backup_directory$fn.".time)
    or die "Can't backup division to $config'backup_directory$fn.time ($!).\n";
  open($fh, ">$config'root_directory/$fn") 
    or die "Can't create division ($!).\n";
  &MacPerl'SetFileInfo('McPL', 'TEXT', $fn) if defined &MacPerl'SetFileInfo;
  warn "updating $fn\n";
  for my $p (@{$this->{'data'}}) {
    if (defined $p && defined $p->{'id'} && $p->{'id'}) {
      printf $fh "%-22s %4d %s; %s",
        $p->{'name'},
        $p->{'rating'},
        join(' ', @{$p->{'pairings'}}),
        join(' ', @{$p->{'scores'}});
      if ($p->{'etc'}) {
	my $etcp = $p->{'etc'};
	for my $key (sort keys %$etcp) {
	  my $wordsp = $etcp->{$key};
 	  for my $i (0..$#$wordsp) { warn "undefined value at i=$i in $key among @$wordsp for $p->{'name'}" unless defined $wordsp->[$i]; }
	  print $fh "; $key @$wordsp" if $wordsp;
	  }
        }
      print $fh "\n";
      }
    }
  close($fh);
  }

=back

=cut

=head1 BUGS

GetRegularUnpaired should automatically gibsonize.

RecursiveSwiss can sometimes take pathologically long to run,
and needs to be improved.

RecursiveSwiss makes illegal use of the 'opp' field in Player.pm

RecursiveSwiss should make use of firsts/seconds when in use.

It might be a good idea for RecursiveSwiss to keep track of what
FIDE calls upfloats and downfloats and to try to avoid repeating
them in consecutive rounds.

Should check for when corrupt .t files have one score from a game but not 
the other.

Should not rely on TSH::Player private functions.

=cut

1;

