#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Player;

use strict;
use warnings;
use Carp;

use TSH::Utility qw(Debug DebugDumpPairings);

=pod

=head1 NAME

TSH::Player - abstraction of a Scrabble player within C<tsh>

=head1 SYNOPSIS

  $p = new Player;
  $s = $p->CountOpponents();
  $s = $p->CountScores();
  $s = $p->DeleteLastScore();
  $s = $p->Division();
  $s = $p->First();
  $s = $p->First($p12);
  $s = $p->FullID();
  $s = $p->GamesPlayed();
  $s = $p->ID();
  $r = $p->MaxRank();
  $p->MaxRank($r);
  $s = $p->Name();
  $r = $p->NewRating();
  $p->NewRating($r);
  $s = $p->Opponent($round0);
  $s = $p->OpponentID($round0);
  $s = $p->OpponentScore($round0);
  $success = PairGRT($grtsub, $psp, $grtargs[, $just_checking]);
  $r = $p->Random();
  $p->Random($r);
  $r = $p->Rating();
  $p->Rating($r);
  $r = $p->Repeats($oppid);
  $p->Repeats($oppid, $r);
  $s = $p->RoundRank($round0);
  $p->RoundRank($round0, $rank);
  $s = $p->RoundSpread($round0);
  $s = $p->RoundWins($round0);
  $s = $p->Score($round0);
  $old = $p->Score($round0, $score);
  @p = TSH::Player::SortByCurrentStanding(@p);
  @p = TSH::Player::SortByInitialStanding(@p);
  @p = TSH::Player::SortByStanding($sr0, @p);
  TSH::Player::SpliceInactive(@ps, $nrounds, $round0);
  $s = $p->TaggedName();
  $s = $p->Team();
  $success = $p->UnpairRound($round0);
  $s = $p->Wins();

=head1 ABSTRACT

This Perl module is used to manipulate players within C<tsh>.

=head1 DESCRIPTION

A Player has (at least) the following member fields, none of which
ought to be accessed directly from outside the class.

  byes        number of byes, last time we counted
  division    pointer to division
  etc         supplementary player data (see below)
  ewins1      rated wins earned in the first half of a split tournament
  ewins2      rated wins earned in the second half of a split tournament
  id          player ID (1-based) # not sure this is still here
  losses      number of losses
  maxrank     highest rank still attainable by player in this tournament
  name        player name
  opp         provisional opponent used within pairing routines
  p1          number of firsts (starts)
  p2          number of seconds (replies)
  p3          number of indeterminates (starts/replies)
  pairings    list of opponent IDs by round (0-based by round, 1-based IDs)
  ratedgames  number of rated games
  ratedwins   number of rated wins
  rating      pre-tournament rating
  repeats     data structure tracking repeat pairings
  rnd         pseudorandom value used to break ties in standings
  rrank       rank by round (1-based with round 0 being preevent rankings)
  rlosses     cumulative losses by round
  rspread     cumulative spread by round
  rwins       cumulative wins by round
  scores      list of this player's scores by round (0-based)
  spread      cumulative spread
  tspread     temporary spread variable used by some routines
  twins       temporary wins variable used by some routines
  wins        number of wins

Supplementary player data is currently as follows:

  board       0-based list indicating board at which player played each round
  off         exists if player is inactive, single value indicates
              type of byes (-50/0/50) to be assigned
  p12         0-based list, 1 if went first, 2 if second, 0 if neither (bye),
              3 if must draw, 4 if indeterminate
  team        Team name

The following member functions are currently defined.

=over 4

=cut

sub Active ($);
sub Board ($$;$);
sub CountOpponents ($);
sub CountScores ($);
sub DeleteLastScore ($);
sub Division ($);
sub First ($$;$);
sub GamesPlayed ($);
sub FullID ($);
sub ID ($);
sub initialise ($);
sub MaxRank ($;$);
sub new ($);
sub Name ($);
sub NewRating ($;$);
sub Opponent ($$);
sub OpponentID ($$);
sub OpponentScore ($$);
sub PairGRT ($$$$;$);
sub Rating ($;$);
sub Repeats ($$;$);
sub RoundRank ($$;$);
sub RoundSpread ($$);
sub RoundWins ($$);
sub Score ($$;$);
sub SortByCurrentStanding (@);
sub SortByInitialStanding (@);
sub SortByStanding ($@);
sub SpliceInactive (\@$$);
sub Spread ($);
sub TaggedName ($);
sub UnpairRound ($$);
sub Wins ($);

=item $n = $p->Active();

Return true if the player is active for pairings.

=cut

sub Active ($) { 
  my $this = shift;
  return !exists $this->{'etc'}{'off'};
  }

=item $n = $p->Board($r0);

Set/get the board number at which the player played in zero-based round $r0.

=cut

sub Board ($$;$) { 
  my $this = shift;
  my $r0 = shift;
  my $newboard = shift;
  my $boardp = $this->{'etc'}{'board'};
  unless (defined $boardp) { $this->{'etc'}{'board'} = $boardp = []; }
  my $oldboard = $boardp->[$r0] || 0;
  if (defined $newboard) {
    if ($r0 > $#$boardp + 1) {
      push(@$boardp, (0) x ($r0 - $#$boardp - 1));
      }
    $boardp->[$r0] = $newboard;
    $this->Division->Dirty(1);
    }
  return $oldboard;
  }

=item $n = $p->CountOpponents();

Count how many opponents (including byes) a player has been paired with.

=cut

sub CountOpponents ($) { 
  my $this = shift;
  return 1+$#{$this->{'pairings'}};
  }

=item $n = $p->CountScores();

Count how many scores a player has recorded.

=cut

sub CountScores ($) { 
  my $this = shift;
  return 1+$#{$this->{'scores'}};
  }

=item $n = $p->DeleteLastScore();

Deletes the player's last score. 
Should be used with extreme caution, and only after checking with
the player's opponent if any.

=cut

sub DeleteLastScore ($) { 
  my $this = shift;
  return pop @{$this->{'scores'}};
  }

=item $n = $p->Division();

Return the player's division.

=cut

sub Division ($) { 
  my $this = shift;
  return $this->{'division'};
  }

=item $n = $p->First($round0[, $newvalue]);

Returns a number indicating whether the player did or will go
first in a zero-based round.

  0: did not play this round
  1: went first (started)
  2: went second (replied)
  3: must draw
  4: unknown pending earlier firsts/seconds or other information

=cut

sub First ($$;$) { 
  my $this = shift;
  my $round0 = shift;
  my $newp12 = shift;
  my $p12sp = $this->{'etc'}{'p12'};
  if (!defined $p12sp) {
    $p12sp = $this->{'etc'}{'p12'} = [];
    }
  my $oldp12 = $p12sp->[$round0];
  $oldp12 = 4 unless defined $oldp12;
  if (defined $newp12) {
    if ($round0 < 0 || $round0 > $#$p12sp+1) {
      $this->Division()->Tournament->TellUser('eplyrror',
	'p12', $this->TaggedName(), $round0+1, $newp12);
      }
    elsif ($newp12 !~ /^[0-4]$/) {
      $this->Division()->Tournament->TellUser('eplyrbv',
	'p12', $this->TaggedName(), $round0+1, $newp12);
      }
    else {
      $p12sp->[$round0] = $newp12;
#     print "set p12 for $this->{'id'} in r0 $round0 to $newp12\n";
      }
    }
  return $oldp12;
  }

=item $n = $p->FullID();

Return the player's 1-based ID number, formatted by
prepending either the division name (if there is more than
one division) or a number sign.

=cut

sub FullID ($) { 
  my $this = shift;
  my $dname = '';
  if ($this->Division()->Tournament()->CountDivisions() > 1) {
    $dname = $this->Division()->Name();
    $dname .= '-' if $dname =~ /\d$/;
    }
  return sprintf("%s$config::player_number_format", uc($dname), $this->{'id'});
  }

=item $n = $p->GamesPlayed();

Return the number of games a player has played

=cut

sub GamesPlayed ($) { 
  my $this = shift;
  return scalar(@{$this->{'scores'}});
  }

=item $n = $p->ID();

Return the player's 1-based ID number.

=cut

sub ID ($) { 
  my $this = shift;
  return $this->{'id'};
  }

=item $d->initialise();

(Re)initialise a Division object, for internal use.

=cut

sub initialise ($) {
  my $this = shift;
  $this->{'name'} = '';
  }

=item $n = $p->Losses();

Return the player's total losses so far this tournament.

=cut

sub Losses ($) { 
  my $this = shift;
  return ($this->{'losses'} || 0);
  }

=item $d = new Player;
Create a new Player object.  

=cut

=item $r = $p->MaxRank();
=item $p->MaxRank($r);

Set or get the player's rating

=cut

sub MaxRank ($;$) { TSH::Utility::GetOrSet('maxrank', @_); }

sub new ($) { return TSH::Utility::new(@_); }

=item $n = $p->Name();

Return the player's name.

=cut

sub Name ($) { 
  my $this = shift;
  return ($this->{'name'} || '?');
  }

=item $r = $p->NewRating();
=item $p->NewRating($r);

Set or get the player's newly calculated rating estimate

=cut

sub NewRating ($;$) { TSH::Utility::GetOrSet('newr', @_); }

=item $n = $p->Opponent($round0);

Return the player's opponent in 0-based
round $round0.

=cut

sub Opponent ($$) {
  my $this = shift;
  my $round0 = shift;
  my $oppid = $this->OpponentID($round0);
  if ($oppid) {
    return $this->Division()->Player($oppid);
    }
  else {
    return undef;
    }
  }

=item $n = $p->OpponentID($round0);

Return the 1-based ID number of the player's opponent in 0-based
round $round0.

=cut

sub OpponentID ($$) { 
  my $this = shift;
  my $round0 = shift;
# confess unless defined $round0;
  return $this->{'pairings'}[$round0];
  }

=item $n = $p->OpponentScore($round0);

Return the player's opponent's score in 0-based
round $round0.

=cut

sub OpponentScore ($$) {
  my $this = shift;
  my $round0 = shift;
  my $opp = $this->Opponent($round0);
  if ($opp) {
    return $opp->Score($round0);
    }
  else {
    return undef;
    }
  }

=item $success = PairGRT($psp, $grtsub, $filter, \@grtargs[, $just_checking]);

Pair the players listed in @$psp allowing each player to be paired
only with the opponents that pass &$filter, ranking them in preference
order according to the Guttman-Rosler Transform sub $grtsub, which
must pack player index in $psp as an 'N' at the end of its output
string.  $grtsub and $filter are passed the following arguments:

  - $psp 
  - index of current player in $psp
  - index of opponent in $psp
  - @grtargs

=cut

sub PairGRT ($$$$;$) { 
  my $psp = shift;
  my $grt = shift;
  my $filter = shift;
  my $argsp = shift;
  my $just_checking = shift;
# Debug 'GRT', 'PGRT psp: %s', join(',',map { $_->ID() } @$psp);
  for my $i (0..$#$psp) {
    my $p = $psp->[$i];
    my (@opps) = 
      # Guttman-Rosler Transform
      map { $psp->[unpack('N', substr($_, -4))] }
      sort 
      map { &$grt($psp, $i, $_, @$argsp) }
      grep { &$filter($psp, $i, $_, @$argsp) }
      (0..$i-1,$i+1..$#$psp);
    Debug 'GRT', 'pref %d: %s', $p->ID(), join(',',map { 
	defined $_ ? $_->ID() : '???';
#	$_->ID() 
      } @opps);
    $psp->[$i]{'pref'} = \@opps;
    }
  if (main::ResolvePairings($psp, $just_checking)) {
    DebugDumpPairings 'GRT', $psp->[0]->CountOpponents()-1, $psp
      unless $just_checking;
    return 1;
    }
  else {
    return 0;
    }
  }

=item $r = $p->Random();
=item $p->Random($r);

Set or get the player's random value.

=cut

sub Random ($;$) { TSH::Utility::GetOrSet('rnd', @_); }

=item $r = $p->Rating();
=item $p->Rating($r);

Set or get the player's rating

=cut

sub Rating ($;$) { TSH::Utility::GetOrSet('rating', @_); }

=item $n = $p->Repeats($oppid);
=item $n = $p->Repeats($oppid, $n);

Set or get the number of repeat pairings of this player with $oppid.

=cut

sub Repeats ($$;$) { 
  my $this = shift;
  my $oppid = shift;
  my $newrepeats = shift;
  my $repeatsp = $this->{'repeats'};
  unless (defined $repeatsp) { $this->{'repeats'} = $repeatsp = []; }
  my $oldrepeats = $repeatsp->[$oppid];
  if (defined $newrepeats) {
    if ($oppid < 0) {
      TSH::Utility::Error("Can't set repeats for " 
	. $this->TaggedName() 
	. " vs #$oppid to $newrepeats: round number is out of range."
        );
      }
    else {
      $repeatsp->[$oppid] = $newrepeats;
      }
    }
  if ((!defined $oldrepeats) && !defined $newrepeats) {
    TSH::Utility::Error("Can't get repeats for " 
      . $this->TaggedName() 
      . " vs #$oppid: repeats are not yet computed."
      );
    $oldrepeats = 0;
    }
  return $oldrepeats;
  }

=item $n = $p->RoundLosses($round0);

Return the player's cumulative losses as of 0-based round $round0.

=cut

sub RoundLosses ($$) { 
  my $this = shift;
  my $round0 = shift;
  return 0 if $round0 < 0;
  return 
    exists $this->{'rlosses'}[$round0] 
      ? $this->{'rlosses'}[$round0]
      : ($this->{'losses'} || 0);
  }

=item $n = $p->RoundRank($round0, $rank);

Set or get the player's rank in 0-based
round $round0.

=cut

sub RoundRank ($$;$) { 
  my $this = shift;
  my $round0 = shift;
  my $newrank = shift;
  my $ranksp = $this->{'rrank'};
  unless (defined $ranksp) { $this->{'rrank'} = $ranksp = []; }
  # Normally we work internally with zero-based indexing for round numbers.
  # Here we use one-based indexing and reserve the 'round 0 ranking' for
  # preevent ranking.
  my $round = $round0 + 1;
  my $oldrank = $ranksp->[$round];
# printf STDERR "r=%d new=%d old=%d\n", $round, ($newrank||-1), ($oldrank||-1);
  if (defined $newrank) {
    if ($round < 0) {
      $this->Division()->Tournament->TellUser('eplyrror',
	'rank', $this->TaggedName(), $round, $newrank);
      }
    else {
      $ranksp->[$round] = $newrank;
      }
    }
  if ((!defined $newrank) && !defined $oldrank) {
    TSH::Utility::Error("Can't get rank for " 
      . $this->TaggedName() 
      . " in round $round: ranking is not yet computed." 
      );
    $oldrank = 0;
    }
  return $oldrank;
  }

=item $n = $p->RoundSpread($round0);

Return the player's cumulative spread as of 0-based round $round0.

=cut

sub RoundSpread ($$) { 
  my $this = shift;
  my $round0 = shift;
  return 0 if $round0 < 0;
  return 
    exists $this->{'rspread'}[$round0] 
      ? $this->{'rspread'}[$round0]
      : $this->{'spread'};
  }

=item $n = $p->RoundWins($round0);

Return the player's cumulative wins as of 0-based round $round0.

=cut

sub RoundWins ($$) { 
  my $this = shift;
  my $round0 = shift;
  return 0 if $round0 < 0;
  return 
    exists $this->{'rwins'}[$round0] 
      ? $this->{'rwins'}[$round0]
      : ($this->{'wins'} || 0);
  }

=item $n = $p->Score($round0[, $score]);

Return or set the player's score in 0-based
round $round0.

=cut

sub Score ($$;$) { 
  my $this = shift;
  my $round0 = shift;
  my $newscore = shift;
  my $scoresp = $this->{'scores'};
  my $oldscore = $scoresp->[$round0];
  if (defined $newscore) {
    if ($round0 < 0 || $round0 > $#$scoresp+1) {
      $this->Division()->Tournament->TellUser('eplyrror',
	'score', $this->TaggedName(), $round0+1, $newscore);
      }
    else {
      $scoresp->[$round0] = $newscore;
      }
    }
  return $oldscore;
  }

=item SortByCurrentStanding(@players)

Sorts players according to their current standings and returns the sorted list.

=cut

sub SortByCurrentStanding (@) {
  return sort {
    $b->{wins} <=> $a->{wins} ||
    $a->{losses} <=> $b->{losses} ||
    $b->{spread} <=> $a->{spread} ||
    $b->{rating} <=> $a->{rating} ||
    $b->{rnd} <=> $a->{rnd};
    } @_;
  }

=item SortByInitialStanding(@players)

Sorts players according to their initial standings and returns the sorted list.

=cut

sub SortByInitialStanding (@) {
  return sort {
    $b->{rating} <=> $a->{rating} ||
    $b->{rnd} <=> $a->{rnd}
    } @_;
  }

=item SortByStanding($round0, @players)

Sorts players according to their standings as of zero-based round
$round0 and returns the sorted list.

=cut

sub SortByStanding ($@) {
  my $sr0 = shift;

# local($^W) = 0;
  return sort { 
#  die ("Incomplete player $a->{'name'} ($a):\n  ".join(', ',keys %$a)."\n") unless defined $a->{'wins'} && defined $a->{'spread'} && defined $a->{'rating'} && defined $a->{'rnd'};
    $sr0 >= 0 ? 
      ((defined ($b->{'rwins'}[$sr0]) ? $b->{'rwins'}[$sr0] : $b->{'wins'})<=>(defined ($a->{'rwins'}[$sr0]) ? $a->{'rwins'}[$sr0] : $a->{'wins'}) ||
      ((defined ($a->{'rlosses'}[$sr0]) ? $a->{'rlosses'}[$sr0] : $a->{'losses'})<=>(defined ($b->{'rlosses'}[$sr0]) ? $b->{'rlosses'}[$sr0] : $b->{'losses'})) ||
      ((defined ($b->{'rspread'}[$sr0]) ? $b->{'rspread'}[$sr0] : $b->{'spread'})<=>(defined $a->{'rspread'}[$sr0] ? $a->{'rspread'}[$sr0] : $a->{'spread'})) || 
      $b->{'rating'}<=>$a->{'rating'} ||
      $b->{'rnd'}<=>$a->{'rnd'})
    : ($b->{rating}<=>$a->{rating} || $b->{rnd} <=> $a->{rnd})
    ; } @_;
  }

=item SpliceInactive(@ps, $nrounds, $round0)

Removes all inactive players from C<@ps>
and assigns them byes for the next $nrounds$ rounds
beginning with round $round0
unless they already have pairings/scores for those rounds.

=cut

sub SpliceInactive (\@$$) {
  my $psp = shift;
  my $count = shift;
  my $round0 = shift;

  return if $round0 < 0;
  for (my $i=0; $i<=$#$psp; $i++) {
    my $p = $psp->[$i];
    my $off = $p->{'etc'}{'off'};
    next unless defined $off;
    splice(@$psp, $i--, 1);
    
    my $pairingsp = $p->{'pairings'};
    next if $#$pairingsp < $round0-1;
    my $scoresp = $p->{'scores'};
    for my $i ($round0..$round0+$count-1) {
      $pairingsp->[$i] = 0 unless defined $pairingsp->[$i];
      $scoresp->[$i] = $off->[0] unless defined $scoresp->[$i];
      }
    }
  }

=item $n = $p->Spread();

Return the player's cumulative spread so far this tournament.

=cut

sub Spread ($) { 
  my $this = shift;
  return ($this->{'spread'} || 0);
  }

=item $n = $p->TaggedName();

Return a formatted version of the player's name, including
their player number.  You should call the wrapper TSH::Utility::TaggedName
unless you are 100% sure that the player pointer is valid.

=cut

sub TaggedName ($) { 
  my $this = shift;
  my $clean_name = $this->Name();
  $clean_name =~ s/,$//; # kludge to allow names ending in digits
  my $fullid = $this->FullID();
  my $team = $config::show_teams ? "/$this->{'etc'}{'team'}[0]" : '';
  defined $this && length($clean_name)
    ? "$clean_name ($fullid$team)"
    : 'nobody';
  }

=item $s = $p->Team();

Get the player's team name.

=cut

sub Team ($) { 
  my $this = shift;
  $this->{'etc'}{'team'}[0] || '';
  }

=item $success = $p->UnpairRound($r0);

Remove pairings for player in round $r0, return success.

=cut

sub UnpairRound ($$) { 
  my $this = shift;
  my $r0 = shift;

  return 0 if $#{$this->{'pairings'}} != $r0;
  my $opp = pop @{$this->{'pairings'}};
  if ($#{$this->{'etc'}{'board'}} >= $r0) {
    $#{$this->{'etc'}{'board'}} = $r0 - 1;
    }
  if ($#{$this->{'etc'}{'p12'}} >= $r0) {
    $#{$this->{'etc'}{'p12'}} = $r0 - 1;
    }
  if ($#{$this->{'scores'}} >= $r0) {
    $#{$this->{'scores'}} = $r0 - 1;
    }
  print "... $this->{'id'} unpaired from $opp.\n";
  return 1;
  }

=item $n = $p->Wins();

Return the player's total wins so far this tournament.

=cut

sub Wins ($) { 
  my $this = shift;
  return ($this->{'wins'} || 0);
  }

=back

=cut

=head1 BUGS

Rather than calling C<Division::Synch()> when the C<.t> file is
loaded (which substantially delays the loading of large files),
the relevant statistics that it computes should be computed only
as needed.

Not all routines call TSH::Division::Dirty(1) when needed yet.

Team() should look more like Board().

First() should check consistency with opponent.

=cut

1;

