#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package UserMessage;

=pod

=head1 NAME

UserMessage - User diagnostic message class

=head1 SYNOPSIS

  my $um = new UserMessage($message_filename);
  $um->SetBriefFilter(\&sub);
  $um->SetExplainFilter(\&sub);
  $um->Show($code, @args);
  $um->Explain();
  $um->Explain($code);
  
=head1 ABSTRACT

This Perl library manages user diagnostic messages.
Messages are stored in a text file to facilitate translation.
Messages may be informational notes, warnings or errors.
Each message has a detailed explanation associated with it,
which the user may request if the brief text is insufficient.

=head1 DESCRIPTION

=over 4

=cut

use Symbol;

sub DefaultFilter ($);
sub Explain($;$);
sub new ($$);
sub SetErrorFilter ($\&);
sub SetExplainFilter ($\&);
sub SetNoteFilter ($\&);
sub SetWarningFilter ($\&);
sub Show($$@);

=item DefaultFilter($text);

The default filter for showing any kind of message just calls print.

=cut

sub DefaultFilter ($) {
  my $this = shift;
  print $this;
  }

=item $success = $um->Explain()
=item $success = $um->Explain($code))

Give the detailed text for an error message.  If no code is
specified, then give the explanation for the last message
issued.  If none has been issued, do nothing.

=cut

sub Explain ($;$) {
  my $this = shift;
  my $code = (shift || $this->{'lastcode'});
  if ($code) {
    my $s = $this->{'messages'}{$code};
    if (defined $s) {
      &{$this->{'explainfilter'}}($code, $s->{'detail'});
      return 1;
      }
    else {
      return 0;
      }
    }
  else {
    return 0;
    }
  }

=item $um = new UserMessage($filename);

Create a new UserMessage object and initialize it based on the
message file indicated.

=cut

sub new ($$) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $this = {
    'errorfilter' => \&DefaultFilter,
    'explainfilter' => \&DefaultFilter,
    'filename' => shift,
    'lastcode' => undef,
    'messages' => {},
    'notefilter' => \&DefaultFilter,
    'warningfilter' => \&DefaultFilter,
    };
  
  my $fh = gensym;
  unless (open($fh, "<$this->{'filename'}")) {
    warn "Can't open $this->{'filename'}: $!";
    return;
    }
  local($/) = '';
  while (<$fh>) {
    my (%data) = map { split(/=/, $_, 2) } split(/\s*\n+/);
    die "Message database entry has no message code\nAborting" 
      unless exists $data{'code'};
    die "Message database has duplicate entries for [$data{'code'}]\n"
      if exists $this->{'messages'}{$data{'code'}};
    $this->{'messages'}{$data{'code'}} = \%data;
    for my $field (qw(brief detail type)) {
      die "Message database entry for $data{'code'} is missing required information ($field).\n" 
        unless exists $data{$field};
      }
    die "Message database is corrupt: [$data{'code'}] has bad message type: $data{'type'}\n"
      unless $data{'type'} =~ /^(?:note|warning|error)$/;
    }
  close($fh);

  bless($this, $class);
  return $this;
  }

=item SetErrorFilter(\&sub);

Set the filter for displaying errors.

=cut

sub SetErrorFilter ($\&) {
  my $this = shift;
  my $sub = shift;
  $this->{'errorfilter'} = $sub;
  }

=item SetExplainFilter(\&sub);

Set the filter for displaying explanationss.

=cut

sub SetExplainFilter ($\&) {
  my $this = shift;
  my $sub = shift;
  $this->{'explainfilter'} = $sub;
  }

=item SetNoteFilter(\&sub);

Set the filter for displaying notes.

=cut

sub SetNoteFilter ($\&) {
  my $this = shift;
  my $sub = shift;
  $this->{'notefilter'} = $sub;
  }

=item SetWarningFilter(\&sub);

Set the filter for displaying warnings.

=cut

sub SetWarningFilter ($\&) {
  my $this = shift;
  my $sub = shift;
  $this->{'warningfilter'} = $sub;
  }

=item $um->Show($code, @args)

Give the brief text for the message, filling in sprintf
fields from @args.  Use one of the output filters depending on
the message type.

=cut

sub Show ($$@) {
  my $this = shift;
  my $code = shift;
  my $datap = $this->{'messages'}{$code};
  unless ($datap) {
    warn "Unknown message code: $code";
    return;
    }
  my $s = $this->{'messages'}{$code};
  my $type = $s->{'type'};
  die "No message type for code $code: $type"
    unless defined $type;
  die "Bad message type for code $code: $type"
    unless $type =~ /^(?:note|warning|error)$/;
  $this->{'lastcode'} = $code;
  &{$this->{$type.'filter'}}($code, $type, sprintf($s->{'brief'}, @_));
  }

=back

=cut

=head1 BUGS

None reported so far.

=cut

1;
