#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Log;

use strict;
use warnings;

use Symbol;
use TSH::Utility;

=pod

=head1 NAME

TSH::Log - Record displayed information in text and HTML format

=head1 SYNOPSIS

  my $log = new TSH::Log($division, $type, $round, $notitle);
  $log->Write($text, $html);
  $log->Close();

=head1 ABSTRACT

This class manages the copying of information displayed on the
console to text and HTML files.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$;$$);
sub Close($);
sub new ($$$;$$);
sub MakeHTMLIndex ();
sub ReportHeader ($$$);
sub ReportTrailer ();
sub Write ($$$);

=item $log->Close()

Close a log and all files associated with it.

=cut

sub Close ($) {
  my $this = shift;
  $this->Write('', "</table>\n");
  {
    my ($text, $html) = ReportTrailer;
    $this->Write($text, $html);
  }
  for my $fh (qw(text_fh html_fh subhtml_fh)) {
    if ($this->{$fh}) {
      close($this->{$fh});
      $this->{$fh} = undef;
      }
    }
  MakeHTMLIndex;
  }

=item $log->initialise($division, $type, $round, $notitle)

Initialise the log object, create files.

=cut

sub initialise ($$$;$$) {
  my $this = shift;
  my $dp = shift;
  my $type = lc shift;
  my $round = lc (shift||'');
  my $notitle = shift;
  my $Type = ucfirst $type;
  my ($fh, $fn, %fhs);
  my $round_000 = ($round =~ /^\d+$/) ? sprintf("-%03d", $round) : '';

  my $dname = '';
  my $dname_hyphen = '';
  if ($dp) {
    $dname = $dp->Name();
    $dname_hyphen = "$dname-";
    }

  unless ($config::no_text_files) {
    $fh = gensym;
    $fn = "$dname_hyphen$type.doc";
    open($fh, ">$config::root_directory/$fn") or TSH::Utility::Error "Can't create $fn: $!\n";
    $this->{'text_fh'} = $fh;
    }

  $fh = gensym;
  $fn = "$dname_hyphen$type.html";
  open($fh, ">$config::root_directory/$fn") or TSH::Utility::Error "Can't create $fn: $!\n";
  $this->{'html_fh'} = $fh;

  $fh = gensym;
  $fn = "${config::html_directory}$dname_hyphen$type$round_000.html";
  open($fh, ">$fn") or TSH::Utility::Error "Can't create $fn: $!\n";
  $this->{'subhtml_fh'} = $fh;

  if ($type ne 'grid') {
    my $title = $Type;
    $title = "Round $round $title" if $round =~ /^\d+$/;
    if ($main::gTournament->CountDivisions() > 1) { 
      $title = $dp ? "Division $dname $title": $title;
      };
    my ($text, $html) = ReportHeader $title, $type, $notitle;
    $this->Write($text, $html);
    }
  my $type_ = $type; $type_ =~ s/-/_/g;
  $this->Write('', <<"EOF");
<table class="$type_" align=center cellspacing=0>
<tr class=top1>
EOF

  return $this;
  }

=item MakeHTMLIndex();

Used internally to update the HTML index of HTML files.

=cut

sub MakeHTMLIndex () {
  my $dir;
  unless (opendir($dir, $config::html_directory)) {
    TSH::Utility::Error "Can't open $config::html_directory: $!\n";
    return;
    }
  my @rounds;
  my %divisions;
  for my $file (readdir $dir) {
    next unless $file =~ /^(\w+)-([-a-z]+)(?:-(\d+))?\.html$/i;
    next if $file =~ /scorecard\.html/;
    my $div = lc $1;
    my $type = $2;
    my $round = ($3 || 0);
    next unless $main::gTournament->GetDivisionByName($div);
    $divisions{$div}++;
    $rounds[$round]{$div}{$type} = $file;
    }
  closedir $dir;
  my (@divisions) = sort keys %divisions;
  
  my $fh;
  unless (open $fh, ">$config::html_directory/index.html") {
    TSH::Utility::Error "Can't create HTML index file: $!\n";
    return;
    }
  print $fh (ReportHeader "Event Coverage Index", 'index', 0)[1];
  print $fh "<table class=index align=center>\n";
  if (@divisions > 1) {
    print $fh "<tr><th class=empty>&nbsp;</th>";
    for my $div (@divisions) {
      print $fh "<th class=division>Div. \U$div\E</th>\n";
      }
    print $fh "</tr>\n";
    }
  for my $round (0..$#rounds) {
    my $rp = $rounds[$round];
    next unless $rp;
    if ($round) {
      print $fh "<tr><td class=round>Round $round</td>\n";
      }
    else {
      print $fh "<tr><td class=round>&nbsp;</td>\n";
      }
    for my $div (@divisions) {
      print $fh "<td class=links>";
      my $rdp = $rp->{$div};
      if ($rdp) {
	for my $type (sort keys %$rdp) {
	  print $fh qq(<div class=link><a href="$rdp->{$type}">$type</a></div>\n);
	  }
        }
      print $fh "</td>";
      }
    print $fh "</tr>\n";
    }
  print $fh "</table>\n";
  print $fh ReportTrailer;
  close $fh;
   }

=item $d = new Log;

Create a new Log object.  

=cut

sub new ($$$;$$) { return TSH::Utility::new(@_); }

=item ($text, $html) = ReportHeader $title, $type, $bare

Render a standard report header.

=cut

sub ReportHeader ($$$) {
  my $title = shift;
  my $type = shift;
  my $bare = shift;
  my $html = '';
  my $type_ = $type; $type =~ s/-/_/;
  $title =~ s/Alpha-pairings/Alpha Pairings/;
  $html .= <<"EOF";
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN"
	"http://www.w3.org/TR/REC-html40/loose.dtd">
<html><head><title>$title</title>
<meta http-equiv="Content-Type" content="text/html; charset=iso-8859-1">
<link rel="stylesheet" href="tsh.css" type="text/css">
</head>
<body class="$type_">
EOF
  $html .= $config'html_top if $config'html_top;
  if ($bare) { return ('', $html); }
  else { 
    return ($title =~ /Alpha/ ? '' : "$title\n\n", "$html<h1>$title</h1>\n"); 
    }
  }

=item ($text, $html) = ReportTrailer $title, $type, $bare

Render a standard report trailer.

=cut

sub ReportTrailer () {
  return ('', "<p class=notice>This report was generated using <cite>tsh</cite> version $main::gkVersion.  For more information about <cite>tsh</cite>, please ask John Chew.</p></body></html>");
  }

=item $log->Write($text,$html)

Write text to logs and console.

=cut

sub Write ($$$) {
  my $this = shift;
  my $text = shift;
  my $html = shift;
  my $fh;
  print $text;
  $fh = $this->{'text_fh'}; print $fh $text if $fh;
  $fh = $this->{'html_fh'}; print $fh $html if $fh;
  $fh = $this->{'subhtml_fh'}; print $fh $html if $fh;
  }

=back

=cut

=head1 BUGS

It's arguable that Close() should be part of the object destructor.

The filename conventions and indexing don't work well with the
C<ResultsByRound> command.

The options to new() are getting too numerous and should be replaced
by an option hash.

The hack for alpha-pairings should be incorporated into that hash.

=cut

1;
