#!/usr/bin/perl

# Copyright (C) 2006 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Server;

use strict;
use warnings;

use HTTP::Server;
use TSH::Config;
use Net::Domain qw(hostfqdn);

our (@ISA) = qw(HTTP::Server);

=pod

=head1 NAME

TSH::Server - provide web access to the tsh process

=head1 SYNOPSIS

  my $server = new TSH::Server($port) or die;
  $server->Start() or die;
  while ($server->Run()) {}
  $server->Stop();
  
=head1 ABSTRACT

This subclass of HTTP::Server provides single-threaded access to
the tsh process.

=head1 DESCRIPTION

=over 4

=cut

sub new ($@);
sub ConfigItem ($);
sub Content ($);
sub ContentConfig ($;$);
sub ContentDivision ($$);
sub ContentRoot ();
sub HTMLFooter ();
sub HTMLHeader ($);
sub SaveConfigChanges ($);
sub Start ($);

=item $server = new TSH::Server($port);

Create a new TSH::Server object.  

C<port>: port to listen on for new connections.

=cut

sub new ($@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
  my $port = shift;
  my $this = $class->SUPER::new(
    'port' => $port,
    'content' => \&TSH::Server::Content,
    'version' => "tsh/$main::gkVersion",
    );
  bless($this, $class);
  return $this;
  }

=item $error = TSH::Server::ConfigCheck();

Checks to see if the currently loaded configuration can be used to
run tsh in server mode, returns an error message if not.

=cut

sub ConfigCheck () {
  my @problems;
  unless ($config::max_rounds) {
    push(@problems, "The mandatory max_rounds option has not been specified.");
    }
  if (@problems) {
    if (@problems == 1) {
      return "You must correct this configuration before you can proceed. "
        . $problems[0];
      }
    else { die; }
    }
  else { return ''; }
  }

=item $html = TSH::Server::ConfigItem($name);

Returns a line of $html code for use in ContentConfig.

=cut

my (%entities) 
  = ('&' => '&amp;', '>' => '&gt;', '<' => '&lt;', '"' => '&quot;');

sub ConfigItem ($) {
  my $name = shift;
  my $value = eval "\$config::$name";
  my $type = TSH::Config::UserOptionType($name);
  my $help = TSH::Config::UserOptionHelp($name);
  my $html = "<tr><td class=name>$name</td>";
  if (!defined $type) {
    $html .= "<td>unknown config item</td>";
    }
  if ($type eq 'boolean') {
    my $checked = $value ? ' checked': '';
    $html .= qq(<td class=value><input name=$name type=checkbox value=1$checked></td>);
    }
  elsif ($type eq 'integer') {
    $value = 0 unless $value;
    $html .= qq(<td class=value><input name=$name type=text maxlength=10 size=10 value="$value"></td>);
    }
  elsif ($type eq 'string') {
    $value = '' unless defined $value;
    $value =~ s/(["&<>])/$entities{$1}/g;
    $html .= qq(<td class=value><input name=$name type=text maxlength=256 size=40 value="$value"></td>);
    }
  else {
    $html .= "<td>unknown config type: $type</td>";
    }
  if ($help) {
    $html .= "<td class=help>$help</td>";
    }
  $html .= "</tr>";
  return $html;
  }

=item $response = TSH::Server::Content($request);

Generate the content needed to serve an HTTP request.
See HTTP::Server for details.

=cut

my (%url_map) = (
  '/favicon.ico' => '/doc/tsh.ico',
  '/tsh.css' => "/report/tsh.css",
  );

sub Content ($) {
  my $request = shift;
  my $url = $request->URL();
  if ($url !~ /\.(?:css|ico)$/ && (my $error = ConfigCheck()))
    { return ContentConfig($request, $error); }
  # canonicalise
  if ($url =~ /^\/(?:doc)$/) 
    { return HTTP::Server::Redirect("$url/"); }
  $url .= "index.html" if $url =~ /\/$/;
  $url = $url_map{$url} || $url;
  $url =~ s/\/{2,}/\//g;
  while ($url =~ s/\/\.\//\//g) {} 
  while ($url =~ s/\/([^\/]+)\/\.\.\//\//g) { }
  # dispatch division page requests
  if ($url =~ /^\/division\/([^\/]+\/)?index\.html$/) { 
    my $div = $1;
    if (length($div)) { $div =~ s/.$//; }
    return ContentDivision($request, $div); 
    }
  # dispatch root page requests
  if ($url =~ /^\/index\.html$/) { return ContentRoot(); }
  # dispatch config page requests
  if ($url =~ /^\/config\/index\.html$/)
    { return ContentConfig($request); }
  # dispatch documentation requests
  if ($url =~ /^\/(doc\/\w+)\.(css|html|ico)$/) 
    { return HTTP::Server::StaticContent($1, $2); }
  # dispatch report requests
  if ($url =~ /^\/report\/([-\w]+)\.(css|html)$/) 
    { return HTTP::Server::StaticContent("$config::html_directory/$1", $2); }
  # else display error message
  my $html = "<h1>Unrecognized URL: $url</h1>";
  $html .= "<p>Here are the details of the request you sent.</p>";
  $html .= "<h2>Message Start</h2><ul><li>Method: ".($request->Method())."</li><li>URL: ".($request->URL())."</li><li>HTML-Version: ".($request->HTTPVersion())."</li></ul>\n";
  $html .= "<h2>Message Headers</h2><ul>";
  my %headers = $request->Headers();
  for my $key (sort keys %headers) {
    $html .= "<li>\u$key: $headers{$key}</li>";
    }
  $html .= "</ul>";
  $html .= "<h2>Message Body</h2><pre>" . ($request->Body()) . "</pre>";
  $html .= "</body></html>";
  return new HTTP::Message( 'status-code' => 404, 'body' => $html );
  }

=item $response = TSH::Server::ContentConfig($request[, error]);

Generate the content needed to serve a request for '/config/index.html'.

=cut

sub ContentConfig ($;$) {
  my $request = shift;
  my $error = shift;
  my $formp = $request->FormData();
  my $html = HTMLHeader('Configure') . "<h1>tsh Configuration</h1>";
  if ($formp->{'post'}) {
    $html .= SaveConfigChanges $formp;
    $error = ConfigCheck();
    }
  if ($error) {
    $html .= "<div class=failure>$error</div>";
    }
  $html .= "<p class=p1>The following options control <cite>tsh</cite>&rsquo;s behaviour. If you want to change them, click or type appropriately within the table, then click on the Save Changes button below it.</p>";
  $html .= qq(<form method="post" action="/config/" enctype="application/x-www-form-urlencoded">);
  $html .= "<table class=config align=center>";
  for my $key (sort(TSH::Config::UserOptions())) { $html .= ConfigItem $key; }
  $html .= "</table>";
  $html .= "<p class=p2><input type=submit name=post value=\"Save Changes\"></p>";
  $html .= "</form>";
  $html .= "<p class=p1>The following table lists the player divisions in your tournament and the name of the file in which each division&rsquo;s information is stored.</p>";
  $html .= HTMLFooter();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => $html,
    );
  }

=item $response = TSH::Server::ContentDivision($request, $division);

Generate the content needed to serve a request for '/division/DIVNAME'.

=cut

sub ContentDivision ($$) {
  # TODO: do some sort of caching here?
  my $request = shift;
  my $div = TSH::Division::CanonicaliseName(shift);
  my $html = HTMLHeader('Edit');
  my (@dps) = $::gTournament->Divisions();
  my $dp = $::gTournament->GetDivisionByName($div) || $dps[0];
  $div = $dp->Name();
  $html .= "<div class=divlist>";
  $html .= "<span class=label>Division:</span>";
  for my $adp ($::gTournament->Divisions()) {
    my $adiv = $adp->Name();
    if ($div eq $adiv) {
      $html .= qq(<span class=here>$adiv</span>);
      }
    else {
      $html .= qq(<span class=there><a href="/division/$adiv/">$adiv</a></span>); 
      }
    }
  $html .= "</div>";
  $html .= qq(<form method="post" action="/division/$div" enctype="application/x-www-form-urlencoded">);
  $html .= qq(<table class=dedit cellspacing=0>);
  $html .= "<tr>";
  $html .= qq(<th class=id>ID</th>);
  $html .= qq(<th class=name>Name</th>);
  for my $round (1..$config::max_rounds) {
    $html .= qq(<th class=round>Round $round</th>); 
    }
  $html .= "</tr>";
  my (@players) = $dp->Players();
  my $pfmt = '%0' . length(scalar(@players)) . 'd';
  for my $pp (@players) {
    my $id = $pp->ID();
    my $name = $pp->Name();
    $html .= sprintf(qq(<tr><td class=id>$pfmt</td><td class=name>%s</td>),
      $id, $name);
    for my $round (1..$config::max_rounds) {
      my $round0 = $round - 1;
      my $ms = $pp->Score($round0);
      my $oid = $pp->OpponentID($round0);
      my $on = '';
      if ($oid) { 
	$oid = sprintf($pfmt, $oid);
	my $op = $pp->Opponent($round0);
	if ($op) {
	  $on = $pp->Opponent($round0)->Initials();
	  }
        }
      else { $oid = '-'; }
      my $os = $pp->OpponentScore($round0);
      my $scores;
      if ((defined $ms) && (defined $os)) {
	$scores = "$ms-$os";
        }
      elsif ($oid ne '-') {
	$scores = ((defined $ms) ? $ms : '?') . '-' . 
	  ((defined $os) ? $os : '?');
        }
      elsif (defined $ms) {
	$scores = sprintf("%+d", $ms);
	$oid = 'bye';
        }
      else {
	$on = '??';
	$scores = '?';
        }
      $html .= qq(<td class=round><span class=opp>$oid$on</span><span class=score>$scores</span></td>);
      }
    $html .= qq(</tr>);
    }
  $html .= qq(</table>);
  $html .= qq(</form>);
  $html .= HTMLFooter();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => $html,
    );
  }

=item $response = TSH::Server::ContentRoot();

Generate the content needed to serve a request for '/'.

=cut

sub ContentRoot () {
  my $html = HTMLHeader('Home');
  $html .= "<div class=divlist>";
  warn "root\n";
# $html .= "<span class=label>Division:</span>";
# for my $dp ($::gTournament->Divisions()) {
#   my $div = $dp->Name();
#   $html .= qq(<a href="/division/$div/">$div</span>); 
#   }
# $html .= "</div>";
  $html .= HTMLFooter();
  return new HTTP::Message(
    'status-code' => 200,
    'body' => $html,
    );
  }

=item $html = TSH::Server::HTMLFooter();

Return an HTML footer.

=cut

sub HTMLFooter () {
  return "</body></html>";
  }

=item $html = TSH::Server::HTMLHeader($section);

Return an HTML header suitable for displaying in section $section.

=cut

my (@navbardata) = (
  {'label' => 'Home', 'url' => '/'},
  {'label' => 'Configure', 'url' => '/config/'},
  {'label' => 'Edit', 'url' => '/division/'},
  {'label' => 'Help', 'url' => '/doc/'},
  {'label' => 'Reports', 'url' => '/report/'},
  );

sub HTMLHeader ($) {
  my $section = shift;
  my $html = <<"EOF";
<html>
<head>
<title>tsh $main::gkVersion</title>
<link rel=stylesheet href="/report/tsh.css">
</head>
<body>
EOF
  $html .= "<div class=topbar>";
  $html .= "<span class=label>tsh $::gkVersion:</span> ";
  $html .= "<span class=event>$config::event_name</span> "
    if $config::event_name;
# $html .= qq(<a href="notyet">change event</a>);
  $html .= "</div>";
  $html .= "<div class=seclist>";
  for my $nbd (@navbardata) {
    if ($nbd->{'label'} eq $section) 
      { $html .= "<span class=here>$nbd->{'label'}</span>"; }
    else
      { $html .= qq(<span class=there><a href="$nbd->{'url'}">$nbd->{'label'}</a></span>); }
    }
  $html .= "</div>";
  return $html;

  die "This stuff needs a home";
#   $html .= "</tr>";
#   $html .= <<"EOF";
# <script language="JavaScript">
# function Division (div) {
#   alert(div);
#   }
# </script>
# EOF
  }

=item $html = TSH::Server::SaveConfigChanges($formp);

Saves changes to tsh's configuration as indicated by the form variables
in the hash %$formp.  Returns a message informing the user whether or
not the operation succeeded.

=cut

sub SaveConfigChanges ($) {
  my $formp = shift;
  my $fh;
  my $changed = 0;
  for my $key (TSH::Config::UserOptions()) {
    my $newvalue = $formp->{$key};
    my $type = TSH::Config::UserOptionType($key);
    my $oldvaluep = $config::{$key};
    if ($type eq 'boolean') { 
      if ($newvalue xor $$oldvaluep) {
	$changed = 1;
	$$oldvaluep = $newvalue ? 1 : 0;
        }
      }
    elsif ($type eq 'integer') { 
      $newvalue = $newvalue ? 0+$newvalue : 0; 
      if ($newvalue != ($$oldvaluep||0)) {
	$changed = 1;
	$$oldvaluep = $newvalue;
        }
      }
    elsif ($type eq 'string') { 
      $newvalue = (defined $newvalue) ? $newvalue : '';
      if ($newvalue ne ((defined $$oldvaluep) ? $$oldvaluep : '')) {
	$changed = 1;
	$$oldvaluep = $newvalue;
        }
      }
    else 
      { return "<div class=failure>Unknown key type ($type) for $key.</div>"; }
    }
  return $config::config->Save();
  }

=item $server->Start() or die;

Start the server by opening the listener socket.

=cut

sub Start ($) {
  my $this = shift;
  print "Starting server...\n";
  my $result = $this->SUPER::Start();
# my $host = hostfqdn(); # takes too long
  my $port = $this->{'port'};
  print <<"EOF";
tsh is now in server mode.  You can connect to the server at 
http://localhost:$port from this machine, or from other machines
by replacing 'localhost' with your machine's network address.
To start tsh in interactive mode, do not include a "config port" line
in your configuration file.
EOF
  }

=back

=cut

1;

