#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package HTTP::Client;

=pod

=head1 NAME

HTTP::Client - HTTP client class

=head1 SYNOPSIS

  use HTTP::Client;
  use HTTP::Message;
  my $c = new HTTP::Client;
  my $m = new HTTP::Message(
    'method' => 'POST',
    'url' => '/cgi-bin/submit-rdata.pl',
    'http-version' => '1.1',
    'message-headers' => {
      'content-type' => 'multipart/form-data',
      'host' => 'localhost',
      },
    );
  $m->FormData('password', 'frobozz');
  $c->Connect("localhost") or die;
  $c->Send($m);
  my $r = $c->Get();
  $c->Close();
  if (my $error = $r->Error()) {
    print "$error\n";
    }
  my $html = $r->Body();
  
=head1 ABSTRACT

This Perl module implements a HyperText Transfer Protocol client.

=head1 DESCRIPTION

=over 4

=cut

use Fcntl qw(F_GETFL F_SETFL O_NONBLOCK);
use Socket;
use IO::Socket::INET;

use HTTP::Message;

sub new ($@);
sub Close ($);
sub Connect ($$);
sub Die ($$);
sub Receive($);
sub Send ($$);
sub SendRaw ($$);

=item $c = new HTTP::Client($key1 => $value1, ...);

Create a new HTTP::Client object.  
All parameter keys are optional.
C<exit_on_error> gives a boolean value specifying whether or not to
exit on any error.

=cut

sub new ($@) {
  my $proto = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $this = {
    'state' => 'init',
    'exit_on_error' => 1,
    };
  my %param = @_;
  for my $key (qw(exit_on_error)) {
    if (exists $param{$key}) { $this->{$key} = $param{$key}; }
    elsif (!exists $this->{$key}) {
      die "Missing required parameter ($key).";
      }
    }
  
  bless($this, $class);
  return $this;
  }

=item $success = $c->Close();

Closes the HTTP connection, which cannot be used again until it is
reopened.  
Fails only if the connection was not open to begin with.

=cut

sub Close ($) {
  my $this = shift;
  
  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'open') 
    { $this->Die("Connection is not open"); return undef; }
  $this->{'state'} = 'init';
  return 1;
  }

=item $success = $c->Connect($hostname);

Opens a connection to the named host.

=cut

sub Connect ($$) {
  my $this = shift;
  my $hostname = shift;

  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'init') 
    { $this->Die("Connection is already open"); return undef; }

  my $sfh = $this->{'socket'} = 
    IO::Socket::INET->new(
      PeerAddr => $hostname,
      PeerPort => 'http(80)',
      Proto => 'tcp',
      Blocking => 1,
      )
    or $this->Die("Couldn't create socket for web connection: $!");
  $this->{'state'} = 'open';
  return 1;
  }

=item $c->Die($message);

Sets the module error message and exits if C<exit_on_error> was set.

=cut

sub Die ($$) { 
  my $this = shift;
  my $message = shift;
  $this->{'error'} = $message;
  $this->{'state'} = 'error';
  print STDERR "$message\n";
  exit 1 if $this->{'exit_on_error'};
  }

=item $message = $c->Receive();

Reads an HTTP message (as a C<HTTP::Message>) from an open 
connection, returns undef or dies on failure.

=cut

sub Receive ($) {
  my $this = shift;

  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'open') 
    { $this->Die("Connection is not open"); return undef; }

  return HTTP::Message->new('handle' => $this->{'socket'});
  }

=item $success = $c->Send($request);

Sends an HTTP message to the server.

=cut

sub Send ($$) {
  my $this = shift;
  my $request = shift;

  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'open') 
    { $this->Die("Connection is not open"); return undef; }

  my $s = $request->ToString();
  $this->SendRaw($s);

  return 1;
  }

=item $success = $c->SendRaw($request);

Writes text to an open connection.

=cut

sub SendRaw ($$) {
  my $this = shift;
  my $request = shift;

  return undef if $this->{'state'} eq 'error';
  if ($this->{'state'} ne 'open') 
    { $this->Die("Connection is not open"); return undef; }

  my $old = select($this->{'socket'});
  $| = 1;
  print $request;
  select $old;
# print "[begin request]\n$request\n[end request]\n";

  return 1;
  }

=back

=cut

1;
