#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::InitFontes;

use strict;
use warnings;

use TSH::Player;
use TSH::Utility qw(Debug DebugOn);

our (@ISA) = qw(TSH::PairingCommand);

=pod

=head1 NAME

TSH::Command::InitFontes - implement the C<tsh> InitFontes command

=head1 SYNOPSIS

  my $command = new TSH::Command::InitFontes;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::InitFontes is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub PairPartiallyPaired ($$$);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  DebugOn 'IF';
  $this->{'help'} = <<'EOF';
Use the InitFontes command to manually add three rounds suitable
for starting a Swiss-paired tournament.
Groups of four players are randomly drawn, one from each quartile,
to form round-robin groups of three.
EOF
  $this->{'names'} = [qw(if initfontes)];
  $this->{'argtypes'} = [qw(NumberOfIFRounds Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $success = $command->PairPartiallyPaired($nrounds, $division);

Try to pair those players who have some pairings in the first $nrounds
rounds amongst themselves, so that the rest can be paired in quads.

=cut

sub PairPartiallyPaired ($$$) {
  my $this = shift;
  my $nrounds = shift;
  my $dp = shift;
  my $tournament = $dp->Tournament();
  if ($nrounds != 3) { 
    $tournament->TellUser('eifnot3', $nrounds);
    return 0;
    }
  # TODO: should check for active players
  my (@ps) = grep { 
    (defined $_->OpponentID(0))
    || (defined $_->OpponentID(1))
    || (defined $_->OpponentID(2)) 
    } $dp->Players();

  Debug 'IF', 'Partly paired: %s', join(',', map($_->{'id'}, @ps));
  for my $r0 (0..2) {
    my (@rps) = grep { !defined $_->OpponentID($r0) } @ps;
    Debug 'IF', 'Need to pair in Rd. %d: %s', $r0+1, join(',', map($_->{'id'}, @rps));
    if (@rps % 2) { $tournament->TellUser('eifpppo', 1); return 0; }
    while (@rps > 4) {
      my $p = shift @rps;
      my $offset = int((3-$r0)*@rps/4);
      my $found = 0;
      for my $i (1..@rps) {
	my $j = ($i + $offset) % @rps;
	my $opp = $rps[$j];
	next if $p->CountRepeats($opp) > 0;
	splice(@rps, $j, 1);
	$dp->Pair($p->ID(), $opp->ID(), $r0, 0);
	$found = 1;
	Debug 'IF', '%d vs %d on try %d, %d left', $p->ID(), $opp->ID(), $i, scalar(@rps);
	last;
	}
      unless ($found) {
	$tournament->TellUser('eifstuck', "can't pair #$p->{'id'} $p->{'name'}");
	return 0;
	}
      }
    Debug 'IF', 'Still left: %s', join(',', map($_->{'id'}, @rps));
    if (@rps == 2) {
      my $p = shift @rps;
      my $opp = shift @rps;
      if ($p->CountRepeats($opp) > 0) {
	$tournament->TellUser('eifstuck', "$p->{'id'} and $opp->{'id'} have already played");
	return 0;
        }
      $dp->Pair($p->ID(), $opp->ID(), $r0, 0);
      Debug 'IF', '%d vs %d ok', $p->ID(), $opp->ID();
      }
    # try 1-3 2-4
    if ($rps[0]->CountRepeats($rps[2]) == 0
     && $rps[1]->CountRepeats($rps[3]) == 0) {
      $dp->Pair($rps[0]->ID(), $rps[2]->ID(), $r0, 0);
      $dp->Pair($rps[1]->ID(), $rps[3]->ID(), $r0, 0);
      }
    # try 1-4 2-3
    elsif ($rps[0]->CountRepeats($rps[3]) == 0
     && $rps[1]->CountRepeats($rps[2]) == 0) {
      $dp->Pair($rps[0]->ID(), $rps[3]->ID(), $r0, 0);
      $dp->Pair($rps[1]->ID(), $rps[2]->ID(), $r0, 0);
      }
    # try 1-2 3-4
    elsif ($rps[0]->CountRepeats($rps[1]) == 0
     && $rps[2]->CountRepeats($rps[3]) == 0) {
      $dp->Pair($rps[0]->ID(), $rps[1]->ID(), $r0, 0);
      $dp->Pair($rps[2]->ID(), $rps[3]->ID(), $r0, 0);
      }
    else {
      $tournament->TellUser('eifstuck', "Can't pair last four");
      return 0;
      }
    }
  $dp->Dirty(1);
  $dp->Update();
  return 1;
  }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($nrounds, $dp) = @_;
  if ($dp->LastPairedRound0() != -1) {
    if ($config::allow_gaps) {
      if (!$this->PairPartiallyPaired($nrounds, $dp)) { return 0; }
      }
    else {
      $tournament->TellUser('eifhasp');
      return 0; 
      }
    }
  $tournament->TellUser('iifok', $dp->Name());

  # calculate pairings
  if ($nrounds == 3) {
    my (@rboard) = (0,0,0);
    my @rrs = ();
    my $sortedp = $dp->GetUnpairedRound(0);
    TSH::Player::SpliceInactive @$sortedp, 3, 0;
    @$sortedp = TSH::Player::SortByInitialStanding @$sortedp;
    my $np = $#$sortedp + 1;
    if ($np < 4) { $tournament->TellUser('eifneed4', $np); return; }
    if ($np % 4) { # players don't divide exactly into quads
      my @oddballs;
      my @formats;
      if ($np % 4 == 1) { # number of players is 1 (mod 4)
        # pick five players one from each quintile
	# TODO: think about avoiding giving byes to top seeds
        for (my $section=4; $section>=0; $section--) {
          push(@oddballs, splice(@$sortedp, int($np*($section+rand(1))/5),1));
          }
        @formats = ([3,4,undef,0,1],[1,0,3,2,undef],[undef,2,1,4,3]);
        }
      elsif ($np % 4 == 2) { # number of players is 2 (mod 4)
        # pick six players one from each sextile
        for (my $section=5; $section>=0; $section--) {
          push(@oddballs, splice(@$sortedp, int($np*($section+rand(1))/6),1));
          }
        @formats = ([5,2,1,4,3,0],[3,4,5,0,1,2],[1,0,3,2,5,4]);
        }
      elsif ($np % 4 == 3) { # number of players is 3 (mod 4)
	# TODO: think about avoiding giving byes to top seeds
        # pick three players one from each third
        for (my $section=2; $section>=0; $section--) {
          push(@oddballs, splice(@$sortedp, int($np*($section+rand(1))/3),1));
          }
        @formats = ([2,undef,0],[1,0,undef],[undef,2,1]);
        }
      Debug 'IF', 'Odd players: %s', join(',', map($_->{'id'}, @oddballs));
      for my $r0 (0..2) {
	my $format = $formats[$r0];
	for my $i (0..$#$format) {
	  my $opp;
	  my $oid;
	  my $p = $oddballs[$i];
	  my $pid = $p->ID();
	  if (defined(my $oii = $format->[$i])) {
	    $opp = $oddballs[$oii];
	    $oid = $opp->ID();
	    }
	  else {
	    $opp = undef;
	    $oid = 0;
	    }
	  next if $pid < $oid;
	  Debug 'IF', "$pid vs $oid in %d", $r0+1;
	  $dp->Pair($pid, $oid, $r0, 0);
	  if ($oid) {
	    $rboard[$r0]++;
	    $oddballs[$i]->Board($r0, $rboard[$r0]);
	    $opp->Board($r0, $rboard[$r0]) if $opp;
	    }
	  }
	}
      }
    # at this point, number of remaining players in @$sortedp is divisible by four.
    if ($#$sortedp % 4 != 3) { die "Assertion failed."; }
    # repeatedly pick one random player from each quartile
    for (my $n4 = int(@$sortedp/4); $n4 > 0; $n4--) {
      my @rr = ();
# print "sortedp:[", join(',', map($_->{'id'}, @$sortedp)), "]\n";
      for (my $quartile = 3; $quartile >= 0; $quartile--) 
        { push(@rr, splice(@$sortedp, $quartile*$n4 + rand($n4), 1)); }
      push(@rrs, \@rr);
      }
    # assign pairings
    for my $rr (@rrs) {
#     print "[", join(',', map($_->{'id'}, @$rr)), "]\n";
      Debug 'IF', 'RR4: %s', join(',', map($_->{'id'}, @$rr));
      if ($#$rr == 3) {
	for my $r0 (0..2) {
	  # RR4 gets paired 14.23, 13.24, 12.34
	  my $format = ([3,2,1,0],[2,3,0,1],[1,0,3,2])[$r0];
          for my $i (0..3) {
	    my $p = $rr->[$i];
	    my $pid = $p->ID();
	    my $opp = $rr->[$format->[$i]];
	    my $oid = $opp->ID();
	    next if $pid < $oid;
	    $dp->Pair($pid, $oid, $r0, 0);
	    $rboard[$r0]++;
	    $p->Board($r0, $rboard[$r0]);
	    $opp->Board($r0, $rboard[$r0]);
            }
          }
        }
      }
    $dp->Dirty(1);
    $dp->Update();
    $tournament->TellUser('idone');
    }
  else { 
    $tournament->TellUser('eifnot3', $nrounds);
    }
  return;
  }

=back

=cut

=head1 BUGS

Makes some inappropriate use of TSH::Player internals.

Should work for any number of rounds, using Clark pairings as a default ordering.

=cut

1;
