#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::PRiZes;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::PRiZes - implement the C<tsh> PRiZes command

=head1 SYNOPSIS

  my $command = new TSH::Command::PRiZes;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::PRiZes is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);
sub ShowPrize ($$$$);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display a prize table.
EOF
  $this->{'names'} = [qw(prz prizes)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;

  unless (@config::prizes) {
    push(@config::prizes, 
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>1, 'value'=>'$3500', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>2, 'value'=>'$750', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>3, 'value'=>'$500', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>4, 'value'=>'$250', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>5, 'value'=>'SamBoard', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>6, 'value'=>'Pocket SamTimer', },
      { 'division'=>'A', 'type'=>'overseed', 'subtype'=>1, 'value'=>'$200', 'groupname' => 'Group AB', 'members' => [ 1..23] },
      { 'type' => 'signup', 'subtype' => 'Group A High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'Group A Best Computer Word', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'pick', 'groupname' => 'Group A', 'members' => [ 1..14] },
      { 'type' => 'signup', 'subtype' => 'Group B High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'Group B Best Computer Word', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'pick', 'groupname' => 'Group B', 'members' => [ 15..23] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$350', 'groupname' => 'Group C', 'members' => [ 24..36 ] },
      { 'type' => 'signup', 'subtype' => 'Group C High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'Group C Best Computer Word', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'pick', 'groupname' => 'Group C', 'members' => [ 24..36] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$320', 'groupname' => 'Group D', 'members' => [ 37..50 ] },
      { 'type' => 'signup', 'subtype' => 'Group D High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'Group D Best Computer Word', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'pick', 'groupname' => 'Group D', 'members' => [ 37..50] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$320', 'groupname' => 'Group E', 'members' => [ 51..64 ] },
      { 'type' => 'signup', 'subtype' => 'Group E High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'Group E Best Computer Word', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'pick', 'groupname' => 'Group E', 'members' => [ 51.. 64 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$280', 'groupname' => 'Group F', 'members' => [ 65..78 ] },
      { 'type' => 'signup', 'subtype' => 'Group F High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'Group F Best Computer Word', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'pick', 'groupname' => 'Group F', 'members' => [ 65..78] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$270', 'groupname' => 'Group G', 'members' => [ 79..92 ] },
      { 'type' => 'signup', 'subtype' => 'Group G High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'Group G Best Computer Word', 'value' => 'pick' },
      { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'pick', 'groupname' => 'Group G', 'members' => [ 79..92] },
    );
    }
  if (!defined $config::max_rounds) {
    $tournament->TellUser('eneed_max_rounds');
    return;
    }
  my $partial = 0;
  dp:for my $dp ($tournament->Divisions()) {
    for my $p ($dp->Players()) {
      for my $r0 (0..$config::max_rounds-1) {
	if (!defined $p->Score($r0)) { $partial = 1; last dp; }
	}
      }
    }

  my $logp = new TSH::Log(undef, 'prizes', undef);
  $logp->Write(sprintf("%3s %s\n\n", '#', 'Prize, Winner'), <<'EOF');
<th class=number>#</th>
<th class=description>Description</th>
<th class=value>Value</th>
<th class=winner>Winner(s)</th>
</tr>
EOF
  $logp->Write("Based on PARTIAL results\n\n", "<tr><td colspan=4 class=warning>Based on PARTIAL results</td></tr>\n");
  for my $p0 (0..$#config::prizes) {
    my $prize = $config::prizes[$p0];
    my $p1 = $p0+1;
    $logp->Write(sprintf("%3d ", $p1), "<tr><td class=number>$p1</td>");
    $this->ShowPrize($tournament, $logp, $prize);
    $logp->Write("\n", "</tr>\n");
    }
  $logp->Close();
  return 0;
  }

=item $this->ShowPrize($tournament, $logp, $prize);

Add the data for the given prize to the log.

=cut

sub ShowPrize ($$$$) {
  my $this = shift;
  my $tournament = shift;
  my $logp = shift;
  my $prize = shift;
  my $type = $prize->{'type'} || '';
  if ($type eq 'rank') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $value = $prize->{'value'} || '?';
    for my $p ($dp->Players()) {
      if ($p->RoundRank($lastr0) == $subtype) {
	my $pname = sprintf("%s %g-%g %+d", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread());
	$logp->Write("Rank: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Rank: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      }
    }
  elsif ($type eq 'grouprank') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [];
    my $groupname = $prize->{'groupname'} || 'Group ?';
    my $value = $prize->{'value'} || '?';
    my $lastw = 0;
    my $lastl = 0;
    my $lasts = 0;
    my $rank = 0;
    my (@sorted) = TSH::Player::SortByCurrentStanding grep { defined $_ } 
      map { $dp->Player($_) } @$members;
    for my $i (0..$#sorted) {
      my $p = $sorted[$i];
      my $w = $p->Wins();
      my $l = $p->Losses();
      my $s = $p->Spread();
      if ($lastw != $w || $lastl != $l || $lasts != $s) {
	$lastw = $w;
	$lastl = $l;
	$lasts = $s;
	$rank = $i+1;
        }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %g-%g %+d (overall rank %d)", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread(), $p->RoundRank($lastr0));
	$logp->Write("$groupname Rank: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>$groupname Rank: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'highloss') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [];
    my $groupname = $prize->{'groupname'} || 'Group ?';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = grep { defined $_ } map { $dp->Player($_) } @$members;
    for my $p (@players) {
      my $hl = -9999;
      for my $r0 (0..$lastr0) {
	my $ms = $p->Score($r0);
	my $os = $p->OpponentScore($r0);
	next unless defined $ms;
	next unless defined $os;
	next unless $ms < $os;
	$hl = $ms if $hl < $ms;
        }
      $p->{'xhl'} = $hl;
      }
    @players = sort { $b->{'xhl'} <=> $a->{'xhl'} } @players;
    my $lastv = -9999;
    my $rank = 0;
    for my $i (0..$#players) {
      my $p = $players[$i];
      my $v = $p->{'xhl'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %d", $p->TaggedName(), $p->{'xhl'});
	$logp->Write("$groupname High Loss: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>$groupname High Loss: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'overseed') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [];
    my $groupname = $prize->{'groupname'} || 'Group ?';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = sort { $a->RoundRank($lastr0)-$a->{'id'} <=>
      $b->RoundRank($lastr0)-$b->{'id'} } map { $dp->Player($_) } @$members;
    my $lastv = -@players;
    my $rank = 0;
    for my $i (0..$#players) {
      my $p = $players[$i];
      my $v = $p->RoundRank($lastr0) - $p->{'id'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %g-%g %+d (overall rank %d)", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread(), $p->RoundRank($lastr0));
	$logp->Write("$groupname OverSeed: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>$groupname OverSeed: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'signup') {
    my $subtype = $prize->{'subtype'} || '';
    my $value = $prize->{'value'} || '?';
    $logp->Write("$subtype; Value: $value; Winner: ",
      "<td class=description>$subtype</td>"
      ."<td class=value>$value</td>"
      ."<td class=name></td>");
    }
  }

=back

=cut

=head1 BUGS

DoNSARatings() makes unauthorized use of TSH::Player internals.

This was copied and pasted at great haste and ought to be checked
for duplicate code against RoundStandings and RATings.

Should use new ratings code to autosplit tournaments.

Should always use CalculateSplitRatings.  Require the specification
of config max_rounds, and throw out all the old code.

=cut

1;
