#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RATings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RATings - implement the C<tsh> RATings command

=head1 SYNOPSIS

  my $command = new TSH::Command::RATings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RATings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub DoABSPRatings ($$);
sub DoNSARatings ($$);
sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item DoABSPRatings($tournament, $dp)

Display ratings estimates as calculated by the Association of
British Scrabble Players.

=cut

sub DoABSPRatings ($$) {
  my $tournament = shift;
  my $dp = shift;
  my $datap = $dp->{'data'};

  eval 'require ABSP';
  if ($@) {
    $tournament->TellUser('enomod', 'ABSP.pm');
    return;
    }
  ABSP::CalculateRatings(\@$datap);

  my $logp = new TSH::Log($dp, 'ratings', $dp->MostScores());
  $logp->Write("Rank  Won-Lost Spread InR  ToR  +-R Player\n\n", <<'EOF');
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
<th class=rating>Input<br>Rating</th>
<th class=rating>Tournament<br>Rating</th>
<th class=rating>Rating<br>Change</th>
<th class=name>Name</th>
</tr>
EOF

  my $lastw = -1; my $lasts = 0; my $lastl = -1; my $rank = 0; my $i = 0;
  for my $p (TSH::Player::SortByCurrentStanding $dp->Players()) {
    next unless $p->Active();
    # calculate rank
    my $wins = $p->Wins();
    my $losses = $p->Losses();
    my $spread = $p->Spread();
    my $rating = $p->Rating();
    my $newr = $p->NewRating();
    $i++;
    if ($wins != $lastw || $spread != $lasts || $losses != $lastl) 
      { $lastw = $wins; $lasts = $spread; $rank = $i; $lastl = $losses; }
    my (@fields) = ($rank, $wins, $losses,
	$spread, $rating, $newr,
	$rating ? sprintf("%+3d", $newr-$rating) : '',
	$p->TaggedName());
    $logp->Write(
      sprintf("%4d %4.1f-%4.1f %+5d  %3s %3s %3s %s\n", @fields),
      sprintf(<<'EOF', @fields));
<tr>
<td class=rank>%d</td>
<td class=wl>%4.1f-%4.1f</td>
<td class=spread>%+d</td>
<td class=rating>%d</td>
<td class=rating>%d</td>
<td class=rating>%s</td>
<td class=name>%s</td>
</tr>
EOF
    }
  $logp->Close();
  0;
  }

=item DoNSARatings($tournament, $dp)

Display ratings estimates as calculated by the National Scrabble
Association.

=cut

sub DoNSARatings ($$) {
  my $tournament = shift;
  my $dp = shift;

  # prepare CalculateRatings arguments - ugh
  my (@ps);
  for my $p ($dp->Players()) {
    my $id = $p->ID();
#   print $id-1, ' => ', $p->Name(), "\n";
    $ps[$id-1] = {
      'ewins1' => ($p->{'ewins1'}||0),
      'ewins2' => ($p->{'ewins2'}||0),
      'name' => $p->Name(),
      'oldr' => $p->Rating(),
      'opps' => [ map { ($_||0)-1 } @{$p->{'pairings'}} ],
      'rgames' => $p->{'ratedgames'},
      'scores' => $p->{'scores'},
      'totalg' => 100,
      'id' => $id,
      };
    }
  for my $lib (qw(ratings ratings2)) {
    eval 'require "$lib.pl"';
    if ($@) {
      $tournament->TellUser('enomod', "$lib.pl");
      return;
      }
    }
  if ($config::rating_system && $config::rating_system =~ /lct/i) {
    &ratings2::UseClubMultipliers(1);
    }
  # TODO: support triply split ratings
# for my $i (0..$#ps) { my $p = $ps[$i]; print $i, '-'; print $p->{'name'}, ' '; } print "\n";
# &ratings2'CalculateRatings(\@ps, 'oldr', 1, 'newr', 10000, 'ewins');
  &ratings2::CalculateRatings(\@ps, 'oldr', 1, 'midr', $config::split1, 'ewins1');
  &ratings2::CalculateRatings(\@ps, 'midr', $config::split1+1, 'newr', 10000, 'ewins2');

  my $ms1 = $dp->{'maxs'}+1;
  my $logp = new TSH::Log($dp, 'ratings', $ms1);
  $logp->Write("Rank  Won-Lost Spread OldR NewR Delta Player\n\n", <<'EOF');
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
<th class=rating>Old<br>Rating</th>
<th class=rating>New<br>Rating</th>
<th class=rating>Rating<br>Change</th>
<th class=name>Name</th>
EOF
  if ($dp->LastPairedRound0()+1 > $dp->MostScores()) {
    $logp->Write('', "<th class=oppname>Next Opponent</td></tr>\n");
    }
  $logp->Write('', "</tr>");

  my $lastw = -1; my $lasts = 0; my $lastl = -1; my $rank = 0; my $i = 0;
  for my $p (TSH::Player::SortByCurrentStanding $dp->Players()) {
    next unless $p->Active();
    my $wins = $p->Wins();
    my $losses = $p->Losses();
    my $spread = $p->Spread();
    $i++;
    if ($wins != $lastw || $spread != $lasts || $losses != $lastl) 
      { $lastw = $wins; $lasts = $spread; $rank = $i; $lastl = $losses; }
    my $newr = $ps[$p->ID()-1]{'newr'};
    my (@fields) = ($rank, $wins, $losses,
	$spread,
	$p->Rating(), 
	$newr,
	$p->Rating() ? sprintf("%+5d", $newr-$p->{'rating'}) : '',
	$p->TaggedName());
    $logp->Write(
      sprintf("%4d %4.1f-%4.1f %+5d  %4d %4d %5s %s\n", @fields),
      sprintf(<<'EOF', @fields));
<tr>
<td class=rank>%d</td>
<td class=wl>%4.1f-%4.1f</td>
<td class=spread>%+d</td>
<td class=rating>%d</td>
<td class=rating>%d</td>
<td class=rating>%s</td>
<td class=name>%s</td>
EOF
    if (my $opp = $p->Opponent($p->CountScores())) {
      $logp->Write('',
	sprintf("<td class=oppname>%s</td></tr>\n",
	  $opp->TaggedName()
	  )
	);
      }
    $logp->Write('', "</tr>");
    }
  $logp->Close();
  0;
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display ratings estimates based on current standings
within a division.
EOF
  $this->{'names'} = [qw(rat ratings)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($dp) = @_;

  if ($config::rating_system && $config::rating_system =~ /^absp$/i) {
    DoABSPRatings $tournament, $dp;
    }
  else {
    DoNSARatings $tournament, $dp;
    }
  return 0;
  }

=back

=cut

=head1 BUGS

DoNSARatings() makes unauthorized use of TSH::Player internals.

=cut

1;
