#!/usr/bin/perl

# Copyright (C) 2006 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::SUBMIT;

use strict;
use warnings;

use TSH::Utility;
use HTTP::Client;
use HTTP::Message;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::SUBMIT - implement the C<tsh> SUBMIT command

=head1 SYNOPSIS

  my $command = new TSH::Command::SUBMIT;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::SUBMIT is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);
sub SubmitNSA ($$);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to submit tournament data for ratings calculation
over the Internet.
EOF
  $this->{'names'} = [qw(submit)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  if ($config::rating_system && $config::rating_system =~ /absp/) {
    print "ABSP rating submission will not be implemented until someone asks for it.\n";
    return 0; 
    }
  $this->SubmitNSA($tournament);
  }

=item $command->SubmitNSA($tournament);

Submits ratings data to the NSA.

=cut

sub SubmitNSA ($$) { 
  my $this = shift;
  my $tournament = shift;
  for my $key (qw(director_name event_date event_name)) {
    unless ($config::{$key} && ${$config::{$key}}) {
      print "You must specify config $key to use this command.\n";
      return 0;
      }
    }
  my $http = new HTTP::Client('exit_on_error' => 0);
  my $message = new HTTP::Message(
    'method' => 'POST',
    'url' => '/cgi-bin/submit-rdata.pl',
    'headers' => {
      'content-type' => 'multipart/form-data',
#     'host' => 'localhost',
      'host' => 'www.scrabble-assoc.com',
      },
    );
  my $note = $config::ratings_note || '';
  $note .= " Automatically submitted from tsh $::gkVersion.";
  my $rdata = '';
  my @unrated_flags;
  for my $dp ($tournament->Divisions()) {
    $rdata .= "#division " . $dp->Name() . "\n";
    my $fn = TSH::Config::MakeRootPath($dp->File());
    my $fh = TSH::Utility::OpenFile("<", $fn) or die "Can't open $fn: $!\n";
    local($/) = undef;
    $rdata .= <$fh>;
    close($fh);
    for my $pp ($dp->Players()) {
      next if $pp->Rating();
      my $n = int(@unrated_flags/4);
      my $name = $pp->Name();
      $name =~ s/,//;
#     print "unrated: $name\n";
#     push(@unrated_flags, "unrated$n", $name, "unrated_ok_$n", "yes");
      }
    }
  my $event_type = ($config::rating_system && $config::rating_system =~ /lct/)
    ? 'LCT' : 'ORT';
  $message->FormData({
    'password' => 'satire',
    'dirname' => $config::director_name,
    'evtname' => $config::event_name,
    'evtdate' => $config::event_date,
    'evttype' => $event_type,
    'rdatafile' => { 'type'=>'file', 'filename'=>'all.t', 'data'=>$rdata },
    'Confirm' => 1,
    @unrated_flags,
    });
  $http->Connect('www.scrabble-assoc.com') or return;
# $http->Connect('localhost') or return;
  $http->Send($message);
# print "Sending:\n---\n", $message->ToString(), "\n---\n";
  my $response = $http->Receive();
  my $status = $response->StatusCode();
  my $body = $response->Body();
# print "Received:\n---\n$body\n---\n";
  if ($status ne '200') {
    print "The submission URL is unreachable ($status).  Either you are having network\nproblems or the server is down.\n";
    }
  elsif ($body =~ /<h1>Software error|red>Unexpected Error/) {
    print "The submission page is reporting a software error.  Please contact John Chew.\n";
    }
  elsif ($body =~ /<p class=report>\n([^\0]*?)<\/p>/) {
    my $reply = $1;
    $reply =~ s/<br>/\n/g; 
    $reply =~ s/\n{2,}/\n/g;
    $reply =~ s/^\n//;
    $reply =~ s/\n$//;
    $reply =~ s/Warning: .*\n//g;
    $reply =~ s/<\/?span.*?>//g;
    $reply = TSH::Utility::Wrap(2, $reply);
    print "The submission page replies:\n$reply\n";
    }
  elsif ($body =~ /: Payment Redirect/) {
    print "Your ratings submission has been accepted.\nYou should now go online to pay the ratings fee:\nhttps://secure.cosmoweb.net/scrabble-assoc/html/ratfee.html\n";
    }
  else {
    print "UNEXPECTED ERROR\n---\n$status\n$response->{'body'}\n";
    }
  $http->Close();
  }

=back

=cut

1;
