#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ShowManyPairings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);
use TSH::Command::ShowPairings;

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ShowManyPairings - implement the C<tsh> ShowManyPairings command

=head1 SYNOPSIS

  my $command = new TSH::Command::ShowManyPairings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ShowManyPairings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub MeasureLengths ($$);
sub Run ($$@);
sub ShowHeader ($$$$$);
sub ShowPlayer ($$$$$);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display the pairings for the specified division
and range of rounds.
EOF
  $this->{'names'} = [qw(smp showmanypairings)];
  $this->{'argtypes'} = [qw(RoundRange Division)];
  $this->{'board_format'} = undef;
  $this->{'board_sformat'} = undef;
  $this->{'p12_format'} = undef;
  $this->{'p12_sformat'} = undef;
  $this->{'pnumber_format'} = undef;
  $this->{'pnumber_sformat'} = undef;
  $this->{'round_format'} = undef;
  $this->{'table_format'} = undef;
  $this->{'table_sformat'} = undef;
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->MeasureLengths($division);

Measure lengths of text objects for use in the current run.

=cut

sub MeasureLengths ($$) {
  my $this = shift;
  my $dp = shift;

  my $np = $dp->CountPlayers();
  $this->{'board_sformat'}=$this->{'board_format'}='%0'.length($np/2).'d';
  $this->{'board_sformat'} =~ s/d$/s/;
  $this->{'pnumber_format'}='%0'.length($np).'d';
  $this->{'pnumber_sformat'} = '%-' . length($np) . 's';
  $this->{'p12_format'} = 
  $this->{'round_format'} = '%' . length($config::max_rounds || 10) . 'd';
  $this->{'p12_format'} = '%3d' if $this->{'p12_format'} =~ /^%[12]d$/;
  ($this->{'p12_sformat'} = $this->{'p12_format'}) =~ s/d$/s/;
  my $tables = $config::tables{$dp->Name()};
  if (defined $tables) {
    my $all_numeric = 1;
    my $max_length = 1;
    for my $table (@$tables) {
      my $length = length($table);
      $max_length = $length if $max_length < $length;
      $all_numeric = 0 if $table =~ /\D/;
      }
    $this->{'table_sformat'} = $this->{'table_format'} 
      = $all_numeric ? "%0${max_length}d" : "%-${max_length}s";
    $this->{'table_sformat'} =~ s/d$/s/;
    $this->{'entry_format'} = '%-' . length( sprintf("$this->{'pnumber_sformat'}".($config::assign_firsts ? '-1': '')."-$this->{'table_sformat'}", '', '')) . 's';
    }
  else {
    $this->{'entry_format'} = '%-' . length( sprintf("$this->{'pnumber_sformat'}".($config::assign_firsts ? '-1': '')."-$this->{'board_sformat'}", '', '')) . 's';
    }
  }

=item $command->Run($tournament, @parsed_args);

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($firstr1, $lastr1, $dp) = @_;
  return if $firstr1 > $lastr1;
  my $firstr0 = $firstr1-1;
  my $lastr0 = $lastr1-1;

  $this->MeasureLengths($dp);
  my $last_score_round0 = $dp->MostScores() - 1;
  for my $r1 ($firstr1..$lastr1) {
    $dp->CheckAutoPair($r1) || last;
    my $sr0 = $r1 - 2;
    $sr0 = $last_score_round0 if $sr0 > $last_score_round0;
    $dp->ComputeBoards($sr0, $r1-1);
    }
  if ($lastr0 > $dp->LastPairedRound0()) {
    $tournament->TellUser('enopryet', $dp->Name(), $lastr1);
    return;
    }
  my $logp = new TSH::Log($dp, 'multipair', "$firstr1-$lastr1",
    { 'title' => "Division ".$dp->Name()." Pairings: Round"
      . ($firstr1 == $lastr1 ? " $firstr1" : "s $firstr1-$lastr1")});
  # sample config line: perl $config'reserved{'P'}[13] = 4; # (to permanently station disabled player #13 in division P at board 4) 
  my $reserved = $config::reserved{$dp->Name()};
  $this->ShowHeader($dp, $logp, $firstr1, $lastr1);
  for my $p ($dp->Players()) {
    $this->ShowPlayer($logp, $p, $firstr0, $lastr0);
    }
  $logp->Close();
  $dp->Update(); # in case table numbers were changed
  }

=item $command->ShowHeader($division, $log, $first_round1, $last_round1);

Used internally to show the header of the pairings listings

=cut

sub ShowHeader ($$$$$) {
  my $this = shift;
  my $dp = shift;
  my $tournament = $dp->Tournament();
  my $logp = shift;
  my $first_round1 = shift;
  my $last_round1 = shift;
  my $tables = $config::tables{$dp->Name()};

  $logp->Write(
    sprintf("$this->{'pnumber_sformat'} $config::name_format", '#', 'Player'),
    <<'EOF');
<th class=id>#</th>
<th class=name>Player</th>
EOF
  if ($config::track_firsts) {
    $logp->Write(sprintf(" $this->{'p12_sformat'} $this->{'p12_sformat'}", 
      '1st', '2nd'), "<th class=p12>Firsts</th><th class=p12>Seconds</th>");
    }
  my $entry_columns = $config::assign_firsts ? 3 : 2;
  for my $r1 ($first_round1..$last_round1) {
    $logp->Write(sprintf(" $this->{'entry_format'}", "R$r1"), 
      "<th class=round colspan=$entry_columns>Rd. $r1</th>");
    }
  $logp->Write('', '</tr><tr class=subhead><th class=empty colspan='.($config::track_firsts ? 4 : 2)
    .'</th>'.(('<th class=opp>Opp</th>'.($config::assign_firsts?'<th class=p12>1/2</th>':'').'<th class=board>'.($tables?$config::table_title:'Board')) x ($last_round1-$first_round1+1)));
  $logp->Write("\n", "</tr>");
}

=item $command->ShowPlayer($logp, $player, $firstr0, $lastr0);

Used internally to show a row of pairings information for a player.

=cut

sub ShowPlayer ($$$$$) {
  my $this = shift;
  my $logp = shift;
  my $p = shift;
  my $firstr0 = shift;
  my $lastr0 = shift;

  my $dp = $p->Division();
  my $id = $p->ID();
  my $name = $p->Name();

  my $tables = $config::tables{$dp->Name()};
  $logp->Write(sprintf("$this->{'pnumber_format'} $config::name_format",
    $id, $name), "<tr><td class=id>$id</td><td class=name>$name</td>");
  if ($config::track_firsts) {
    my (@p12) = (0,0,0,0,0);
    for my $r0 (0..$firstr0 - 1) { $p12[$p->First($r0)]++; }
    $logp->Write(sprintf(" $this->{'p12_format'} $this->{'p12_format'}",
      $p12[1], $p12[2]),
      "<td class=p12>$p12[1]</td><td class=p12>$p12[2]</td>");
    }
  my $empty_p12_html=$config::assign_firsts ? '<td class=p12>&nbsp;</td>' : '';
  for my $r0 ($firstr0..$lastr0) {
    my $oid = $p->OpponentID($r0);
    my $text;
    my $html;
    my $p12 = $p->First($r0); $p12 =~ s/[^12]/ /;
    if (!defined $oid) { # unpaired
      $text = sprintf($this->{'pnumber_sformat'}, '') 
        . ($config::assign_firsts ? '- ' : '')
        . '-' .  sprintf($tables ? $this->{'table_sformat'} : $this->{'board_sformat'}, '');
      $html = "<td class=opp>&nbsp;</td>$empty_p12_html<td class=board>&nbsp;</td>";
      }
    elsif (!$oid) { # bye
      $text = sprintf($this->{'entry_format'}, 'Bye');
      $html = "<td class=opp>Bye</td>$empty_p12_html<td class=board>&nbsp;</td>";
      }
    elsif ($tables) {
      my $board = $p->Board($r0);
      my $table = $tables->[$board-1];
      $text = sprintf("$this->{'pnumber_format'}".($config::assign_firsts?"-$p12":"")."-$this->{'table_format'}", $oid, $table);
      $html = "<td class=opp>$oid</td>".($config::assign_firsts?"<td class=p12>$p12</td>":"")."<td class=board>$table</td>"; }
    else {
      my $board = $p->Board($r0);
      $text = sprintf("$this->{'pnumber_format'}".($config::assign_firsts?"-$p12":"")."-$this->{'board_format'}", $oid, $board);
      $html = "<td class=opp>$oid</td>".($config::assign_firsts?"<td class=p12>$p12</td>":"")."<td class=board>$board</td>";
      }
    $logp->Write(' ' . $text, $html);
    }
  $logp->Write(
    "\n",
    "</tr>");
  }

=back

=cut

=head1 BUGS

None known.

=cut

1;
