#!/usr/bin/perl

# Copyright (C) 2006 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Config;

use strict;
use warnings;

use FileHandle;
use File::Path;
use File::Spec;
use File::Temp qw(tempfile);
use TSH::Division;
use TSH::Utility;
use UNIVERSAL qw(isa);

=pod

=head1 NAME

TSH::Config - Manipulate tsh's configuration settings

=head1 SYNOPSIS

  my $config = new TSH::Config($config_filename) or die "no such file";
  my $config = new TSH::Config(); # select default configuration file
  $config->Load($tournament); # populates package 'config' with loaded values
  $config::key = 'new value';
  $config->Save(); # updates configuration file from package 'config'
  @option_names = TSH::Config::UserOptions();
  $help = TSH::Config::UserOptionHelp($option_name);
  $type = TSH::Config::UserOptionType($option_name);

=head1 ABSTRACT

This class manages tsh's configuration settings.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($;$);
sub new ($;$);
sub CheckDirectory ($$);
sub ChooseFile($);
sub Load ($$);
sub MakeBackupPath ($);
sub MakeHTMLPath ($);
sub MakeRootPath ($);
sub Save ($);
sub UserOptions ();
sub UserOptionHelp ($);
sub UserOptionType ($);
sub ValidateLocalDirectory ($$$);

# information about individual configuration options
my (%user_option_data) = (
  'assign_firsts' => { 'type' => 'boolean', 'help' => 'If checked, tsh decides who plays first (starts) or second (replies) in each game. Should be checked in the U.K. and unchecked in Canada and the United States.' },
  'backup_directory' => { 'type' => 'string', 'help' => 'Specifies where journalled ".t" and ".tsh" files are kept.', 'validate' => \&ValidateLocalDirectory, },
  'director_name' => { 'type' => 'string', 'help' => 'Gives the name of the director of this event, for use in ratings submission.' },
  'event_date' => { 'type' => 'string', 'help' => 'Gives the date(s) of this event, for use in report headers and ratings submission.' },
  'event_name' => { 'type' => 'string', 'help' => 'Gives the name of this event, for use in report headers and ratings submission.' },
  'html_directory' => { 'type' => 'string', 'help' => 'Specifies where generated web files are kept.', 'validate' => \&ValidateLocalDirectory, },
  'max_rounds' => { 'type' => 'integer', 'help' => 'Gives the number of rounds in this tournament. This parameter is mandatory when using tsh in server mode.' },
  'no_text_files' => { 'type' => 'boolean', 'help' => 'If checked, tsh creates only web HTML files. Leave this checked unless you prefer the retro look.' },
  'port' => { 'type' => 'integer', 'help' => 'If set to a nonzero value, enables the web interface that you are using and specifies its TCP/IP port number.  Do not change unless you know what this is.' },
  'track_firsts' => { 'type' => 'boolean', 'help' => 'If checked, keeps track of who played first (started) or second (replied) in each game. Should be checked in most parts of the world now.' },
  );

=item $log->initialise($argv)

Initialise the log object, create files.
$argv should contain whatever the user specified on the command line,
typically undef or the name of an event directory.

=cut

sub initialise ($;$) {
  my $this = shift;
  my $argv = shift;
  if (!defined($this->{'filename'} = ChooseFile($argv))) {
    return undef;
    }
  my $time = time; # can't use utime undef, undef without AIX warnings
  utime $time, $time, MakeRootPath($this->{'filename'});
  return $this;
  }

=item $d = new TSH::Config($filename);

Create a new TSH::Config object.  If the optional parameter 
$filename is omitted, a default value is chosen.

=cut

sub new ($;$) { return TSH::Utility::new(@_); }

=item CheckDirectory($path, $message);

Checks to see if $path exists, creates it if necessary and possible,
reports an error message on failure.

=cut

sub CheckDirectory ($$) {
  my ($dir, $what) = @_;
  return if -d $dir;
  mkpath $dir, 1, 0755;
  return if -d $dir;
  $_[0] = File::Spec->curdir();
  warn "Cannot create $dir, so $what will have to go in the main tsh directory.\n";
  }

=item $filename = ChooseFile($argv);

Used internally to choose a default configuration file.
$argv should contain whatever the user specified on the command line,
typically undef or the name of an event directory.

=cut

sub ChooseFile ($) {
  my $argv = shift;
  $config::root_directory = '.';
  # first choice: something specified on the command line
  if (defined $argv) {
    # if it's a directory
    if (-d $argv) {
      for my $try (qw(config.tsh tsh.config)) {
	if (-e "$argv/$try") {
	  $config::root_directory = $argv;
	  return $try;
	  }
        }
      die "$argv is a directory but has neither config.tsh nor tsh.config\n";
      }
    # else it's a configuration file
    elsif (-f $argv) {
      if ($argv =~ /\//) {
	($config::root_directory, $argv) = $argv =~ /^(.*)\/(.*)/;
	return $argv;
        }
      else { return $argv; }
      }
    else { die "No such file or directory: $argv\n"; }
    }
  # second choice: the directory containing the newest of */config.tsh
  elsif (my (@glob) = glob('*/config.tsh')) {
    # if more than one, choose newest
    if (@glob > 1) {
      my $newest_age = -M $glob[0];
      while (@glob > 1) {
	my $this_age = -M $glob[1];
	if ($this_age < $newest_age) {
	  $newest_age = $this_age;
	  shift @glob;
	  }
	else {
	  splice(@glob, 1, 1);
	  }
        }
      }
    ($config'root_directory, $glob[0]) = $glob[0] =~ /^(.*)\/(.*)/;
#   print "Directory: $config'root_directory\n";
    if ($config::root_directory =~ /^sample\d$/) {
      print <<"EOF";
You are about to run tsh using its sample event "$config::root_directory".  This is fine 
for practice, but might not be want you meant.  If you want to continue with
the sample event, please press the Return key now.  To choose a different
event, enter its name and then press the Return key.
EOF
      print "[$config::root_directory] ";
      my $choice = scalar(<STDIN>);
      $choice =~ s/^\s+//;
      $choice =~ s/\s+$//;
      $config::root_directory = $choice if $choice =~ /\S/;
      unless (-f "$config::root_directory/$glob[0]") {
	die "Cannot find config.tsh in $config::root_directory/.\n";
        }
      }
    return $glob[0];
    }
  # third choice: ./tsh.config
  if (-f 'tsh.config') {
    # don't use colour here, as colour configuration hasn't been found
    print "Warning: use of tsh.config is deprecated.\n";
    print "Please place your event files in a subdirectory, and save your\n";
    print "configuration in that subdirectory under the name config.tsh.\n";
    return 'tsh.config';
    }
  die "Cannot find a configuration file.\n";
  }

=item $config->Load($tournament);

Load a configuration and associated division data into the
TSH::Tournament object $tournament and the package 'config'.

=cut

sub Load ($$) {
  my $this = shift;
  my $tournament = shift;
  my $fn = $this->{'filename'};

  if ($^O eq 'MacOS') { 
    $config::backup_directory = ':old:'; 
    $config::html_directory = ':html'; 
    }
  else { 
    $config::backup_directory = './old/'; 
    $config::html_directory = './html/'; 
    }
  $config'max_name_length = 22;
  $config'name_format = '%-22s';
  $config'external_path = [qw(./bin)];
  $config::table_format = '%3s';
  my $fh = new FileHandle("$config::root_directory/$fn", "<")
    or die "Can't open $fn: $!\n";
  local($_);
  $tournament->TellUser('iloadcfg', "$config::root_directory/$fn");
  while (<$fh>) { s/(?:^|[^\\])#.*//; s/^\s*//; s/\s*$//; next unless /\S/;
    if (/^division?\s+(\S+)\s+(.*)/i) {
      my $dname = $1;
      my $dfile = $2;
      my $dp = new TSH::Division;
      $dp->Name($dname);
      $dp->File($dfile);
      $tournament->AddDivision($dp);
      }
    elsif (s/^perl\s+//i) { 
      eval $_;
      print "eval: $@\n" if length($@);
      }
    elsif (s/^config\s+//i) { 
      my $s = q($config::) . $_;
      eval $s;
      print "eval($s): $@\n" if length($@);
      }
    elsif (s/^autopair\s+//i) { 
      if (/^(\w+)\s+(\d+)\s+(\d+)\s+(\w+)\s+(.*)/) {
	my ($div, $sr, $round, $command, $args) = ($1, $2, $3, $4, $5);
	if ($sr >= $round) {
	  $tournament->TellUser('eapbr', $div, $sr, $round);
	  exit(1);
	  }
	my (@args) = split(/\s+/, $args);
	my $commandp = $tournament->GetCommandByName($command);
	$commandp = $commandp->Load() if isa $commandp, 'TSH::VCommand';
	if ($commandp && (isa $commandp, 'TSH::PairingCommand')) {
	  $config::autopair{uc $div}[$round] = [$sr, $command, @args];
	  }
	else {
	  print "'$commandp'\n";
	  $tournament->TellUser('ebadapc', $command);
	  exit(1);
	  }
        }
      else {
	chomp;
        $tournament->TellUser('ebadap', $_, $fn);
	exit(1);
        }
      }
    else {
      $tournament->TellUser('ebadcfg', $_);
      exit(1);
      }
    }
  close($fh);
# print "Configuration file loaded.\n";
  if ($config::external_path) {
    package main;
#   print "Loading external(s):";
    for $global::path (@$config::external_path) {
      my $config = "$global::path/tshxcfg.txt";
      if (-r "$config") {
	my $rv = do $config;
	if ($@) { print "Can't load externals [\@$@]"; }
	if ($!) { print "Can't load externals [!$!]"; }
	unless ($rv) { print "Can't load externals [X]"; }
        }
      }
#   print "\n";
    }
  CheckDirectory ((MakeRootPath $config::backup_directory), "backups");
  CheckDirectory ((MakeRootPath $config::html_directory), "web files");
  unless (-f MakeHTMLPath 'tsh.css') {
    my $fh;
    $fh = new FileHandle("lib/tsh.css", "<");
    if ($fh) {
      local($/) = undef;
      my $css = <$fh>;
      $fh->close();
      $fh = new FileHandle((MakeHTMLPath 'tsh.css'), ">");
      print $fh $css;
      $fh->close();
      }
    }
  unless ($config::player_number_format) {
    $config'player_number_format = 
      $tournament->CountDivisions() == 1 ? '#%s' : '%s';
    }
  $config::split1 = 1000000 unless $config::split1;
  # can't read tournament data until $config::split1 is set.
  for my $dp ($tournament->Divisions()) {
    $dp->Read();
    }
  for my $div (keys %config::gibson_groups) {
    my $divp = $config::gibson_groups{$div};
    for my $gibson_group (@$divp) {
      my $first = $gibson_group->[0];
      for my $i (1..$#$gibson_group) {
	$config::gibson_equivalent{$div}[$gibson_group->[$i]] = $first;
#	print "$gibson_group->[$i] equiv $first in $div.\n";
	}
      }
    }
  if (defined $config::entry) {
    if ($config::entry =~ /^absp$/i) { 
      $config::entry = 'absp';
      }
    elsif ($config::entry =~ /^nsa$/i) {
      $config::entry = 'nsa';
      }
    else {
      $tournament->TellUser('ebadconfigentry', $config::entry);
      $config::entry = 'nsa';
      }
    if (defined $config::assign_firsts) {
      if (!$config::assign_firsts) {
	$tournament->TellUser('wsetassignfirsts');
	}
      }
    $config::assign_firsts = 1;
    if (defined $config::track_firsts) {
      if (!$config::track_firsts) {
	$tournament->TellUser('wsettrackfirsts');
	}
      }
    $config::track_firsts = 1;
    }
  else {
    $config::entry = 'nsa';
    }
  $config::table_title = 'Table' unless defined $config::table_title;
  }

=item $path = MakeBackupPath($relpath);

Return a path to a file in the configured backup directory.

=cut

sub MakeBackupPath ($) {
  my $relpath = shift;
  return File::Spec->file_name_is_absolute($config::backup_directory)
    ? File::Spec->join($config::backup_directory, $relpath)
    : File::Spec->join($config::root_directory,
      $config::backup_directory, $relpath);
  }

=item $path = MakeHTMLPath($relpath);

Return a path to a file in the configured HTML directory.

=cut

sub MakeHTMLPath ($) {
  my $relpath = shift;
  return File::Spec->file_name_is_absolute($config::html_directory)
    ? File::Spec->join($config::html_directory, $relpath)
    : MakeRootPath(File::Spec->join($config::html_directory, $relpath));
  }

=item $path = MakeRootPath($relpath);

Return a path to a file in the root directory

=cut

sub MakeRootPath ($) {
  my $relpath = shift;
  return File::Spec->file_name_is_absolute($relpath)
    ? $relpath
    : File::Spec->join($config::root_directory, $relpath)
  }

=item $config->Save();

Save the current configuration in its configuration file.

=cut

sub Save ($) {
  my $this = shift;
  my $fn = "$config::root_directory/$this->{'filename'}";
  my $fh = new FileHandle($fn, "<");
  unless ($fh) {
    return "<div class=failure>Could not open '$fn': $!</div>";
    }
  local($/) = undef;
  my $s = <$fh>;
  close($fh);
  # update user options
  while (my ($key, $datap) = each %user_option_data) {
    my $value = ${$config::{$key}};
    my $type = $datap->{'type'};
    unless (defined $value) {
      if ($type eq 'boolean') { $value = 0; }
      elsif ($type eq 'integer') { $value = 0; }
      elsif ($type eq 'string') { $value = ''; }
      else { die "oops - unknown type $datap->{'type'} for $key\n"; }
      }
    if ($type eq 'string') { $value = '"' . quotemeta($value) . '"'; }
    $s =~ s/^\s*config\s+$key\s*=.*$/config $key = $value/m
      or $s .= "config $key = $value\n";
    }
  my ($newfh, $newfn) = tempfile(DIR => '.', UNLINK => 0);
  print $newfh $s;
  close($newfh);
  my $pwd = `pwd`;
  rename($fn, $config::backup_directory . time . '.tsh')
    && rename($newfn, $fn)
    && return "<div class=success>Your changes were saved.</div>";
  return "<div class=success>Your changes were not saved. ($config::backup_directory; $pwd) The operating system error was: $!</div>";
  }

=item @option_names = UserOptions();

Returns a list of user-configurable option names.

=cut

sub UserOptions () {
  return keys %user_option_data;
  }

=item $help = UserOptionHelp($option_name);

Returns the help text associated with a user-configurable option name.

=cut

sub UserOptionHelp ($) {
  my $key = shift;
  return $user_option_data{$key}{'help'};
  }

=item $type = UserOptionType($option_name);

Returns the type of a user-configurable option name:
boolean, integer or string.

=cut

sub UserOptionType ($) {
  my $key = shift;
  return $user_option_data{$key}{'type'};
  }

=item $error = UserOptionValidate($option_name, $option_value);

Checks to see if $option_value is a valid value for $option_name.
If it isn't, an error message is returned.  If it is, $option_value
may be cleaned slightly before being returned.

=cut

sub UserOptionValidate ($$) {
  my $key = shift;
  my $sub = $user_option_data{$key}{'validate'};
  return '' unless $sub;
  return &$sub($key, $_[0]);
  }

=item $error = $config->ValidateLocalDirectory($option_name, $option_value);

Returns an error message if $option_value is not a valid value for $option_name.
Returns the null string if it is valid.
Cleans up $option_value if necessary.
Checks to see if the value designates a local directory that exists
or can be created.
Paths may be relative to C<$config::root_directory>.

=cut

sub ValidateLocalDirectory ($$$) {
  my $this = shift;
  my $key = shift;
  my $path = $_[0];
  $path =~ s/^\s+//;
  $path =~ s/\s+$//;
  $path = '/' unless $_[0] =~ /\/$/;
  $path = File::Spec->canonpath($path);
  my $fqpath = $path;
  if (!File::Spec->file_name_is_absolute($path)) {
    $fqpath = File::Spec->join($config::root_directory, $path);
    }
  if (-e $fqpath) { 
    unless (-d $fqpath) {
      return "$path exists but is not a directory.";
      }
    }
  else {
    mkpath $fqpath, 0, 0755
      or return "$fqpath does not exist and cannot be created.";
    }
  $_[0] = $path;
  return 1;
  }

=head1 BUGS

Should create a configuration file if none is found.

Should offer to create a configuration file when the ChooseFile
returns a sample directory.

tshxcfg.txt should be read in and eval'ed, so that its code can
have access to a lexical copy of $tournament.

=cut

1;
