#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Tournament;

use strict;
use warnings;

use TSH::Utility;
use TSH::VCommand;
use UserMessage;

=pod

=head1 NAME

TSH::Tournament - abstraction of a Scrabble tournament within C<tsh>

=head1 SYNOPSIS

  $t = new Tournament;
  $t->AddCommand($command);
  $t->ReplaceCommand($command);
  $c = GetCommandByName($cn);
  $p = GetPlayerByName($pname);
  @cn = $t->CommandNames();
  $t->AddDivision($d);
  $n = $t->CountDivisions();
  @d = $t->Divisions();
  $d = $t->GetDivisionByName($dn);
  $t->Explain($code);
  $t->RegisterPlayer($p);
  $t->TellUser($code, @args);
  $success = $t->RunCommand(@argv);

=head1 ABSTRACT

This Perl module is used to manipulate tournaments within C<tsh>.

=head1 DESCRIPTION

=over 4

=cut

sub AddCommand ($$);
sub AddDivision ($$);
sub CommandNames ($);
sub CountDivisions ($);
sub Divisions ($);
sub ErrorFilter($$$);
sub Explain ($;$);
sub ExplainFilter ($$);
sub FindPlayer ($$$$);
sub GetCommandByName ($$);
sub GetPlayerByName ($$);
sub GetDivisionByName ($$);
sub initialise ($);
sub new ($);
sub NoteFilter($$$);
sub RegisterPlayer($$);
sub RunCommand ($@);
sub TellUser ($$@);
sub WarningFilter($$$);

=item $t->AddCommand($d)

Add a command to the tournament's user interface.

=cut

sub AddCommand ($$) { 
  my $this = shift;
  my $command = shift;
  for my $name ($command->Names()) {
    if (exists $this->{'cmdhash'}{$name}) {
      TSH::Utility::Error "Command name conflict for $name.  Some commands may not be available.\n";
      }
    $this->{'cmdhash'}{$name} = $command;
#   print "AddCommand: $name.\n";
    }
  $this->{'cmdlist'} = [];
  }

=item $t->AddDivision($d)

Add a division to the tournament.

=cut

sub AddDivision ($$) { 
  my $this = shift;
  my $dp = shift;
  my $dname = $dp->Name();
  if (exists $this->{'divhash'}{$dname}) {
    die "Duplicate division: $dname.\nAborting";
    }
  push(@{$this->{'divlist'}}, $dp);
  $this->{'divhash'}{$dname} = $dp;
  $dp->Tournament($this);
  }

=item @d = $t->CommandNames()

Return a list of a tournament's command names.

=cut

sub CommandNames ($) { 
  my $this = shift;
  my $csp = $this->{'cmdlist'};
  return @$csp if $csp && @$csp;
  $csp = $this->{'cmdlist'} = [sort keys %{$this->{'cmdhash'}}];
  return @$csp;
  }

=item $n = $t->CountDivisions()

Count how many divisions a tournament has.

=cut

sub CountDivisions ($) { 
  my $this = shift;
  return scalar(@{$this->{'divlist'}});
  }

=item @d = $t->Divisions()

Return a list of a tournament's divisions.

=cut

sub Divisions ($) { 
  my $this = shift;
  return @{$this->{'divlist'}};
  }

=item ErrorFilter($code, $type, $message);

Callback subroutine used by UserMessage.pm

=cut

sub ErrorFilter ($$$) {
  my $code = shift;
  my $type = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    print STDERR "Error: $message [$code]\n";
    return;
    }
  TSH::Utility::PrintColour 'red', "Error: $message";
  print " [$code]\n";
  }

=item $t->Explain();
=item $t->Explain($code);

Explain a message code (or the last one) to the user.

=cut

sub Explain ($;$) {
  my $this = shift;
  my $code = shift;

  $this->{'message'}->Explain($code) ||
  $this->TellUser('ebadhuh', $code);
  }

=item ExplainFilter($code, $message);

Callback subroutine used by UserMessage.pm

=cut

sub ExplainFilter ($$) {
  my $code = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    return;
    }
  print TSH::Utility::Wrap(0, "[$code] $message");
  }

=item $pp = $t->FindPlayer($name1, $name2, $dp);

Find a player whose name matches /$name1.*,$name2/ in division $dp.
If $dp is null, look in all divisions.

=cut

sub FindPlayer ($$$$) {
  my $this = shift;
  my $name1 = shift;
  my $name2 = shift;
  my $dp = shift;
  my $dname = $dp && $dp->Name();
  my $pattern = qr/$name1.*$name2/i;
  my @matched;
  while (my ($name, $pp) = each %{$this->{'pbyname'}}) {
    next unless $name =~ /$pattern/;
    next if $dname && $pp->Division->Name() ne $dname;
    push(@matched, $pp);
    }
  if (@matched == 0) { $this->TellUser('enomatch', "$name1,$name2"); }
  elsif (@matched == 1) { return $matched[0]; }
  elsif (@matched <= 10) {
    $this->TellUser('emultmatch', "$name1,$name2", 
      join('; ', map { $_->TaggedName() } @matched));
    }
  else { $this->TellUser('emanymatch', "$name1,$name2"); }
  return undef;
  }

=item $c = $t->GetCommandByName($cn);

Obtain a Command pointer given a command name.

=cut

sub GetCommandByName ($$) {
  my $this = shift;
  my $cname = shift;
  my $c = $this->{'cmdhash'}{lc $cname};
  return $c;
  }

=item $pp = $t->GetPlayerByName($name);

Obtain a Player pointer given a player name.

=cut

sub GetPlayerByName ($$) {
  my $this = shift;
  my $pname = shift;
  return $this->{'pbyname'}{$pname};
  }

=item $d = $t->GetDivisionByName($dn);

Obtain a Division pointer given the division's name in not 
necessarily canonical style.

=cut

sub GetDivisionByName ($$) {
  my $this = shift;
  my $dname = shift;
  $dname = TSH::Division::CanonicaliseName($dname);
  return $this->{'divhash'}{$dname};
  }

=item $t->initialise();

(Re)initialise a Tournament object, for internal use.

=cut

sub initialise ($) {
  my $this = shift;
  # all fields should be listed here, regardless of whether they need init
  $this->{'cmdhash'} = {};
  $this->{'cmdlist'} = [];
  $this->{'divhash'} = {};
  $this->{'divlist'} = [];
  $this->{'message'} = undef;
  $this->{'pbyname'} = {};

  my $um = $this->{'message'} 
    = new UserMessage($config::message_file || 'lib/messages.txt');
  $um->SetErrorFilter(\&ErrorFilter);
  $um->SetExplainFilter(\&ExplainFilter);
  $um->SetNoteFilter(\&NoteFilter);
  $um->SetWarningFilter(\&WarningFilter);

  TSH::VCommand::LoadAll($this);
  }

=item $t = new Tournament;

Create a new Tournament object.  

=cut

sub new ($) { return TSH::Utility::new(@_); }

=item $t->ReplaceCommand($d)

Replace a command in the tournament's user interface,
either because a new version has been downloaded or
because the command has been invoked for the first
time and a stub command object is being replaced.

=cut

sub ReplaceCommand ($$) { 
  my $this = shift;
  my $command = shift;
  for my $name ($command->Names()) {
    unless (exists $this->{'cmdhash'}{$name}) {
      TSH::Utility::Error "Adding '$name' to the command list, replacing nothing?!\n";
      }
    $this->{'cmdhash'}{$name} = $command;
#   print "ReplaceCommand: $name.\n";
    }
  # 'cmdlist' will be rebuilt by the 'help' command.
  # Could insert here, but probably faster to rebuild once when help is
  # asked for rather than insert each time a command is added
  $this->{'cmdlist'} = []; 
  }

=item NoteFilter($code, $type, $message);

Callback subroutine used by UserMessage.pm

=cut

sub NoteFilter ($$$) {
  my $code = shift;
  my $type = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    return;
    }
  print "$message [$code]\n";
  }

=item $boolean = $t->QuittingTime();
=item $t->QuittingTime($boolean);

Get/set whether or not it's time to quit.

=cut

sub QuittingTime ($;$) { TSH::Utility::GetOrSet('quit', @_); }

=item $t->RegisterPlayer($p);

Register a player so that they can subsequently be looked up by name.

=cut

sub RegisterPlayer ($$) {
  my $this = shift;
  my $p = shift;
  $this->{'pbyname'}{$p->Name()} = $p;
  }

=item $success = $t->RunCommand(@argv);

Dispatch a user command to the appropriate TSH::Command object.

=cut

sub RunCommand($@) {
  my $this = shift;
  my (@argv) = @_;
  my $arg0 = lc $argv[0];
  my $commandsp = $this->{'cmdhash'};
  my $command = $commandsp->{$arg0};
  unless ($command) {
    TSH::Utility::Error "No such command: $argv[0]";
    return 0;
    }
  my (@parsed_argv)= $global'parser->Parse(\@argv, $command->ArgumentTypes());
  if ($global'parser->Error()) {
    print $command->Usage();
    return 0;
    }
  $command = $commandsp->{$arg0}; # ArgumentTypes() might have called VCommand::Load()
  $command->Run($this, @parsed_argv);
  return 1;
  }

=item $t->TellUser($code, @args);

Tell the user something important.

=cut

sub TellUser ($$@) {
  my $this = shift;
  my $code = shift;
  my @args = @_;

  $this->{'message'}->Show($code, @args);
  }

=item $t->UpdateDivisions(\%dirty_divisions);

Update divisions that have been marked as dirty.

=cut

sub UpdateDivisions ($) {
  my $this = shift;
  die if defined shift;
  for my $dp ($this->Divisions()) {
    next unless $dp->Dirty();
    my $dname = $dp->Name();
    print "Updating Division $dname.\n";
    $dp->Update();
    }
  }

=item WarningFilter($code, $type, $message);

Callback subroutine used by UserMessage.pm

=cut

sub WarningFilter ($$$) {
  my $code = shift;
  my $type = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    print STDERR "Warning: $message [$code]\n";
    return;
    }
  TSH::Utility::PrintColour 'red', 'Warning: ';
  print "$message [$code]\n";
  }

=back

=cut

=head1 BUGS

None reported yet.

=cut

1;

