#!/usr/bin/perl -w

## tsh - tournament shell
## Copyright (C) 1998-2003 by John J. Chew, III.

=head1 NAME

B<tsh> - Scrabble tournament management shell

=head1 SYNOPSIS

B<tsh> [directory|configuration-file]

=head1 DESCRIPTION

For user information, see the accompanying HTML documentation.
This builtin (pod) documentation is intended for program maintenance 
and development.

=cut

## Version history: moved to doc/news.html

## public libraries

BEGIN { unshift(@::INC, "$::ENV{'HOME'}/lib/perl") if defined $::ENV{'HOME'}; }
use strict;
use lib './lib/perl';
use Fcntl ':flock';
use FileHandle;

# use warnings FATAL => 'uninitialized';

## private libraries

use TSH::Command;
use TSH::Config;
use TSH::PairingCommand;
use TSH::ParseArgs;
use TSH::Log;
use TSH::Player;
use TSH::Tournament;
use TSH::XCommand;
use UserMessage;
use TSH::Utility qw(Debug);

## global constants
our $gkVersion = '3.150';

## prototypes

# sub CheckGroupRepeats ($$);
# sub CmdFactorPairQuads ($$);
sub DefineExternal ($$$;$);
# sub DoFactor ($$$$);
# sub DoFactorGroup ($$$);
sub lint ();
sub LockFailed ($);
sub LockOff ();
sub LockOn ();
sub Main ();
sub Prompt ();
sub ReopenConsole ();
sub RunInteractive ($);
sub RunServer ($);
sub Use ($);

## global variables

our $gTournament; 

=head1 SUBROUTINES

=over 4

=cut

# Part of the not yet ready CmdFactorPairQuads
# sub CheckGroupRepeats ($$) {
#   my $psp = shift;
#   my $repeats = shift;
#   for my $i (0..$#$psp) {
#     my $repeatsp = $psp->[$i]{'repeats'};
#     die "Player $psp->[$i]{'name'} has no repeats information.\n"
#       unless ref($repeatsp) eq 'ARRAY';
#     for my $j ($i+1..$#$psp) {
#       if ((my $this_repeats = $repeatsp->[$psp->[$j]{'id'}]) > $repeats) {
# #	TSH::Utility::Error "Warning: $psp->[$i]{'name'} and $psp->[$j]{'name'} have played each other $this_repeats time(s).\n";
# 	return 0;
#         }
#       }
#     }
#   return 1;
#   }

# FactorPairQuads is not yet ready for public use
# sub CmdFactorPairQuads ($$) { my($argvp, $args) = @_;
#   my ($factor, $repeats, $sr, $dp) 
#     = ParseArgs $argvp, [qw(factor repeats based-on-round division)];
#   return 0 unless defined $dp;
#   my $sr0 = $sr-1;
#   $dp->CheckRoundHasResults($sr0) or return 0;
#   print "Calculating Factored Pairings for Division $dp->{'name'} based on round $sr, $repeats repeats allowed, factoring by $factor.\n";
#   DoFactor $dp, $repeats, $sr0, $factor;
#   return 0;
#   }
# 
  
sub DefineExternal ($$$;$) {
  my $name = lc shift;
  my $script = shift;
  my $template = shift;
  my $default_args = shift;

  my $command = new TSH::XCommand("$global::path/$script", [$name], $template, 
    $default_args);
  $gTournament->AddCommand($command);
# print " $name";
  return 1;
  }

# DoFactor is not yet ready for public use
# sub DoFactor ($$$$) { my ($dp, $repeats, $sr0, $factor) = @_;
#   my $datap = $dp->{'data'};
#   my $theKeyRound = $sr0;
# 
#   my $tobepaired = $dp->GetRegularUnpaired($sr0, 'nobyes');
#   unless (@$tobepaired) {
#     TSH::Utility::Error "No players can be paired.\n";
#     return 0;
#     }
# # die "Assertion failed" unless @$tobepaired % 2 == 0;
#   my $minbyes = 0;
#   if (@$tobepaired % 2) {
#     $minbyes = CountByes $dp;
#     }
#   my (@ranked) = TSH::Player::SortByStanding $theKeyRound, @$tobepaired;
# 
#   my @pair_list;
#   my $group_number = 0;
#   while (@ranked) {
#     $group_number++;
#     my (@group) = @ranked > $factor + $factor
#       ? splice(@ranked, 0, $factor)
#       : splice(@ranked, 0);
#     my (@group_list) = DoFactorGroup \@group, $repeats, $minbyes;
#     unless (@group_list) {
#       TSH::Utility::Error "Can't factor group #$group_number. Division is partially paired.\n";
#       last;
#       }
#     push(@pair_list, @group_list);
#     }
# 
#   # store pairings
#   {
#     my $board = 1;
#     while (@pair_list) {
#       my $gp = shift @pair_list;
#       # make sure previous board numbers are set
#       for my $pp (@$gp) {
# 	my $tp = $pp->{'etc'}{'board'};
# 	my $pairingsp = $pp->{'pairings'};
# 	if (!defined $tp) {
# 	  $pp->{'etc'}{'board'} = [ (0) x @$pairingsp ];
# 	  }
# 	elsif ($#$tp < $#$pairingsp) {
# 	  push(@{$pp->{'etc'}{'board'}}, (0) x $#$pairingsp - $#$tp);
# 	  }
#         }
#       if (@$gp == 3) {
# 	# TODO: this somewhat duplicates InitFontes and should use a table
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[1]{'id'}, 0);
# 	push(@{$gp->[1]{'pairings'}},
# 	  0,              $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[1]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, 0);
# 	push(@{$gp->[1]{'etc'}{'board'}}, 0, $board, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board, 0, $board);
# 	$board += 1;
# 	}
#       elsif (@$gp == 4) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[3]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[1]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board, $board+1, $board+1);
# 	$board += 2;
# 	}
#       elsif (@$gp == 5) {
# 	# This table is not the one used in InitFontes
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[2]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[0]{'id'}, 0);
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[0]{'id'}, 0,              $gp->[4]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  0,              $gp->[1]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board,   $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board,  0);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board,   0,      $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, 0,       $board+1, $board+1);
# 	$board += 2;
#         }
#       elsif (@$gp == 6) {
# 	push(@{$gp->[0]{'pairings'}},
# 	  $gp->[5]{'id'}, $gp->[3]{'id'}, $gp->[1]{'id'});
# 	push(@{$gp->[1]{'pairings'}},
# 	  $gp->[2]{'id'}, $gp->[4]{'id'}, $gp->[0]{'id'});
# 	push(@{$gp->[2]{'pairings'}},
# 	  $gp->[1]{'id'}, $gp->[5]{'id'}, $gp->[3]{'id'});
# 	push(@{$gp->[3]{'pairings'}},
# 	  $gp->[4]{'id'}, $gp->[0]{'id'}, $gp->[2]{'id'});
# 	push(@{$gp->[4]{'pairings'}},
# 	  $gp->[3]{'id'}, $gp->[1]{'id'}, $gp->[5]{'id'});
# 	push(@{$gp->[5]{'pairings'}},
# 	  $gp->[0]{'id'}, $gp->[2]{'id'}, $gp->[4]{'id'});
# 	push(@{$gp->[0]{'etc'}{'board'}}, $board, $board, $board);
# 	push(@{$gp->[1]{'etc'}{'board'}}, $board+1, $board+1, $board);
# 	push(@{$gp->[2]{'etc'}{'board'}}, $board+1, $board+2, $board+1);
# 	push(@{$gp->[3]{'etc'}{'board'}}, $board+2, $board, $board+1);
# 	push(@{$gp->[4]{'etc'}{'board'}}, $board+2, $board+1, $board+2);
# 	push(@{$gp->[5]{'etc'}{'board'}}, $board, $board+2, $board+2);
# 	$board += 3;
# 	}
#       else { die "Assertion failed"; }
#       }
#       my $p1 = shift @pair_list;
#       my $p2 = shift @pair_list;
#       push(@{$p1->{'pairings'}}, $p2->{'id'});
#       push(@{$p2->{'pairings'}}, $p1->{'id'});
#   } # store pairings
# 
#   print "Done.\n";
#   $dp->Dirty(1);
#   $gTournament->UpdateDivisions();
#   }

# # TODO: this could be more efficient, but was written live at NSC 2005
# sub DoFactorGroup ($$$) {
#   my $psp = shift; # must not modify contents
#   # 0 indicates no repeats allowed, ..., 3 means up to 3 repeats = 4 pairings
#   my $repeats = shift;
#   # TODO: allow for possibility that we have to increase minbytes after 1st plr
#   my $minbyes = shift;
# 
#   print "DFG: " . (1+$#$psp) . ' ' . join(',', map { $_->{'id'} } @$psp) . "\n";
#   if (@$psp == 4 || @$psp == 6) {
#     if (CheckGroupRepeats $psp, $repeats) {
# #     print "DFG: returning $#$psp+1\n";
#       return ([@$psp]);
#       }
#     else {
# #     print "DFG: returning failure\n";
#       return ();
#       }
#     }
#   elsif (@$psp == 3) {
#     for my $p (@$psp) {
#       if ($p->{'byes'} != $minbyes) {
# #	print "DFG: $p->{'name'} already has $p->{'byes'} bye(s).\n";
# 	return ();
#         }
#       }
#     return ([@$psp]);
#     }
#   elsif (@$psp == 5) {
#     my $possible_bye_players = 0;
#     for my $p (@$psp) {
#       if ($p->{'byes'} == $minbyes) {
# 	$possible_bye_players++;
#         }
#       }
#     if ($possible_bye_players < 3) {
#       for my $p (@$psp) {
#         print "DFG5: $p->{'name'} already has $p->{'byes'} bye(s).\n";
#         }
#       return ([
# 	sort { $b->{'byes'} <=> $a->{'byes'} } @$psp
#         ]);
#       }
#     }
#   elsif (@$psp < 7) {
#     die "DoFactorGroup: bad group size: " . scalar(@$psp) . "\n";
#     }
#   my $s = int(@$psp/4);
#   my $p1 = $psp->[0];
#   my $j1 = 0;
#   # first try to pair within quartiles
#   for my $j2 ($s..$s+$s-1) {
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $j3 ($s+$s..$s+$s+$s-1) {
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $j4 ($s+$s+$s..$s+$s+$s+$s-1) {
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
#   # then try to pair anywhere within the group
#   for my $i2 (0..$#$psp) {
#     my $j2 = ($i2 + $s) % @$psp;
#     next if $j2 == $j1;
#     my $p2 = $psp->[$j2];
#     my $rep2 = $p2->{'repeats'};
#     next if $rep2->[$p1->{'id'}] > $repeats;
#     for my $i3 (0..$#$psp) {
#       my $j3 = ($i3 + $s + $s) % @$psp;
#       next if $j3 == $j1 || $j3 == $j2;
#       my $p3 = $psp->[$j3];
#       my $rep3 = $p3->{'repeats'};
#       next if $rep3->[$p2->{'id'}] > $repeats;
#       next if $rep3->[$p1->{'id'}] > $repeats;
#       for my $i4 (0..$#$psp) {
# 	my $j4 = ($i4 + $s + $s + $s) % @$psp;
# 	next if $j4 == $j1 || $j4 == $j2 || $j4 == $j3;
# 	my $p4 = $psp->[$j4];
# 	my $rep4 = $p4->{'repeats'};
# 	next if $rep4->[$p3->{'id'}] > $repeats;
# 	next if $rep4->[$p2->{'id'}] > $repeats;
# 	next if $rep4->[$p1->{'id'}] > $repeats;
# 	my (@unpaired) = @$psp[grep 
# 	  { $_ != $j1 && $_ != $j2 && $_ != $j3 && $_ != $j4 }
# 	  0..$#$psp];
# 	my (@quads) = DoFactorGroup \@unpaired, $repeats, $minbyes;
# 	if (@quads) {
# 	  unshift(@quads, [$p1,$p2,$p3,$p4]);
# #	  print "DFG: returning 4*($#quads+1)\n";
# 	  return @quads;
# 	  }
# 	}
#       }
#     }
# # print "DFG: returning failure.\n";
#   return ();
#   }

sub lint () {
  $config'table_format = '';
  $config'gibson = undef;
  %config::gibson_equivalent = ();
  %config::autopair = ();
# $config'reserved{''} = '';
# $config'tables{''} = '';
  lint;
  }

sub LockFailed ($) {
  my $reason = shift;
  print <<"EOF";
System call failed: $reason

You should not run more than one copy of tsh using the same 
configuration file at the same time.  tsh uses a "lock file" called
tsh.lock to keep track of when it is running.  This copy of tsh
was unable to get access to the lock file.  The most likely reason
for this is that tsh is already in use.
EOF
  exit 1;
  }

sub LockOff () {
  flock($global'lockfh, LOCK_UN)
    or die "Can't unlock tsh.lock - something is seriously wrong.\n";
  close($global'lockfh)
    or die "Can't close tsh.lock - something is seriously wrong.\n";
  }

sub LockOn () {
  my $error;

  my $fn = TSH::Config::MakeRootPath('tsh.lock');
  $global'lockfh = new FileHandle $fn, O_CREAT | O_RDWR
    or die "Can't open tsh.lock - check to make sure tsh isn't already running.\n";
  flock($global'lockfh, LOCK_EX | LOCK_NB) 
    or LockFailed "flock: $!";
  seek($global'lockfh, 0, 0) 
    or die "Can't rewind tsh.lock - something is seriously wrong.\n";
  truncate($global'lockfh, 0) 
    or die "Can't truncate tsh.lock - something is seriously wrong.\n";
  print $global'lockfh "$$\n"
    or die "Can't update tsh.lock - something is seriously wrong.\n";
  } 

sub Main () {
  srand;

  ReopenConsole if $^O eq 'MacOS';
  my $dir = @::ARGV ? shift @::ARGV : undef;
  $config::config = new TSH::Config($dir);
  LockOn;
  $gTournament = new TSH::Tournament;
  $gTournament->TellUser('iwelcome', $gkVersion);
  $config::config->Load($gTournament);
  $global'parser = new TSH::ParseArgs;
  if (defined $config::event_name) {
    my $name = $config::event_name;
    $name .= ", " . $config::event_date if defined $config::event_date;
    $gTournament->TellUser('ievtname', $name);
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/) {
      print "\e]0;tsh $gkVersion - $config::event_name\a";
      }
    }
  if ($config::port) { RunServer $gTournament; }
  else { RunInteractive $gTournament; }
  LockOff;
  if (defined $config::event_name) {
    if ($::ENV{'TERM'} && $::ENV{'TERM'} =~ /^xterm/) {
      print "\e]0;\a";
      }
    }
  }

sub Prompt () { 
  TSH::Utility::PrintColour 'yellow on blue', 'tsh>';
  print ' ';
  }

sub ReopenConsole () {
  close(STDOUT);
  close(STDERR);
  close(STDIN);
  open(STDOUT, "+>dev:console:tsh console") || die;
  open(STDERR, "+>dev:console:tsh console") || die;
  open(STDIN, "<dev:console:tsh console") || die;
  $| = 1;
  }

sub RunInteractive ($) {
  my $tournament = shift;
  eval "use Term::ReadLine";
  my $term = new Term::ReadLine "tsh $gkVersion";
# my $ofh = $term->OUT || \*STDOUT;
  while (1) {
    $_ = $term->readline('tsh> ');
    last unless defined $_;
    s/^\s+//; 
    my(@argv) = split;
    next unless /\S/;
    if (!$tournament->RunCommand(@argv)) 
      { print "Enter 'help' for help.\n"; }
    last if $tournament->QuittingTime();
    }
  }

sub RunServer ($) {
  my $tournament = shift;
  Use "TSH::Server";
  my $server = new TSH::Server($config::port, $tournament);
  unless ($server->Start()) {
    $tournament->TellUser('etshtpon', $server->Error());
    RunInteractive $tournament;
    return;
    }
  while ($server->Run()) { }
  $server->Stop();
  }

# sub RunVanilla ($) {
#   my $tournament = shift;
#   Prompt;
#   while (<>) {
#     next unless /\S/;
#     s/^\s+//;
#     my(@argv) = split;
#     if (!$tournament->RunCommand(@argv)) 
#       { print "Enter 'help' for help.\n"; }
#     last if $tournament->QuittingTime();
#     }
#   continue {
#     Prompt;
#     }
#   }

=item Use $module or die

Carefully evaluate "use $module" and try to explain to a naive user
the meaning of failure.

=cut

sub Use ($) {
  my $module = shift;
  eval "use $module";
  die "Couldn't load $module, most likely because your tsh software distribution\nis incomplete.  Please download it again.  Here is what Perl says caused\nthe problem:\n$@" if $@;
  return 1;
  }

=back

=cut

=head1 BUGS

Here is the list of planned improvements to C<tsh>, in roughly
descending order of priority.

=over 4

=item *
  Should not have a global variable called $gTournament.
  Global variables are bad.

=item *
 default commands should start with RR if number of players is <=
 number of remaining rounds - 1 (and last round is not partially paired)

=item *
 team match pairing commands

=item *
 Fontes Gibsonization possibility warnings 

=item *
 do not allow pairing commands to exceed max_rounds

=item *
 webupdater should be configurable and documented

=item *
 default values for command parameters?

=item *
 prompt for and load or create a config.tsh file if none specified
 and default would be a sample directory

=item *
 ratings submission command (external)

=item *
 return values from externals

=item *
 some configuration variables to externals

=item *
 a command to unpair selective pairings (maybe a 1-arg form of pair)

=item *
 division data complete message and trigger

=item *
 photos on scorecards 

=item *
 virtual scorecards on web

=item *
 printing support

=item *
 add more internal cross-references in the documentation

=item *
 proofread documentation for typographic style

=item *
 load large divisions in separate threads

=item *
 a report listing last lines of scorecards for all players, 
 so that players can check their results

=item *
 choose random seed for firsts/seconds in a way that can't be jiggered
 by a director

=item *
 should not create a ratings-0.html file

=item *
 correctly rate NSA players who are on high multipliers

=item *
 hovering on photos should enlarge them

=item *
 Using the manual PAIR command sometimes leads to more than one game
 at one board.  It should swap boards as necessary.

=item *
 It should at least be configurable to require tsh to keep winners
 of games stationary.

=item * 
 Swiss pairings should try to minimize the number of players promoted
 between groups each round.

=back

=cut

END { 
  sleep 10 if $^O eq 'MSWin32'; # to prevent error messages from disappearing
  }

## main code
Main;

