#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::AUPAIR;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::AUPAIR - implement the C<tsh> AUPAIR command

=head1 SYNOPSIS

  my $command = new TSH::Command::AUPAIR;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::AUPAIR is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to create a AUPAIR .TOU file,
to submit to a rating system requiring that input format.
EOF
  $this->{'names'} = [qw(aupair)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  unless ($config::event_name) { $tournament->TellUser('eneedename'); return; }
  for my $dp (@_) {
    my $dname = $dp->Name();
    my $fn = TSH::Config::MakeRootPath("$dname.TOU");
    my $fh = TSH::Utility::OpenFile(">", $fn);
    unless ($fh) {
      TSH::Utility::Error("Can't create $fn: $!");
      return;
      }
    my $nrounds = $dp->MostScores();
    if ($nrounds != $dp->LeastScores()) {
      TSH::Utility::Error "Not all results are in yet in round $nrounds.";
      }
    print $fh "*M$config::event_name\015\012";
    print $fh "*$dname\015\012";
    print $fh "                                      0\015\012"; # high word

    for my $p ($dp->Players()) {
      my ($surname, $given) = split(/, */, $p->Name(), 2);
      if (!defined $given) {
	TSH::Utility::Error "Can't find a comma separating surname from given name: $p->{'name'}\n";
	return 0;
        }
      printf $fh "%-20.20s", "$given $surname";
      for my $r (1..$nrounds) {
	my $r0 = $r - 1;
	printf $fh " %1s%3d%4d", 
#	  ($p->First($r0)==2 ? '2' : ' '),
          $p->Score($r0) > ($p->OpponentScore($r0)||0) ? '2' : ' ',
	  $p->Score($r0),
	  $p->OpponentID($r0);
        } # for $r0
      print $fh "\015\012";
      } # for my $p
    print $fh "*** END OF FILE ***\015\012";
    close $fh;
    }
  $tournament->TellUser('idone');
  }

=back

=cut

=head1 BUGS

Makes extensive inappropriate use of Player.pm internals.

Currently only exports one division at a time.

=cut

1;
