#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::EditScore;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::EditScore - implement the C<tsh> EditScore command

=head1 SYNOPSIS

  my $command = new TSH::Command::EditScore;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::EditScore is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to edit player scores that have already been
entered.  Specify the division, player number and round number whose
scores you want to change.  The command displays the player's
scorecard and prompts you for a change.  You may enter '?' at the
prompt for a review of your choices.  
EOF
  $this->{'names'} = [qw(es editscore)];
  $this->{'argtypes'} = [qw(Division Player Round0)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($dp, $id, $round) = @_;
  my $datap = $dp->{'data'};
  my $p = $datap->[$id];
  unless (defined $p) { TSH::Utility::Error "No such player: $id.\n"; return 0; }
  my $round0 = $round-1;
  if ($round) {
    unless (exists $p->{'scores'}[$round0]) 
      { TSH::Utility::Error "Player doesn't yet have scores in round $round.\n"; return 0; }
    unless (exists $p->{'pairings'}[$round0]) 
      { TSH::Utility::Error "Player isn't yet paired in round $round.\n"; return 0; }
    }
      
  while (1) {
    $this->Processor()->Process("sc ".$dp->Name()." $id");
    my ($oppid, $opp, $ms, $os);
    if ($round) {
      $oppid = $p->{'pairings'}[$round0];
      if (!defined $oppid) {
	# should never happen
	TSH::Utility::Error "$p->{'name'} has no opponent in round $round.\n";
	last;
	}
      $opp = $datap->[$oppid];
      $ms = $p->{'scores'}[$round0] || 0;
      $os = $oppid && $opp->{'scores'}[$round0] || 0;
      }

    # prompt for input
    {
      my $prompt = '';
      $prompt .= "$p->{'name'} ($dp->{'name'}$id) R$round ";
      if ($round) {
	$prompt .= '[';
	if ($oppid) {
	  my $vs = $dp->FormatPairing($round0, $id, 'half');
	  if ($config::entry eq 'absp') {
	    $prompt .= sprintf("%+d %s", $ms-$os, $vs);
	    }
	  else {
	    $prompt .= "$ms $os $vs";
	    }
	  }
	else { $prompt .= "bye scoring $ms"; }
	$prompt .= "] ";
	}
      $prompt .= "(? for help)";
      TSH::Utility::Prompt $prompt;
    }
    # parse input
    local($_) = lc scalar(<STDIN>);
    s/^\s+//; s/\s+$//;
    if (/^\?$/) {
      print "? To change focus: D div, R round, P player\n";
      print "? To change data:";
      print " FIRST, SECOND," if $round && $config::track_firsts;
      if (exists $p->{'scores'}[$round0]) {
	if ($config::entry eq 'absp') {
	  print " [+-]score,";
	  }
	else {
	  print " my-score opp-score,";
	  }
        }
      if (exists $p->{'etc'}{'off'}) {
	print " ON,";
        }
      else {
	print " OFF spread,";
        }
      print " RA(TING) n\n";
      }
    elsif (/^d\s+(\S+)$/) {
      my $newdp = $tournament->GetDivisionByName($1);
      if (defined $newdp) { 
        $dp = $newdp;
        $datap = $dp->{'data'};
        $p = $datap->[$id];
	$p = $datap->[$id = 1] unless defined $p;
	if ($round && !exists $p->{'scores'}[$round0]) {
	  TSH::Utility::Error "No corresponding scores in division $dp->{'name'}\n";
	  return 0;
	  }
        }
      else {
	TSH::Utility::Error "No such division.\n";
        }
      }
    elsif (/^r\s+([1-9]\d*)$/) {
      my $newround = $1;
      my $newround0 = $newround - 1;
      unless (exists $p->{'pairings'}[$newround0]) {
	TSH::Utility::Error "Player does not yet have an opponent in round $newround.\n";
        }
      else {
	$round = $newround;
	$round0 = $newround0;
        }
      }
    elsif (/^p\s+([1-9]\d*)$/) {
      my $newid = $1;
      my $newp = $datap->[$newid];
      if (!defined $newp) {
	TSH::Utility::Error "No such player.\n";
        }
      elsif ($round && !exists $newp->{'scores'}[$round0]) {
	TSH::Utility::Error "Player has no scores in round $round.\n";
        }
      else {
        $p = $newp;
        $id = $newid;
        }
      }
    elsif ($round && /^(first|second)$/i) {
      unless ($oppid) {
	TSH::Utility::Error "Player had a bye, went neither first nor second.\n";
	next;
        }
      for my $etcp ($p->{'etc'}, $opp->{'etc'}) {
	if ($#{$etcp->{'p12'}} < $round0) {
	  TSH::Utility::Error "Please record earlier firsts and seconds first.\n";
	  next;
	  }
        }
      if (/^first$/i) {
	$p->{'etc'}{'p12'}[$round0] = 1;
	$opp->{'etc'}{'p12'}[$round0] = 2;
        }
      else {
	$p->{'etc'}{'p12'}[$round0] = 2;
	$opp->{'etc'}{'p12'}[$round0] = 1;
        }
      $dp->Dirty(1);
      }
    elsif ((exists $p->{'etc'}{'off'}) && /^on$/i) {
      # have to Synch() here just in case the player was just turned off
      $dp->Synch();
      delete $p->{'etc'}{'off'};
      # remove any trailing byes back to where last regular pairing is
#     printf "removing (%d,%d,%d).\n", $#{$p->{'pairings'}}, $dp->LastPairedRound0(), $p->{'pairings'}[-1];
      while ($#{$p->{'pairings'}} > $dp->LastPairedRound0()
	&& !$p->{'pairings'}[-1]) {
	pop(@{$p->{'pairings'}});
	$#{$p->{'scores'}} = $#{$p->{'pairings'}}
	  if $#{$p->{'scores'}} > $#{$p->{'pairings'}};
#	printf "removed, now (%d,%d,%d).\n", $#{$p->{'pairings'}}, $dp->LastPairedRound0(), $p->{'pairings'}[-1];
        }
      $dp->Dirty(1);
      }
    elsif ((!exists $p->{'etc'}{'off'}) && /^off\s+(-?\d+)$/i) {
      $p->{'etc'}{'off'} = [$1];
      $dp->Dirty(1);
      }
    elsif ($oppid && exists $p->{'scores'}[$round0] && 
      ($config::entry eq 'absp' ? /^([-+]?\d+)$/ : /^(-?\d+)\s+(-?\d+)$/)) {
      if ($config::entry eq 'absp') {
	if ($1 < 0) {
	  $p->{'scores'}[$round0] = 0;
	  $opp->{'scores'}[$round0] = 0 - $1;
	  }
	else {
	  $p->{'scores'}[$round0] = 0 + $1;
	  $opp->{'scores'}[$round0] = 0;
	  }
	}
      else {
	$p->{'scores'}[$round0] = $1;
	$opp->{'scores'}[$round0] = $2;
        }
      $dp->Dirty(1);
      }
    elsif ((!$oppid) && exists $p->{'scores'}[$round0] && /^(-?\d+)$/) {
      $p->{'scores'}[$round0] = $1;
      $dp->Dirty(1);
      }
    elsif (/^ra(?:t(?:i(?:n(?:g)?)?)?)?\s+(\d+)$/) {
      $p->{'rating'} = $1;
      $dp->Dirty(1);
      }
    else {
      last;
      }
    }
  $tournament->UpdateDivisions();
  0;
  }

=back

=cut

=head1 BUGS

Should allow editing of board/table number.

Should use a subprocessor rather than an event loop.

=cut

1;
