#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::PRiZes;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::PRiZes - implement the C<tsh> PRiZes command

=head1 SYNOPSIS

  my $command = new TSH::Command::PRiZes;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::PRiZes is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);
sub ShowPrize ($$$$);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display a prize table.
EOF
  $this->{'names'} = [qw(prz prizes)];
  $this->{'argtypes'} = [qw()];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $config = $tournament->Config();

  unless (@config::prizes) {
    # Dallas 2007
    push(@config::prizes, 
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>1, 'value'=>'$4000', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>2, 'value'=>'$2000', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>3, 'value'=>'$1000 and UWG', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>4, 'value'=>'$750', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>5, 'value'=>'$500', },
      { 'division'=>'A', 'type'=>'rank', 'subtype'=>6, 'value'=>'$350', },
#     { 'division'=>'A', 'type'=>'overseed', 'subtype'=>1, 'value'=>'$200', 'groupname' => 'Group AB', 'members' => [ 1..23] },
#     { 'type' => 'signup', 'subtype' => 'Group A High Play', 'value' => 'pick' },
      { 'type' => 'signup', 'subtype' => 'High TX Word', 'value' => '$100' },
      { 'type' => 'signup', 'subtype' => 'High Upset 1', 'value' => 'SamBoard' },
      { 'type' => 'signup', 'subtype' => 'High Upset 2', 'value' => 'SamBoard' },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$300', 'groupname' => 'Group A', 'members' => [ 21..41 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$250', 'groupname' => 'Group B', 'members' => [ 42..62 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$200', 'groupname' => 'Group C', 'members' => [ 63..84 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$150', 'groupname' => 'Group D', 'members' => [ 85..106 ] },
      { 'division'=>'A', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'$100', 'groupname' => 'Group E', 'members' => [ 107..128 ] },
    );
    # Oshawa 2007
#   @config::prizes = (
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#     { 'division'=>'A', 'type'=>'rank', 'subtype'=>17, 'value'=>'Rack/Bag', },
#     { 'division'=>'A', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#     { 'division'=>'A', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division A High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division A Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'B', 'type'=>'rank', 'subtype'=>18, 'value'=>'Rack/Bag', },
#      { 'division'=>'B', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'B', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division B High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division B Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'C', 'type'=>'rank', 'subtype'=>20, 'value'=>'Rack/Bag', },
#      { 'division'=>'C', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'C', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division C High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division C Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'D', 'type'=>'rank', 'subtype'=>16, 'value'=>'Rack/Bag', },
#      { 'division'=>'D', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'D', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division D High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division D Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>1, 'value'=>'$200', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>2, 'value'=>'$150', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>3, 'value'=>'$100', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>4, 'value'=>'$50', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>5, 'value'=>'Board', },
#      { 'division'=>'E', 'type'=>'rank', 'subtype'=>18, 'value'=>'Rack/Bag', },
#      { 'division'=>'E', 'type'=>'grouprank', 'subtype'=>1, 'value'=>'Portfolio Scrabble', 'groupname' => 'Newcomer', 'members' => [ 14..18 ] },
#      { 'division'=>'E', 'type'=>'highwin', 'subtype'=>1, 'value'=>'$10', },
#      { 'division'=>'E', 'type'=>'highloss', 'subtype'=>1, 'value'=>'$10', },
#      { 'type' => 'signup', 'subtype' => 'Division E High Play', 'value' => '$10' },
#      { 'type' => 'signup', 'subtype' => 'Division E Most Outrageous Phoney', 'value' => '$10+cookies' },
#      { 'type' => 'signup', 'subtype' => 'Best Candy Word', 'value' => 'Scrabble chocolate' },
#      { 'type' => 'signup', 'subtype' => 'Best Dirty Word', 'value' => 'Bottle of wine' },
#      { 'type' => 'signup', 'subtype' => 'Early Bird', 'value' => 'cookies' },
#      );
    }
  my $max_rounds = $config->Value('max_rounds');
  if (!defined $max_rounds) {
    $tournament->TellUser('eneed_max_rounds');
    return;
    }
  my $partial = 0;
  dp:for my $dp ($tournament->Divisions()) {
    for my $p ($dp->Players()) {
      for my $r0 (0..$max_rounds-1) {
	if (!defined $p->Score($r0)) { warn ($p->Name() . " has no results in round ".($r0+1).", so standings are provisional.\n"); $partial = 1; last dp; }
	} 
      }
    }

  my $logp = new TSH::Log($tournament, undef, 'prizes', undef);
  $logp->Write("Based on PARTIAL results\n\n", "<tr><td colspan=4 class=warning>Based on PARTIAL results</td></tr>\n")
    if $partial;
  $logp->Write(sprintf("%3s %s\n\n", '#', 'Prize, Winner'), <<'EOF');
<th class=number>#</th>
<th class=description>Description</th>
<th class=value>Value</th>
<th class=winner>Winner(s)</th>
</tr>
EOF
  for my $p0 (0..$#config::prizes) {
    my $prize = $config::prizes[$p0];
    my $p1 = $p0+1;
    $logp->Write(sprintf("%3d ", $p1), "<tr><td class=number>$p1</td>");
    $this->ShowPrize($tournament, $logp, $prize);
    $logp->Write("\n", "</tr>\n");
    }
  $logp->Close();
  return 0;
  }

=item $this->ShowPrize($tournament, $logp, $prize);

Add the data for the given prize to the log.

=cut

sub ShowPrize ($$$$) {
  my $this = shift;
  my $tournament = shift;
  my $logp = shift;
  my $prize = shift;
  my $type = $prize->{'type'} || '';
  if ($type eq 'rank') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $value = $prize->{'value'} || '?';
    for my $p ($dp->Players()) {
      next unless $p->Active();
      if ($p->RoundRank($lastr0) == $subtype) {
	my $pname = sprintf("%s %g-%g %+d", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread());
	$logp->Write("Division $dname Rank $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname Rank: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      }
    }
  elsif ($type eq 'grouprank') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [];
    my $groupname = $prize->{'groupname'} || 'Group ?';
    my $value = $prize->{'value'} || '?';
    my $lastw = 0;
    my $lastl = 0;
    my $lasts = 0;
    my $rank = 0;
    my (@sorted) = TSH::Player::SortByCurrentStanding grep { defined $_ } 
      map { $dp->Player($_) } @$members;
    for my $i (0..$#sorted) {
      my $p = $sorted[$i];
      my $w = $p->Wins();
      my $l = $p->Losses();
      my $s = $p->Spread();
      if ($lastw != $w || $lastl != $l || $lasts != $s) {
	$lastw = $w;
	$lastl = $l;
	$lasts = $s;
	$rank = $i+1;
        }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %g-%g %+d (overall rank %d)", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread(), $p->RoundRank($lastr0));
	$logp->Write("$groupname Rank: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>$groupname Rank: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'highloss') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [1..$dp->CountPlayers()];
    my $groupname = (defined $prize->{'groupname'}) ?  "$prize->{'groupname'} " : '';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = grep { defined $_ } map { $dp->Player($_) } @$members;
    for my $p (@players) {
      my $hl = -9999;
      for my $r0 (0..$lastr0) {
	my $ms = $p->Score($r0);
	my $os = $p->OpponentScore($r0);
	next unless defined $ms;
	next unless defined $os;
	next unless $ms < $os;
	$hl = $ms if $hl < $ms;
        }
      $p->{'xhl'} = $hl;
      }
    @players = sort { $b->{'xhl'} <=> $a->{'xhl'} } @players;
    my $lastv = -9999;
    my $rank = 0;
    for my $i (0..$#players) {
      my $p = $players[$i];
      my $v = $p->{'xhl'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %d", $p->TaggedName(), $p->{'xhl'});
	$logp->Write("Division $dname ${groupname}High Loss: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname $groupname High Loss: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'highwin') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [1..$dp->CountPlayers()];
    my $groupname = (defined $prize->{'groupname'}) ?  "$prize->{'groupname'} " : '';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = grep { defined $_ } map { $dp->Player($_) } @$members;
    for my $p (@players) {
      my $hw = -9999;
      for my $r0 (0..$lastr0) {
	my $ms = $p->Score($r0);
	my $os = $p->OpponentScore($r0);
	next unless defined $ms;
	next unless defined $os;
	next unless $ms > $os;
	$hw = $ms if $hw < $ms;
        }
      $p->{'xhw'} = $hw;
      }
    @players = sort { $b->{'xhw'} <=> $a->{'xhw'} } @players;
    my $lastv = -9999;
    my $rank = 0;
    for my $i (0..$#players) {
      my $p = $players[$i];
      my $v = $p->{'xhw'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %d", $p->TaggedName(), $p->{'xhw'});
	$logp->Write("Division $dname ${groupname}High Win: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>Division $dname $groupname High Win: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'overseed') {
    my $dname = $prize->{'division'} || '';
    my $dp = $tournament->GetDivisionByName($dname) or do {
      $tournament->TellUser('ebaddiv', $dname);
      return;
      };
    my $lastr0 = $config::max_rounds - 1;
    $dp->ComputeRanks($lastr0);
    my $subtype = $prize->{'subtype'} || 1;
    my $members = ref($prize->{'members'}) eq 'ARRAY'
      ? $prize->{'members'} : [];
    my $groupname = $prize->{'groupname'} || 'Group ?';
    my (%pids) = map { $_ => 1 } @$members;
    my $value = $prize->{'value'} || '?';
    my (@players) = sort { $a->RoundRank($lastr0)-$a->{'id'} <=>
      $b->RoundRank($lastr0)-$b->{'id'} } map { $dp->Player($_) } @$members;
    my $lastv = -@players;
    my $rank = 0;
    for my $i (0..$#players) {
      my $p = $players[$i];
      my $v = $p->RoundRank($lastr0) - $p->{'id'};
      if ($v != $lastv) { $rank = $i + 1; $lastv = $v; }
      if ($rank == $subtype) {
	my $pname = sprintf("%s %g-%g %+d (overall rank %d)", $p->TaggedName(), $p->Wins(),
	  $p->Losses(), $p->Spread(), $p->RoundRank($lastr0));
	$logp->Write("$groupname OverSeed: $subtype; Value: $value; Winner: $pname",
	  "<td class=description>$groupname OverSeed: $subtype</td>"
	  ."<td class=value>$value</td>"
	  ."<td class=name>$pname</td>");
        }
      elsif ($rank > $subtype) { last; }
      }
    }
  elsif ($type eq 'signup') {
    my $subtype = $prize->{'subtype'} || '';
    my $value = $prize->{'value'} || '?';
    $logp->Write("$subtype; Value: $value; Winner: ",
      "<td class=description>$subtype</td>"
      ."<td class=value>$value</td>"
      ."<td class=name></td>");
    }
  }

=back

=cut

=head1 BUGS

DoNSARatings() makes unauthorized use of TSH::Player internals.

This was copied and pasted at great haste and ought to be checked
for duplicate code against RoundStandings and RATings.

Should use new ratings code to autosplit tournaments.

Should always use CalculateSplitRatings.  Require the specification
of config max_rounds, and throw out all the old code.

=cut

1;
