#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ROTO;

use strict;
use warnings;

use TSH::Utility;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ROTO - implement the C<tsh> ROTO command

=head1 SYNOPSIS

  my $command = new TSH::Command::ROTO;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ROTO is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use the ROTO command to list current standings in a rotisserie
pool, configured using "config rotofile".
EOF
  $this->{'names'} = [qw(roto)];
  $this->{'argtypes'} = [qw(BasedOnRound)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($round) = @_;
  my $round0 = $round - 1;
  unless ($config::rotofile) { $tournament->TellUser('erotocfg'); return; }
  my $fn = TSH::Config::MakeRootPath($config::rotofile);
  my $fh = TSH::Utility::OpenFile("<", $fn);
  unless ($fh) { $tournament->TellUser('erotoope', $fn, $!); return; }
  local($/) = "\n\n";
  my @teams;
  my $team_size;
  while (<$fh>) {
    s/^\n+//; 
    my (@lines) = split(/\n/);
    my %players;
    my (%data);
    for my $line (@lines) {
      next unless $line =~ /\S/;
      $line =~ s/\s+$//;
      my ($command, $args) = split(/\s+/, $line, 2);
      if ($command eq 'owner') {
	if ($data{'owner'}) {
	  $tournament->TellUser('erotoown', $., $data{'owner'}, $args);
	  return;
	  }
	$data{'owner'} = $args;
        }
      elsif ($command eq 'player') {
	my $p = $tournament->GetPlayerByName($args);
	if (!defined $p) {
	  $tournament->TellUser('erotounp', $., $args);
	  return;
	  }
	my $did = $p->Division()->Name() . $p->ID();
	if ($players{$did}++) {
	  $tournament->TellUser('erotodup', $., $args);
	  return;
	  }
	push(@{$data{'players'}}, $p);
	$data{'wins'} += $p->{'twins'} = $p->RoundWins($round0);
	$data{'spread'} += $p->{'tspread'} = $p->RoundSpread($round0);
        $p->{'twins'} =~ s/\.5/+/;
        }
      else {
	$tournament->TellUser('erotosyn', $line);
	return;
        }
      }
    if (defined $team_size) {
      my $this_team_size = scalar(@{$data{'players'}});
      if ($team_size != $this_team_size) {
	$tournament->TellUser('erotosiz', $data{'owner'}, $this_team_size,
          $team_size);
	$team_size = $this_team_size;
        }
      }
    else {
      $team_size = scalar(@{$data{'players'}});
      }
    push(@teams, \%data ) if %data;
    }
  my $logp = new TSH::Log($tournament, undef, 'roto', $round);
  $logp->Write(
    sprintf("%3s "
      . '%5s '
      . "%-30s %s\n", ' W ', 
      ' Sprd',
      'Owner', (' ' x 19) . 'Team'),
    "<th class=wins>Wins</th>"
      . '<th class=spread>Spread</th>'
      . "<th class=owner>Owner</th><th class=team colspan=$team_size>Team</th></tr>\n");
  for my $team (sort { $b->{'wins'} <=> $a->{'wins'} || 
    $b->{'spread'} <=> $a->{'spread'} ||
    lc($a->{'owner'}) cmp lc($b->{'owner'}) } @teams) {
    my $teamwins = $team->{'wins'};
    $teamwins =~ s/\.5/+/ or $teamwins .= ' ';
    $logp->Write(
      sprintf("%3s %+5d %-27.27s" . ('%6s' x $team_size) . "\n",
	$teamwins,
	$team->{'spread'},
	$team->{'owner'},
	map { sprintf("%3s%1s%03d",
	  $_->{'twins'},
	  uc($_->{'division'}{'name'}),
	  $_->{'id'},
	  )
	} @{$team->{'players'}}),
      sprintf("<tr><td class=wins>%s</td>"
        . '<td class=spread>%+d</td>'
	. "<td class=owner>%s</td>\n" 
	. ("<td class=team>%s</td>\n" x $team_size) . "</tr>\n",
	$teamwins,
	$team->{'spread'},
	$team->{'owner'},
	map { 
          my $s = $_->TaggedName();
	  if ($s =~ /(.*) \((.*)\)/) {
	    $s = sprintf("%s<br>%s %s %+d\n", $1, $2, $_->{'twins'},
	      $_->{'tspread'});
	    }
	  $s;
	  } @{$team->{'players'}}),
        );
    }
  $logp->Close();
  }

=back

=cut

1;
