#!/usr/bin/perl

# Copyright (C) 2006 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Config;

use strict;
use warnings;

use FileHandle;
use File::Copy;
use File::Path;
use File::Spec;
use File::Temp qw(tempfile);
use TSH::Division;
use TSH::Utility;
use UNIVERSAL qw(isa);
use JavaScript::Serializable;
our (@ISA);
@ISA = qw(JavaScript::Serializable);
sub EXPORT_JAVASCRIPT () { return map { $_ => $_ } qw(event_date event_name max_name_length max_rounds); }

=pod

=head1 NAME

TSH::Config - Manipulate tsh's configuration settings

=head1 SYNOPSIS

  ## newstyle usage

  my $config = new TSH::Config($tournament, $config_filename) 
    or die "no such file";
  my $config = new TSH::Config($tournament); # use default configuration file
  # in Tournament.pm
  $config->Read(); # reads from designated file
  $config->Setup(); # checks that directories exist, etc.
  # deprecated access methods
  $config->Export(); # exports into 'config' namespace
  $value = $config::key;
  $config::key = $value;
  # preferred access methods
  $value = $config->Value($key);
  $config->Value($key, $value);
  $config->Write(); # updates configuration file

  ## oldstyle usage (remove documentation after testing newstyle)

  my $config = new TSH::Config($config_filename) or die "no such file";
  my $config = new TSH::Config(); # select default configuration file
  $config->Lock() or die; # prevents other users from changing tournament data
  $config->Load($tournament); # populates package 'config' with loaded values
  $config->Unlock(); # releases Lock()
  $config::key = 'new value';
  $config->Save(); # updates configuration file from package 'config'

  ## noninteractive support
  
  @option_names = TSH::Config::UserOptions();
  $help = TSH::Config::UserOptionHelp($option_name);
  $type = TSH::Config::UserOptionType($option_name);
  $time = $config->LastModified(); 
  $js = $config->ToJavaScript();

=head1 ABSTRACT

This class manages tsh's configuration settings.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($;$);
sub new ($;$);
sub CheckDirectory ($$);
sub ChooseFile($$);
sub Export ($);
sub LastModified ($);
sub MakeBackupPath ($$);
sub MakeHTMLPath ($$);
sub MakeRootPath ($$);
sub Read ($);
sub Save ($);
sub Setup ($);
sub UserOptions ();
sub UserOptionHelp ($);
sub UserOptionType ($);
sub ValidateLocalDirectory ($$$);

# information about individual configuration options
my (%user_option_data) = (
  'assign_firsts' => { 'type' => 'boolean', 'help' => 'If checked, tsh decides who plays first (starts) or second (replies) in each game. Should be checked in the U.K. and unchecked in Canada and the United States.' },
  'backup_directory' => { 'type' => 'string', 'help' => 'Specifies where journalled ".t" and ".tsh" files are kept.', 'validate' => \&ValidateLocalDirectory, },
  'director_name' => { 'type' => 'string', 'help' => 'Gives the name of the director of this event, for use in ratings submission.' },
  'event_date' => { 'type' => 'string', 'help' => 'Gives the date(s) of this event, for use in report headers and ratings submission.' },
  'event_name' => { 'type' => 'string', 'help' => 'Gives the name of this event, for use in report headers and ratings submission.' },
  'html_directory' => { 'type' => 'string', 'help' => 'Specifies where generated web files are kept.', 'validate' => \&ValidateLocalDirectory, },
  'max_rounds' => { 'type' => 'integer', 'help' => 'Gives the number of rounds in this tournament. This parameter is mandatory when using tsh in server mode.' },
  'no_text_files' => { 'type' => 'boolean', 'help' => 'If checked, tsh creates only web HTML files. Leave this checked unless you prefer the retro look.' },
  'port' => { 'type' => 'integer', 'help' => 'If set to a nonzero value, enables the web interface that you are using and specifies its TCP/IP port number.  Do not change unless you know what this is.' },
  'track_firsts' => { 'type' => 'boolean', 'help' => 'If checked, keeps track of who played first (started) or second (replied) in each game. Should be checked in most parts of the world now.' },
  );

=item $config->initialise($tournament, $argv)

Initialise the config object.
$tournament should be of type TSH::Tournament.
$argv should contain whatever the user specified on the command line,
typically undef or the name of an event directory.

=cut

sub initialise ($;$) {
  my $this = shift;
  my $tournament = shift;
  my $argv = shift;
  # all fields should be listed here, regardless of whether they need init
  $this->{'filename'} = undef;
  $this->{'tournament'} = $tournament;

  return undef unless $this->ChooseFile($argv);
  my $time = time; # can't use utime undef, undef without AIX warnings
  utime $time, $time, $this->MakeRootPath($this->{'filename'});
  return $this;
  }

=item $d = new TSH::Config($filename);

Create a new TSH::Config object.  If the optional parameter 
$filename is omitted, a default value is chosen.

=cut

sub new ($;$) { return TSH::Utility::new(@_); }

=item CheckDirectory($path, $message);

Checks to see if $path exists, creates it if necessary and possible,
reports an error message on failure.

=cut

sub CheckDirectory ($$) {
  my ($dir, $what) = @_;
  return if -d $dir;
  mkpath $dir, 0, 0755;
  return if -d $dir;
  $_[0] = File::Spec->curdir();
  warn "Cannot create $dir, so $what will have to go in the main tsh directory.\n";
  }

=item $success = $config->ChooseFile($argv);

Used internally to choose a default configuration file.
$argv should contain whatever the user specified on the command line,
typically undef or the name of an event directory.
Sets $this->{'filename'}.
Returns true on success; dies on failure.

=cut

sub ChooseFile ($$) {
  my $this = shift;
  my $argv = shift;
  $this->{'root_directory'} = '.';
  # first choice: something specified on the command line
  if (defined $argv) {
    # if it's a directory
    if (-d $argv) {
      for my $try (qw(config.tsh tsh.config)) {
	if (-e "$argv/$try") {
	  $this->{'root_directory'} = $argv;
	  $this->{'filename'} = $try;
	  return 1;
	  }
        }
      die "$argv is a directory but has neither config.tsh nor tsh.config\n";
      }
    # else it's a configuration file
    elsif (-f $argv) {
      if ($argv =~ /\//) {
        @$this{qw(root_directory filename)} = $argv =~ /^(.*)\/(.*)/;
	return 1;
        }
      else { return $argv; }
      }
    else { die "No such file or directory: $argv\n"; }
    }
  # second choice: the directory containing the newest of */config.tsh
  elsif (my (@glob) = glob('*/config.tsh')) {
    # if more than one, choose newest
    if (@glob > 1) {
      my $newest_age = -M $glob[0];
      while (@glob > 1) {
	my $this_age = -M $glob[1];
	if ($this_age < $newest_age) {
	  $newest_age = $this_age;
	  shift @glob;
	  }
	else {
	  splice(@glob, 1, 1);
	  }
        }
      }
    @$this{qw(root_directory filename)} = $glob[0] =~ /^(.*)\/(.*)/;
#   print "Directory: $this->{'root_directory'}\n";
    if ($this->{'root_directory'} =~ /^sample\d$/) {
      print <<"EOF";
You are about to run tsh using its sample event "$this->{'root_directory'}".
This is fine for practice, but might not be want you meant.  If you
want to continue with the sample event, please press the Return key
now.  To choose a different event, enter its name and then press
the Return key.
EOF
      print "[$this->{'root_directory'}] ";
      my $choice = scalar(<STDIN>);
      $choice =~ s/^\s+//;
      $choice =~ s/\s+$//;
      $this->{'root_directory'} = $choice if $choice =~ /\S/;
      unless (-f "$this->{'root_directory'}/$this->{'filename'}") {
	die "Cannot find config.tsh in $this->{'root_directory'}/.\n";
        }
      }
    return 1;
    }
  # third choice: ./tsh.config
  if (-f 'tsh.config') {
    # don't use colour here, as colour configuration hasn't been found
    print "Warning: use of tsh.config is deprecated.\n";
    print "Please place your event files in a subdirectory, and save your\n";
    print "configuration in that subdirectory under the name config.tsh.\n";
    $this->{'filename'} = 'tsh.config';
    return 1;
    }
  die "Cannot find a configuration file.\n";
  }

=item $config->Export();

Exports $config->{'key'} to $config::key, @config::key or %config::key
depending on the type of its value.
Severely deprecated.

=cut

sub Export ($) {
  my $this = shift;
  while (my ($key, $value) = each %$this) {
    my $ref = ref($value);
    if ($ref eq '') 
      { eval "\$config::$key=\$value"; }
    elsif ($ref eq 'ARRAY')
      { eval "\@config::$key=\@\$value"; }
    elsif ($ref eq 'HASH')
      { eval "\%config::$key=\%\$value"; }
    }
  }

=item $config->LastModified();

Returns the time (in seconds since the Unix epoch) when the most
recent associated file (config.tsh or *.t) was modified.

=cut

sub LastModified ($) {
  my $this = shift;
  my $modtime;
  my (@stat) = stat $this->MakeRootPath($this->{'filename'});
  $modtime = $stat[9] if defined $stat[9];
  my (@files);
  if (defined $this->{'tournament'}) {
    @files = map { $this->MakeRootPath($_->File()) } 
      $this->{'tournament'}->Divisions();
    }
  else {
    my $divsp = $this->Load(undef, 1);
    @files = map { $this->MakeRootPath($_) } values %$divsp;
    }
  for my $fn (@files) {
    (@stat) = stat $fn;
    $modtime = $stat[9] if (defined $stat[9]) && $stat[9] > $modtime;
    }
  return $modtime;
  }


=item $path = $c->MakeBackupPath($relpath);

Return a path to a file in the configured backup directory.

=cut

sub MakeBackupPath ($$) {
  my $this = shift;
  my $relpath = shift;
  return File::Spec->file_name_is_absolute($this->{'backup_directory'})
    ? File::Spec->join($this->{'backup_directory'}, $relpath)
    : File::Spec->join($this->{'root_directory'},
      $this->{'backup_directory'}, $relpath);
  }

=item $path = $c->MakeHTMLPath($relpath);

Return a path to a file in the configured HTML directory.

=cut

sub MakeHTMLPath ($$) {
  my $this = shift;
  my $relpath = shift;
  return File::Spec->file_name_is_absolute($this->{'html_directory'})
    ? File::Spec->join($this->{'html_directory'}, $relpath)
    : $this->MakeRootPath(File::Spec->join($this->{'html_directory'}, $relpath));
  }

=item $path = $c->MakeRootPath($relpath);

Return a path to a file in the root directory

=cut

sub MakeRootPath ($$) {
  my $this = shift;
  my $relpath = shift;
  return File::Spec->file_name_is_absolute($relpath)
    ? $relpath
    : File::Spec->join($this->{'root_directory'}, $relpath)
  }

=item $config->Read();

Read the associated configuration file and set configuration values.

=cut

sub Read ($) {
  my $this = shift;
  my $fn = $this->{'filename'};
  my $tournament = $this->{'tournament'};

  # default values
  $this->{'backup_directory'} = File::Spec->catdir(File::Spec->curdir(),'old');
  $this->{'event_date'} = 'Unknown Date'; 
  $this->{'event_name'} = 'Unnamed Event'; 
  $this->{'external_path'} = [qw(./bin)];
  $this->{'html_directory'} = File::Spec->catdir(File::Spec->curdir(),'html');
  $this->{'max_name_length'} = 22;
  $this->{'name_format'} = '%-22s';
  $this->{'split1'} = 1000000;
  $this->{'table_format'} = '%3s';
  $this->{'table_title'} = 'Table';
  my $fqfn = $this->MakeRootPath($fn);
  my $fh = new FileHandle($fqfn, "<")
    or die "Can't open $fn: $!\n";
  local($_);
  $tournament->TellUser('iloadcfg', $fqfn);
  while (<$fh>) { s/(?:^|[^\\])#.*//; s/^\s*//; s/\s*$//; next unless /\S/;
    if (/^division?\s+(\S+)\s+(.*)/i) {
      my $dname = $1;
      my $dfile = $2;
      my $dp = new TSH::Division;
      $dp->Name($dname);
      $dp->File($dfile);
      $tournament->AddDivision($dp);
      next;
      }
    if (s/^perl\s+//i) { 
      local(*main::gTournament) = \%$tournament; # backward compatibility
      eval $_;
      print "eval: $@\n" if length($@);
      next;
      }
    if (/^config\s+(\w+)\s*(.*?)\s*=\s*(.*)/i) { 
      my $s = "\$this->{'$1'}$2=$3";
      eval $s;
      if ($@) {
	$tournament->TellUser('ebadcfg', $_);
	exit(1);
        }
      next;
      }
    if (s/^autopair\s+//i) { 
      if (/^(\w+)\s+(\d+)\s+(\d+)\s+(\w+)\s+(.*)/) {
	my ($div, $sr, $round, $command, $args) = ($1, $2, $3, $4, $5);
	if ($sr >= $round) {
	  $tournament->TellUser('eapbr', $div, $sr, $round);
	  exit(1);
	  }
	my (@args) = split(/\s+/, $args);
	$this->{'autopair'}{uc $div}[$round] = [$sr, $command, @args];
        }
      else {
	chomp;
        $tournament->TellUser('ebadap', $_, $fn);
	exit(1);
        }
      next;
      }
    $tournament->TellUser('ebadcfg', $_);
    exit(1);
    }
  close($fh); 
# print "Configuration file loaded.\n";

  # computed default values
  unless ($this->{'player_number_format'}) {
    $this->{'player_number_format'} = 
      $tournament->CountDivisions() == 1 ? '#%s' : '%s';
    }
  for my $div (keys %{$this->{'gibson_groups'}}) {
    my $divp = $this->{'gibson_groups'}{$div};
    for my $gibson_group (@$divp) {
      my $first = $gibson_group->[0];
      for my $i (1..$#$gibson_group) {
	$this->{'gibson_equivalent'}{$div}[$gibson_group->[$i]] = $first;
#	print "$gibson_group->[$i] equiv $first in $div.\n";
	}
      }
    }
  if (defined $this->{'entry'}) {
    if ($this->{'entry'} =~ /^absp$/i) { 
      $this->{'entry'} = 'absp';
      }
    elsif ($this->{'entry'} =~ /^nsa$/i) {
      $this->{'entry'} = 'nsa';
      }
    else {
      $tournament->TellUser('ebadconfigentry', $this->{'entry'});
      $this->{'entry'} = 'nsa';
      }
    if (defined $this->{'assign_firsts'}) {
      if (!$this->{'assign_firsts'}) {
	$tournament->TellUser('wsetassignfirsts');
	}
      }
    $this->{'assign_firsts'} = 1;
    if (defined $this->{'track_firsts'}) {
      if (!$this->{'track_firsts'}) {
	$tournament->TellUser('wsettrackfirsts');
	}
      }
    $this->{'track_firsts'} = 1;
    }
  else {
    $this->{'entry'} = 'nsa';
    }
  }

=item $config->Save();

Save the current configuration in its configuration file.

=cut

sub Save ($) {
  my $this = shift;
  my $fn = $this->MakeRootPath($this->{'filename'});
  my $fh = new FileHandle($fn, "<");
  unless ($fh) {
    return "<div class=failure>Could not open '$fn': $!</div>";
    }
  local($/) = undef;
  my $s = <$fh>;
  close($fh);
  # update user options
  while (my ($key, $datap) = each %user_option_data) {
    my $value = ${$config::{$key}};
    my $type = $datap->{'type'};
    unless (defined $value) {
      if ($type eq 'boolean') { $value = 0; }
      elsif ($type eq 'integer') { $value = 0; }
      elsif ($type eq 'string') { $value = ''; }
      else { die "oops - unknown type $datap->{'type'} for $key\n"; }
      }
    if ($type eq 'string') { $value = '"' . quotemeta($value) . '"'; }
    $s =~ s/^\s*config\s+$key\s*=.*$/config $key = $value/m
      or $s .= "config $key = $value\n";
    }
  my $error = '';
  copy($fn, $this->{'backup_directory'} . time . '.tsh')
    or ($error = "A backup copy of the configuration could not be saved: $!");
  TSH::Utility::ReplaceFile($fn, $s)
    or ($error = "A new copy of the configuration could not be written: $!");
  return $error 
    ? "<div class=failure>Your changes were not saved. $error</div>"
    : "<div class=success>Your changes were saved.</div>";
  }

=item $c->Setup();

Perform any initialisation required after the configuration has been
read: create necessary directories.

=cut

sub Setup ($) {
  my $this = shift;
  # Check for backup and HTML directories
  CheckDirectory ($this->MakeRootPath($this->{'backup_directory'}), "backups");
  CheckDirectory ($this->MakeRootPath($this->{'html_directory'}), "web files");
  unless (-f $this->MakeHTMLPath('tsh.css')) {
    my $fh;
    $fh = new FileHandle("lib/tsh.css", "<");
    if ($fh) {
      local($/) = undef;
      my $css = <$fh>;
      $fh->close();
      $fh = new FileHandle($this->MakeHTMLPath('tsh.css'), ">");
      print $fh $css;
      $fh->close();
      }
    }
  }

=item @option_names = UserOptions();

Returns a list of user-configurable option names.

=cut

sub UserOptions () {
  return keys %user_option_data;
  }

=item $help = UserOptionHelp($option_name);

Returns the help text associated with a user-configurable option name.

=cut

sub UserOptionHelp ($) {
  my $key = shift;
  return $user_option_data{$key}{'help'};
  }

=item $type = UserOptionType($option_name);

Returns the type of a user-configurable option name:
boolean, integer or string.

=cut

sub UserOptionType ($) {
  my $key = shift;
  return $user_option_data{$key}{'type'};
  }

=item $error = UserOptionValidate($option_name, $option_value);

Checks to see if $option_value is a valid value for $option_name.
If it isn't, an error message is returned.  If it is, $option_value
may be cleaned slightly before being returned.

=cut

sub UserOptionValidate ($$) {
  my $key = shift;
  my $sub = $user_option_data{$key}{'validate'};
  return '' unless $sub;
  return &$sub($key, $_[0]);
  }

=item $error = $config->ValidateLocalDirectory($option_name, $option_value);

Returns an error message if $option_value is not a valid value for $option_name.
Returns the null string if it is valid.
Cleans up $option_value if necessary.
Checks to see if the value designates a local directory that exists
or can be created.
Paths may be relative to C<$config->{'root_directory'}>.

=cut

sub ValidateLocalDirectory ($$$) {
  my $this = shift;
  my $key = shift;
  my $path = $_[0];
  $path =~ s/^\s+//;
  $path =~ s/\s+$//;
  $path = '/' unless $_[0] =~ /\/$/;
  $path = File::Spec->canonpath($path);
  my $fqpath = $path;
  if (!File::Spec->file_name_is_absolute($path)) {
    $fqpath = File::Spec->join($this->{'root_directory'}, $path);
    }
  if (-e $fqpath) { 
    unless (-d $fqpath) {
      return "$path exists but is not a directory.";
      }
    }
  else {
    mkpath $fqpath, 0, 0755
      or return "$fqpath does not exist and cannot be created.";
    }
  $_[0] = $path;
  return 1;
  }

=item $value = $c->Value($key);
=item $c->Value($key, $value);

Get/set a configuration value.

=cut

sub Value ($;$) { my $this = shift; my $key = shift; TSH::Utility::GetOrSet($key, $this, @_); }

=head1 BUGS

Should create a configuration file if none is found.

Should offer to create a configuration file when the ChooseFile
returns a sample directory.

tshxcfg.txt should be read in and eval'ed, so that its code can
have access to a lexical copy of $tournament.

Should eventually remove Export().

=cut

1;
