#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Tournament;

use strict;
use warnings;

use Fcntl ':flock';
use FileHandle;
use JavaScript::Serializable;
use TSH::Config;
use TSH::Utility;
use UserMessage;
our (@ISA);
@ISA = qw(JavaScript::Serializable);
sub EXPORT_JAVASCRIPT () { return ('divlist' => 'divisions', 'config' => 'config'); }

=pod

=head1 NAME

TSH::Tournament - abstraction of a Scrabble tournament within C<tsh>

=head1 SYNOPSIS

  $t = new Tournament($eventdir);
  $t->LoadConfiguration();
  $t->Config()->Export(); # deprecated
  $t->TellUser('iwelcome', $gkVersion);
  $t->Lock();
  $t->LoadDivisions();
  # edit the tournament data here
  $t->Unlock();

  $um = $t->UserMessage(); # see UserMessage.pm;
  $p = GetPlayerByName($pname);
  $t->AddDivision($d);
  $n = $t->CountDivisions();
  @d = $t->Divisions();
  $d = $t->GetDivisionByName($dn);
  $c = $t->Config();
  $t->Explain($code);
  $t->RegisterPlayer($p);
  $t->TellUser($code, @args);

=head1 ABSTRACT

This Perl module is used to manipulate tournaments within C<tsh>.

=head1 DESCRIPTION

=over 4

=cut

sub AddDivision ($$);
sub Config ($;$);
sub CountDivisions ($);
sub Divisions ($);
sub ErrorFilter($$$);
sub Explain ($;$);
sub ExplainFilter ($$);
sub FindPlayer ($$$$);
sub GetPlayerByName ($$);
sub GetDivisionByName ($$);
sub initialise ($;$);
sub LoadConfiguration ($$);
sub LoadDivisions ($);
sub Lock ($);
sub new ($;$);
sub NoteFilter ($$$);
sub RegisterPlayer($$);
sub TellUser ($$@);
sub Unlock ($);
sub WarningFilter($$$);

=item $t->AddDivision($d)

Add a division to the tournament.

=cut

sub AddDivision ($$) { 
  my $this = shift;
  my $dp = shift;
  my $dname = $dp->Name();
  if (exists $this->{'divhash'}{$dname}) {
    die "Duplicate division: $dname.\nAborting";
    }
  push(@{$this->{'divlist'}}, $dp);
  $this->{'divhash'}{$dname} = $dp;
  $dp->Tournament($this);
  }

=item $c = $t->Config();
=item $t->Config($c);

Get/set a tournament's configuration object.

=cut

sub Config ($;$) { TSH::Utility::GetOrSet('config', @_); }

=item $n = $t->CountDivisions()

Count how many divisions a tournament has.

=cut

sub CountDivisions ($) { 
  my $this = shift;
  return scalar(@{$this->{'divlist'}});
  }

=item @d = $t->Divisions()

Return a list of a tournament's divisions.

=cut

sub Divisions ($) { 
  my $this = shift;
  return @{$this->{'divlist'}};
  }

=item ErrorFilter($code, $type, $message);

Callback subroutine used by UserMessage.pm

=cut

sub ErrorFilter ($$$) {
  my $code = shift;
  my $type = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    print STDERR "Error: $message [$code]\n";
    return;
    }
  TSH::Utility::PrintColour 'red', "Error: $message";
  print " [$code]\n";
  }

=item $t->Explain();
=item $t->Explain($code);

Explain a message code (or the last one) to the user.

=cut

sub Explain ($;$) {
  my $this = shift;
  my $code = shift;

  $this->{'message'}->Explain($code) ||
  $this->TellUser('ebadhuh', $code);
  }

=item ExplainFilter($code, $message);

Callback subroutine used by UserMessage.pm

=cut

sub ExplainFilter ($$) {
  my $code = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    return;
    }
  print TSH::Utility::Wrap(0, "[$code] $message");
  }

=item $pp = $t->FindPlayer($name1, $name2, $dp);

Find a player whose name matches /$name1.*,$name2/ in division $dp.
If $dp is null, look in all divisions.

=cut

sub FindPlayer ($$$$) {
  my $this = shift;
  my $name1 = shift;
  my $name2 = shift;
  my $dp = shift;
  my $dname = $dp && $dp->Name();
  my $pattern = qr/$name1.*$name2/i;
  my @matched;
  while (my ($name, $pp) = each %{$this->{'pbyname'}}) {
    next unless $name =~ /$pattern/;
    next if $dname && $pp->Division->Name() ne $dname;
    push(@matched, $pp);
    }
  if (@matched == 0) { $this->TellUser('enomatch', "$name1,$name2"); }
  elsif (@matched == 1) { return $matched[0]; }
  elsif (@matched <= 10) {
    $this->TellUser('emultmatch', "$name1,$name2", 
      join('; ', map { $_->TaggedName() } @matched));
    }
  else { $this->TellUser('emanymatch', "$name1,$name2"); }
  return undef;
  }

=item $pp = $t->GetPlayerByName($name);

Obtain a Player pointer given a player name.

=cut

sub GetPlayerByName ($$) {
  my $this = shift;
  my $pname = shift;
  return $this->{'pbyname'}{$pname};
  }

=item $d = $t->GetDivisionByName($dn);

Obtain a Division pointer given the division's name in not 
necessarily canonical style.

=cut

sub GetDivisionByName ($$) {
  my $this = shift;
  my $dname = shift;
  $dname = TSH::Division::CanonicaliseName($dname);
  return $this->{'divhash'}{$dname};
  }

=item $t->initialise();

(Re)initialise a Tournament object, for internal use.

=cut

sub initialise ($;$) {
  my $this = shift;
  my $dir = shift;
  # all fields should be listed here, regardless of whether they need init
  $this->{'cmdhash'} = {};
  $this->{'cmdlist'} = [];
  $this->{'config'} = undef;
  $this->{'config'} = new TSH::Config($this, $dir) 
    or die "TSH::Tournament::initialise: could not load configuration";
  $this->{'divhash'} = {};
  $this->{'divlist'} = [];
  $this->{'lockfh'} = undef;
  $this->{'message'} = undef;
  $this->{'pbyname'} = {};

  my $um = $this->{'message'} 
    = new UserMessage($config::message_file || 'lib/messages.txt');
  $um->SetErrorFilter(\&ErrorFilter);
  $um->SetExplainFilter(\&ExplainFilter);
  $um->SetNoteFilter(\&NoteFilter);
  $um->SetWarningFilter(\&WarningFilter);
  }

=item $t->LoadConfiguration($dir);

Read and check configuration file for this tournament.

=cut

sub LoadConfiguration ($$) {
  my $this = shift;
  my $dir = shift;
  $this->{'config'}->Read();
  $this->{'config'}->Setup();
  }

=item $t->LoadDivisions();

Read and parse one .t file for each division to load its data.
Can take a while to run for a large tournament.

=cut

sub LoadDivisions ($) {
  my $this = shift;
  for my $dp (@{$this->{'divlist'}}) {
    $dp->Read();
    }
  }

=item $error = $t->Lock();

At most one process should have a tournament open for writing at
any time.  This is enforced by Lock() and Unlock().

It is safe for other processes to read data without obtaining a lock,
because it is always written to a separate temporary file before
being moved into place.

Dies if something unexpected goes wrong.
Returns an error message if flock() itself fails, typically because
another process already had the lock.
Returns the empty string on success.

=cut

sub Lock ($) {
  my $this = shift;
  my $error;

  (defined $this->{'config'})
    or die "TSH::Tournament::Lock: configuration file has not yet been read.\n";
  my $fn = $this->{'config'}->MakeRootPath('tsh.lock');
  my $fh = $this->{'lockfh'} = new FileHandle $fn, O_CREAT | O_RDWR
    or die "Can't open tsh.lock, probably because of a file or directory permission error ($!).\n";
  flock($fh, LOCK_EX | LOCK_NB) 
    or return $!;
  seek($fh, 0, 0) 
    or die "Can't rewind tsh.lock - something is seriously wrong.\n";
  truncate($fh, 0) 
    or die "Can't truncate tsh.lock - something is seriously wrong.\n";
  print $fh "$$\n"
    or die "Can't update tsh.lock - something is seriously wrong.\n";
  return '';
  } 

=item $t = new Tournament;

Create a new Tournament object.  

=cut

sub new ($;$) { return TSH::Utility::new(@_); }

=item NoteFilter($code, $type, $message);

Callback subroutine used by UserMessage.pm

=cut

sub NoteFilter ($$$) {
  my $code = shift;
  my $type = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    return;
    }
  print "$message [$code]\n";
  }

=item $t->RegisterPlayer($p);

Register a player so that they can subsequently be looked up by name.

=cut

sub RegisterPlayer ($$) {
  my $this = shift;
  my $p = shift;
  $this->{'pbyname'}{$p->Name()} = $p;
  }

=item $t->TellUser($code, @args);

Tell the user something important.

=cut

sub TellUser ($$@) {
  my $this = shift;
  my $code = shift;
  my @args = @_;

  $this->{'message'}->Show($code, @args);
  }

=item $t->Unlock();

Releases a lock created by Lock().  Will die on serious error.

=cut

sub Unlock ($) {
  my $this = shift;
  my $fh = $this->{'lockfh'};
  (defined $fh)
    or die "Can't unlock configuration: no lock handle exists.\n";
  flock($fh, LOCK_UN)
    or die "Can't unlock tsh.lock - something is seriously wrong.\n";
  close($fh)
    or die "Can't close tsh.lock - something is seriously wrong.\n";
  $this->{'lockfh'} = undef;
  }

=item $t->UpdateDivisions(\%dirty_divisions);

Update divisions that have been marked as dirty.

=cut

sub UpdateDivisions ($) {
  my $this = shift;
  die if defined shift;
  for my $dp ($this->Divisions()) {
    next unless $dp->Dirty();
    my $dname = $dp->Name();
    print "Updating Division $dname.\n";
    $dp->Update();
    }
  }

=item $um = $t->UserMessage();
=item $t->UserMessage($um);

Get/set the tournament's user message object, see UserMessage.pm

=cut

sub UserMessage ($;$) { TSH::Utility::GetOrSet('message', @_); }

=item WarningFilter($code, $type, $message);

Callback subroutine used by UserMessage.pm

=cut

sub WarningFilter ($$$) {
  my $code = shift;
  my $type = shift;
  my $message = shift;

  if (!-t STDOUT) { 
    print STDERR "Warning: $message [$code]\n";
    return;
    }
  TSH::Utility::PrintColour 'red', 'Warning: ';
  print "$message [$code]\n";
  }

=back

=cut

=head1 BUGS

None reported yet.

=cut

1;
