#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TFile;

=pod

=head1 NAME

TFile - manipulate John Chew's Scrabble tournament .t files 

=head1 SYNOPSIS

  my $tf = new TFile 'a.t';
  while (my $datap = $tf->ReadLine()) {
    print "$datap->{'name'}\n";
    }
  $tf->Close();

  
=head1 ABSTRACT

This Perl module is used to read tournament data files in '.t' file format.

=head1 DESCRIPTION

=over 4

=cut

use strict;
use warnings;
use FileHandle;

sub new ($$);
sub Close($);
sub FormatLine($);
sub ParseLine($);
sub ReadLine($);

=item $tf = new TFile($filename);

Create a new TFile object.  
Opens file, returns undef on failure.

=cut

sub new ($$) {
  my $proto = shift;
  my $filename = shift;
  my $class = ref($proto) || $proto;
# my $this = $class->SUPER::new();
  my $fh = new FileHandle "<$filename" or return undef;
  local($/) = undef;
  my $this = {
    'filename' => $filename,
    'cursor' => 1, # next line to return
    'lines' => [ 
      undef,
      map { { 'text' => $_ } }
      grep { /[^\s;]/ } 
      map { 
	s/#.*//;
	s/\s+$//;
	s/^\s+//; 
	s/$/;/ unless /;/;
	$_;
      } split(/[\012\015]+/, scalar(<$fh>))
      ],
    };
  $fh->close();
  bless($this, $class);
  return $this;
  }

=item $success = $fh->Close();

Explicitly closes the .t file.

=cut

sub Close ($) {
  my $this = shift;
  
  # Currently a nop, because we have to slurp the whole file in one go
  # to handle different line breaks.  
  return 1;
  }

=item $line = FormatLine($datap);

Recreate a formatted $line from its parsed data.

=cut

sub FormatLine ($) {
  my $p = shift;
  my $s = sprintf("%-22s %4d %s; %s",
    $p->{'name'},
    $p->{'rating'},
    join(' ', map { (defined $_) ? $_ : '00' } @{$p->{'pairings'}}),
    join(' ', map { (defined $_) ? $_ : '00' } @{$p->{'scores'}}));
  if ($p->{'etc'}) {
    my $etcp = $p->{'etc'};
    for my $key (sort keys %$etcp) {
      if (my $wordsp = $etcp->{$key}) {
	for my $i (0..$#$wordsp) { warn "undefined value at i=$i in $key among @$wordsp for $p->{'name'}" unless defined $wordsp->[$i]; }
	$s .= "; $key @$wordsp";
        }
      }
    }
  $s .= "\n";
  return $s;
  }

=item $datap = $fh->ReadLine();

Read and parse one line from the file.
Returns a hash whose keys are C<id>, C<name>, C<rating>, C<pairings>,
C<scores>, C<rnd> and C<etc>.

=cut

sub ReadLine ($) {
  my $this = shift;

  my $id = $this->{'cursor'};
  my $linep = $this->{'lines'}[$id];
  my $text = $linep->{'text'};
  return undef unless defined $text;
  return $linep->{'data'} if $linep->{'data'};
  my $data = $linep->{'data'} = ParseLine $text
    or die "Can't parse line $id in $this->{'filename'}: $text\n";
  $data->{'id'} = $id;
  $data->{'rnd'} = ((length($text) * (100+$this->{'cursor'}) * ord($text)) 
    % 641);
  $this->{'cursor'}++;
  return $data;
  }

=item $datap = ParseLine($line) 

Parse a received input line into a data structure.

=cut

sub ParseLine ($) {
  my $s = shift;
  my($player, $rating, $pairings, $scores, $etc) 
    = $s =~ /^([^;]+[^;\s\d])\s+(\d+)\s*([\d\s]*);\s*([-\d\s]*)((?:;[^;]*)*)$/;
  unless (defined $scores) {
    warn "Can't parse: $s\n";
    return undef;
    }
  my %data = (
#     'division' => $dp,
    'name'     => $player,
    'rating'   => $rating,
#     'rnd'=>rand,
    'pairings' => [map { $_ eq '00' ? undef : $_ } split(/\s+/, $pairings)],
    'scores'   => [map { $_ eq '00' ? undef : $_ } split(/\s+/, $scores)],
    );
  for my $extra (split(/;\s*/, $etc)) {
    next unless $extra =~ /\S/;
    my ($tag, @words) = split(/\s+/, $extra);
    warn "Overwriting duplicate $tag field for $player.\n"
      if exists $data{$tag};
    $data{'etc'}{$tag} = \@words;
    }
  return \%data;
  }

=back

=cut

1;
