#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::CheckRoundScores;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::CheckRoundScores - implement the C<tsh> CheckRoundScores command

=head1 SYNOPSIS

  my $command = new TSH::Command::CheckRoundScores;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::CheckRoundScores is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display ratings estimates based on standings
in a given round within a division.
EOF
  $this->{'names'} = [qw(crs checkroundscores)];
  $this->{'argtypes'} = [qw(Round Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

# TODO: split this up into smaller subs for maintainability

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($r1, $dp) = @_;
  my $r0 = $r1 - 1;

  my $logp = new TSH::Log($tournament, $dp, 'scores', $r0+1);
  $logp->Write("#   Score #   Score\n\n", <<'EOF');
<th class=name>#</th>
<th class=score>Score</th>
<th class=name>#</th>
<th class=score>Score</th>
EOF
  for my $p ($dp->Players()) {
    next if $config::track_firsts && $p->First($r0) == 2;
    my $mid = $p->ID();
    my $ms = $p->Score($r0) || 0;
    my $os = 0;
    my $oid = 0;
    my $opp = $p->Opponent($r0);
    if ($opp) {
      $os = $opp->Score($r0) || 0;
      $oid = $opp->ID();
      }
    $logp->Write(sprintf("%03d %5d %03d %5d\n", $mid, $ms, $oid, $os), <<"EOF");
<tr>
<td class=name>$mid</td>
<td class=score>$ms</td>
<td class=name>$oid</td>
<td class=score>$os</td>
</tr>
EOF
    }
  $logp->Close();
  return 0;
  }

=back

=cut

=head1 BUGS

DoNSARatings() makes unauthorized use of TSH::Player internals.

This was copied and pasted at great haste and ought to be checked
for duplicate code against RoundStandings and RATings.

Should use new ratings code to autosplit tournaments.

Should always use CalculateSplitRatings.  Require the specification
of config max_rounds, and throw out all the old code.

=cut

1;
