#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::RoundRATings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::RoundRATings - implement the C<tsh> RoundRATings command

=head1 SYNOPSIS

  my $command = new TSH::Command::RoundRATings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::RoundRATings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub DoABSPRatings ($$$);
sub DoNSARatings ($$$);
sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item DoABSPRatings($tournament, $dp, $r0)

Display ratings estimates as calculated by the Association of
British Scrabble Players.

=cut

# if we don't include the following line, the parameter passing fails,
# because the ABSP 
sub ABSP::CalculateRatings (\@);

sub DoABSPRatings ($$$) {
  my $tournament = shift;
  my $dp = shift;
  my $r0 = shift;
  my $datap = $dp->{'data'};

  eval 'require ABSP';
  if ($@) {
    $tournament->TellUser('enomod', 'ABSP.pm');
    return;
    }
  ABSP::CalculateRatings @$datap;

  my $has_classes = $dp->Classes();
  my $logp = new TSH::Log($tournament, $dp, 'ratings', $r0+1);
  $logp->Write("Rank  Won-Lost Spread InR ToR +-R ", <<'EOF');
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
<th class=rating>Input<br>Rating</th>
<th class=rating>Tournament<br>Rating</th>
<th class=rating>Rating<br>Change</th>
EOF
  my $text_data_format = "%4d %4.1f-%4.1f %+5d  %3s %3s %3s ";
  my $html_data_format = <<'EOF';
<tr>
<td class=rank>%d</td>
<td class=wl>%4.1f-%4.1f</td>
<td class=spread>%+d</td>
<td class=rating>%d</td>
<td class=rating>%d</td>
<td class=rating>%s</td>
EOF
  if ($has_classes) {
    $logp->Write("C ", "<th class=pclass>Class</th>\n");
    $text_data_format .= "%1s ";
    $html_data_format .= "<td class=pclass>%s</td>\n";
    }
  $logp->Write("Player\n\n", <<'EOF');
<th class=name>Name</th>
</tr>
EOF
  $text_data_format .= "%s\n";
  $html_data_format .= <<'EOF';
<td class=name>%s</td>
</tr>
EOF

  $dp->ComputeRanks($r0);
  my $i = 0;
  for my $p (TSH::Player::SortByStanding($r0, $dp->Players())) {
    next unless $p->Active();
    my $rating = $p->Rating();
    my $newr = $p->NewRating();
    $i++;
    my (@fields) = (
      $p->RoundRank($r0),
      $p->RoundWins($r0),
      $p->RoundLosses($r0),
      $p->RoundSpread($r0),
      $rating,
      $newr,
      $rating ? sprintf("%+3d", $newr-$rating) : ''
      );
    push(@fields, $p->Class()) if $has_classes;
    push(@fields, $p->TaggedName());
    $logp->Write(
      sprintf($text_data_format, @fields),
      sprintf($html_data_format, @fields));
    }
  $logp->Close();
  0;
  }

=item DoNSARatings($tournament, $dp, $r0)

Display ratings estimates as calculated by the National Scrabble
Association.

=cut

sub DoNSARatings ($$$) {
  my $tournament = shift;
  my $dp = shift;
  my $r0 = shift;

  # prepare CalculateRatings arguments - ugh
  my (@ps);
  for my $p ($dp->Players()) {
    my $id = $p->ID();
#   print $id-1, ' => ', $p->Name(), "\n";
    $ps[$id-1] = {
      'name' => $p->Name(),
      'oldr' => $p->Rating(),
      'pairings' => [ map { ($_||0)-1 } @{$p->{'pairings'}}[0..$r0] ],
      'scores' => [ map { $_||0 } @{$p->{'scores'}}[0..$r0] ],
      'lifeg' => $p->{'etc'}{'lifeg'}[0] || 100,
      'id' => $id,
      };
    }
  for my $lib (qw(ratings ratings2)) {
    eval 'require "./$lib.pl"';
    if ($@) {
      $tournament->TellUser('enomod', "$lib.pl");
      return;
      }
    }
  if ($config::rating_system && $config::rating_system =~ /lct/i) {
    &ratings2::UseClubMultipliers(1);
    }
# &ratings2::CalculateRatings(\@ps, 'oldr', 1, 'newr', 10000, 'ewins');
# &ratings2::CalculateRatings(\@ps, 'oldr', 1, 'midr', $config::split1, 'ewins1');
# &ratings2::CalculateRatings(\@ps, 'midr', $config::split1+1, 'newr', 10000, 'ewins2');
  &ratings2::CalculateSplitRatings(\@ps, ($config::max_rounds || $dp->MostScores()),
    {
      'ewins' => 'ewins',
      'lifeg' => 'lifeg',
      'newr' => 'newr',
      'oldr' => 'oldr',
      'rgames' => 'rgames',
      'pairings' => 'pairings',
      'scores' => 'scores',
      'splitr' => 'splitr',
    });

  my $ms1 = $dp->{'maxs'}+1;
  my $has_classes = $dp->Classes();
  my $logp = new TSH::Log($tournament, $dp, 'ratings', $r0+1);
  my $headings = <<'EOF';
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
<th class=rating>Old<br>Rating</th>
<th class=rating>New<br>Rating</th>
<th class=rating>Rating<br>Change</th>
EOF
  $logp->Write("Rank  Won-Lost Spread OldR NewR Delta ", '');
  my $text_data_format = "%4d %4.1f-%4.1f %+5d  %4d %4d %5s ";
  my $html_data_format = <<'EOF';
<tr style="page-break-inside:avoid">
<td class=rank>%d</td>
<td class=wl>%4.1f-%4.1f</td>
<td class=spread>%+d</td>
<td class=rating>%d</td>
<td class=rating>%d</td>
<td class=rating>%s</td>
EOF
  if ($has_classes) {
    $logp->Write('C ',  '');
    $headings .= "<th class=pclass>Class</th>\n";
    $text_data_format .= "%1s ";
    $html_data_format .= "<td class=pclass>%s</td>\n";
    }
  $logp->Write("Player\n\n", '');
  $headings .= "<th class=name>Name</th>\n";
  $text_data_format .= "%s\n";
  $html_data_format .= "<td class=name>%s</td>\n";
  if ($dp->LastPairedRound0() > $r0) {
    $headings .= "<th class=oppname>Next Opponent</th>\n";
    }
  $logp->Write('', "$headings</tr>");

  $dp->ComputeRanks($r0);
  my $i = 0;
  for my $p (TSH::Player::SortByStanding($r0, $dp->Players())) {
    next unless $p->Active();
#   if ($config::pair_page_break && $i && ($i) % $config::pair_page_break == 0) { $logp->Write('',qq(</table><table class=ratings align=center cellspacing=0 style="page-break-before:always"><tr>$headings</tr>)); }
    $i++;
    my $rating = $p->Rating();
    my $newr = $ps[$p->ID()-1]{'newr'};
    my (@fields) = (
      $p->RoundRank($r0),
      $p->RoundWins($r0),
      $p->RoundLosses($r0),
      $p->RoundSpread($r0),
      $rating, 
      $newr,
      $rating ? sprintf("%+5d", $newr-$rating) : '',
      );
    push(@fields, $p->Class()) if $has_classes;
    push(@fields, $p->TaggedName());
    $logp->Write(
      sprintf($text_data_format, @fields),
      sprintf($html_data_format, @fields));
    if (my $opp = $p->Opponent($r0+1)) {
      $logp->Write('',
	sprintf("<td class=oppname>%s</td>\n", $opp->TaggedName()));
      }
    $logp->Write('', "</tr>");
    }
  $logp->Close();
  0;
  }

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display ratings estimates based on standings
in a given round within a division.
EOF
  $this->{'names'} = [qw(rrat roundratings)];
  $this->{'argtypes'} = [qw(Round Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($r1, $dp) = @_;
  my $r0 = $r1 - 1;

  $dp->CheckRoundHasResults($r0) or return 0;
      
  if ($config::rating_system && $config::rating_system =~ /^absp$/i) {
    DoABSPRatings $tournament, $dp, $r0;
    }
  else {
    DoNSARatings $tournament, $dp, $r0;
    }
  return 0;
  }

=back

=cut

=head1 BUGS

DoNSARatings() makes unauthorized use of TSH::Player internals.

=cut

1;
