#!/usr/bin/perl

# Copyright (C) 2007 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::ShowDivisionScoreCards;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility;
use TSH::Report::CSC;

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::ShowDivisionScoreCards - implement the C<tsh> ShowDivisionScoreCards command

=head1 SYNOPSIS

  my $command = new TSH::Command::ShowDivisionScoreCards;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::ShowDivisionScoreCards is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to generate a file containing an electronic version
of all the player
scorecards in a division.
As a side effect, it will assign boards for all rounds which have pairings.
Specify the name of the division whose scorecards you wish to list.
EOF
  $this->{'names'} = [qw(sdsc showdivisionscorecards)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my ($dp) = @_;
  my @text;
  my @html;
  my $sr0 = $dp->MostScores()-1;
  for my $r0 (0..$dp->LastPairedRound0()) {
    $dp->ComputeBoards($sr0, $r0);
    }
  for my $p ($dp->Players()) {
    my $csc = new TSH::Report::CSC($p);
    push(@text, $csc->GetText());
    push(@html, $csc->GetHTML());
    }
  my $logp = new TSH::Log($tournament, $dp, 'scorecard', '',
    { 'title' => "Division Scorecards", 'notitle' => 1, 'noconsole' => 1});
  $logp->Write(join("\f", @text), join(qq(</table><div style="page-break-after:always">&nbsp;</div><table class="scorecard" align=center cellspacing=0><tr class=top1>), @html));
  $logp->Close();
  print "Generated all scorecards for division " . $dp->Name() . ".\n";
  }

=back

=cut

=head1 BUGS

Code makes inappropriate use of private fields in Player.pm.

=cut

1;
