#!/usr/bin/perl

# Copyright (C) 2005 John J. Chew, III <jjchew@math.utoronto.ca>
# All Rights Reserved

package TSH::Command::TeamStandings;

use strict;
use warnings;

use TSH::Log;
use TSH::Utility qw(Debug DebugOn);

# DebugOn('SP');

our (@ISA) = qw(TSH::Command);

=pod

=head1 NAME

TSH::Command::TeamStandings - implement the C<tsh> TeamStandings command

=head1 SYNOPSIS

  my $command = new TSH::Command::TeamStandings;
  my $argsp = $command->ArgumentTypes();
  my $helptext = $command->Help();
  my (@names) = $command->Names();
  $command->Run($tournament, @parsed_arguments);
  
=head1 ABSTRACT

TSH::Command::TeamStandings is a subclass of TSH::Command.

=cut

=head1 DESCRIPTION

=over 4

=cut

sub initialise ($$$$);
sub new ($);
sub Run ($$@);

=item $parserp->initialise()

Used internally to (re)initialise the object.

=cut

sub initialise ($$$$) {
  my $this = shift;
  my $path = shift;
  my $namesp = shift;
  my $argtypesp = shift;

  $this->{'help'} = <<'EOF';
Use this command to display standings as of a specific round.
EOF
  $this->{'names'} = [qw(ts teamstandings)];
  $this->{'argtypes'} = [qw(Division)];
# print "names=@$namesp argtypes=@$argtypesp\n";

  return $this;
  }

sub new ($) { return TSH::Utility::new(@_); }

=item $command->Run($tournament, @parsed_args)

Should run the command in the context of the given
tournament with the specified parsed arguments.

=cut

sub Run ($$@) { 
  my $this = shift;
  my $tournament = shift;
  my $dp = shift;

  my $round = $dp->MostScores();
  my $logp = new TSH::Log($tournament, $dp, 'teams', $round);

  $logp->Write("Rank  Won- Lost Spread Team\n\n", <<'EOF');
<tr>
<th class=rank>Rank</th>
<th class=wl>Won-Lost</th>
<th class=spread>Spread</th>
<th class=name>Team</th>
</tr>
EOF

  my %teams;
  for my $p ($dp->Players()) {
    my $team = $p->Team();
    next unless length($team);
    $teams{$team}{'wins'} += $p->Wins();
    $teams{$team}{'losses'} += $p->Losses();
    $teams{$team}{'spread'} += $p->Spread();
    }

  my $lastw = -1; my $lasts = 0; my $rank = 0; my $i = 0; my $lastl = -1;
  for my $teamname (sort { 
    $teams{$b}->{'wins'} <=> $teams{$a}->{'wins'} 
    || $teams{$a}->{'losses'} <=> $teams{$b}->{'losses'} 
    || $teams{$b}->{'spread'} <=> $teams{$a}->{'spread'}
    } keys %teams) {
    my $teamdata = $teams{$teamname};
    my $wins = $teamdata->{'wins'};
    my $losses = $teamdata->{'losses'};
    my $spread = $teamdata->{'spread'};
    $i++;
    if ($wins != $lastw || $spread != $lasts || $losses != $lastl) {
      $lastl = $losses;
      $lastw = $wins;
      $lasts = $spread;
      $rank = $i;
      }
    my (@fields) = ($rank, $wins, $losses, $spread, $teamname);
    $logp->Write(
      sprintf("%3d %5.1f-%5.1f %+5d  %s\n", @fields),
      sprintf(<<'EOF', @fields));
<tr>
<td class=rank>%d</td>
<td class=wl>%.1f-%.1f</td>
<td class=spread>%+d</td>
<td class=name>%s</td>
</tr>
EOF
    }
  $logp->Close();
  }

=back

=cut

=head1 BUGS

None known

=cut

1;
